//                                               -*- C++ -*-
/**
 * @file		QStudyBrowser.cxx
 * @brief		Browsing panel for current opened study
 *
 * @author	Romuald Conty
 * @date		2006-07-17 11:05:59
 *
 * @par Last change :
 *  $LastChangedBy: dutka $
 *  $LastChangedDate: 2007-11-08 17:13:40 +0100 (jeu 08 nov 2007) $
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */
#include "QStudyBrowser.hxx"

// OT::UI::GUI
#include "QOTAction.hxx"

//Qt
#include <qcursor.h>
#include <qpopupmenu.h>
#include <qheader.h>
#include <qinputdialog.h>
#include <qmessagebox.h>

namespace OpenTURNS
{
	namespace UI
	{
		namespace GUI
		{
			QStudyBrowser::QStudyBrowser ( QOTObject *link, QWidget *parent, const char *name )
					: QListView ( parent, name )
			{
				addColumn ( "" );
				setResizeMode ( QListView::AllColumns );
				setSelectionMode ( QListView::Single );
				setColumnWidthMode ( 0, QListView::Manual );
				this->header() ->hide();

				rootItem_ = new QOTListViewItem ( this, link );
				setRootIsDecorated ( false );

				addChildren ( rootItem_, link, true );

				setOpen ( rootItem_, true );

				/* "Renaming" actions, slots and signals */
				actionRename_ = new QAction ( tr ( "&Rename..." ), Key_F2, this, "renameSelectedItem" );
				connect ( actionRename_, SIGNAL ( activated() ) , this, SLOT ( renameSelectedItem() ) );
				connect ( this, SIGNAL ( itemRenamed ( QListViewItem*, int , const QString & ) ) , this, SLOT ( itemRenamed ( QListViewItem *, int, const QString & ) ) );

				/* Context menu (right click) */
				connect ( this, SIGNAL ( contextMenuRequested ( QListViewItem *, const QPoint &, int ) ) , this, SLOT ( contextMenuRequested ( QListViewItem *, const QPoint &, int ) ) );
				/* Click */
				connect ( this, SIGNAL ( clicked ( QListViewItem *, const QPoint &, int ) ) , this, SLOT ( clicked ( QListViewItem * ) ) );

				show();
			}

			QStudyBrowser::~QStudyBrowser()
			{}

			void QStudyBrowser::renameSelectedItem()
			{
				QListViewItem * sender = selectedItem();
				Q_CHECK_PTR ( sender );
				if ( sender != NULL )
				{
					sender->startRename ( 0 );
				}
			}

			void QStudyBrowser::itemRenamed ( QListViewItem * item, int col, const QString & text )
			{
				QOTListViewItem * otItem = ( QOTListViewItem* ) item;
				otItem->getQOTObject() ->setName ( text );
				otItem->setText ( 0, otItem->getQOTObject() ->getName() );
			}

			void QStudyBrowser::contextMenuRequested ( QListViewItem * item, const QPoint & pos, int col )
			{
				QOTListViewItem * otItem = ( QOTListViewItem* ) item;
				QPopupMenu* popup = new QPopupMenu ( this );

				Q_CHECK_PTR ( otItem );
				if ( otItem != NULL )
				{
					if ( otItem->getQOTObject() ->getRenameEnabled() )
						actionRename_->addTo ( popup );
					QOTActions actions ( otItem->getQOTObject() ->getActions() );
					for ( uint i = 0; i < actions.count(); i++ )
					{
						actions.at ( i )->addTo ( popup );
					}
				}
				if ( popup->count() )
					popup->exec ( QCursor::pos() );
				delete popup;
			}

			void QStudyBrowser::clicked ( QListViewItem * sender )
			//, const QPoint & pos, int col )
			{
				Q_CHECK_PTR ( sender );
				if ( sender != NULL )
				{
					QOTListViewItem * otItem = ( QOTListViewItem* ) sender;
					if ( otItem->getQOTObject() ->inherits ( "OpenTURNS::UI::GUI::Chapter" ) )
					{
						emit clicked ( otItem->getQOTObject() );
					}
					else if ( otItem->getQOTObject() ->inherits ( "OpenTURNS::UI::GUI::Block" ) )
					{
						emit clicked ( otItem->getQOTObject() );
					}
				}
			}

			void QStudyBrowser::addChildren ( QOTListViewItem * parent, QOTObject * otObject, bool recursiveAdd )
			{
				QObjectList * childrenList = new QObjectList();
				otObject->getChildren ( childrenList, "OpenTURNS::UI::GUI::QOTObject", false, true );
				for ( uint i = 0; i < childrenList->count(); i++ )
				{
					QOTObject * otChild = ( QOTObject* ) childrenList->at ( i );
					if ( !otChild->inherits ( "OpenTURNS::UI::GUI::Content" ) )
					{
						QOTListViewItem * otItem = new QOTListViewItem ( parent , otChild );
						connect ( otChild, SIGNAL ( updated() ), otItem, SLOT ( update() ) );
						connect ( otItem, SIGNAL ( childrenChanged ( QOTListViewItem* ) ), this, SLOT ( childrenUpdate ( QOTListViewItem* ) ) );
						connect ( otChild, SIGNAL ( destroyed ( QObject* ) ), this, SLOT ( remove
						          ( QObject* ) ) );
						//connect( otChild, SIGNAL( updated( QOTObject* ) ), this, SLOT( update( QOTObject* ) ) );

						if ( recursiveAdd )
						{
							addChildren ( otItem, otChild, recursiveAdd );
						}
						setOpen ( otItem, true );
					}
				}
				setOpen ( parent, true );
			}

			QOTListViewItem* QStudyBrowser::find ( QObject * child )
			{
				QListViewItemIterator it ( rootItem_ );
				while ( it.current() )
				{
					QOTListViewItem * childItem = ( QOTListViewItem* ) it.current();
					if ( childItem->getQOTObject() == child )
					{
						return childItem;
					}
					++it;
				}
				return NULL;
			}

			void QStudyBrowser::remove
			( QObject * child )
			{
				QOTListViewItem * childItem = find ( child );
				if ( childItem != NULL )
				{
					//removeChildren( childItem );
					delete childItem;
				}
			}

			void QStudyBrowser::removeChildren ( QOTListViewItem * parent )
			{
				QListViewItem * item;
				const uint childCount = parent->childCount();
				for ( uint i = 0; i < childCount; i++ )
				{
					item = parent->firstChild();
					if ( item != NULL )
						delete item;
				}
			}

			void QStudyBrowser::childrenUpdate ( QOTListViewItem* parent )
			{
				Q_CHECK_PTR ( parent );
				removeChildren ( parent );
				addChildren ( parent, parent->getQOTObject(), true );
			}
		} /* namespace GUI */
	} /* namespace UI */
} /* namespace OpenTURNS */

