//                                               -*- C++ -*-
/**
 * @file		QOpenTURNS.cxx
 * @brief		Main window
 *
 * @author	Romuald Conty
 * @date		2006-06-06
 *
 * @par Last change :
 *  $LastChangedBy: dutka $
 *  $LastChangedDate: 2007-11-08 17:13:40 +0100 (jeu 08 nov 2007) $
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */

// Header
#include "QOpenTURNS.hxx"

// Qt
#include <qimage.h>
#include <qpixmap.h>
#include <qtoolbar.h>
#include <qtoolbutton.h>
#include <qpopupmenu.h>
#include <qmenubar.h>
#include <qtextedit.h>
#include <qfile.h>
#include <qfiledialog.h>
#include <qstatusbar.h>
#include <qmessagebox.h>
#include <qapplication.h>
#include <qaccel.h>
#include <qtextstream.h>
#include <qpainter.h>
#include <qpaintdevicemetrics.h>
#include <qwhatsthis.h>
#include <qinputdialog.h>

namespace OpenTURNS
{
	namespace UI
	{
		namespace GUI
		{
			/** Default constructor */
			QOpenTURNS::QOpenTURNS() : QMainWindow ( 0, "QOpenTURNS", WDestructiveClose )
			{
				study_ = NULL;
				studyMenu_ = NULL;
				visualizationFrame_ = NULL;

				dockWindowStudy_ = NULL;

				setIcon ( QPixmap::fromMimeSource ( "mainicon.png" ) );

				workspace_ = new QWorkspace ( this );
				setCentralWidget ( workspace_ );

				statusBar() ->message ( tr ( "Loading configuration..." ), 2000 );

				QRect mainWindowGeometry;
				settings_.setPath ( "OpenTURNS", "QOpenTURNS", QSettings::User );
				// main window geometry
				settings_.beginGroup ( "/QOpenTURNS/Geometry" );
				mainWindowGeometry.setX ( settings_.readNumEntry ( "x", 0 ) );
				mainWindowGeometry.setY ( settings_.readNumEntry ( "y", 0 ) );
				mainWindowGeometry.setWidth ( settings_.readNumEntry ( "width", 1024 ) );
				mainWindowGeometry.setHeight ( settings_.readNumEntry ( "height", 768 ) );
				settings_.endGroup();
				setGeometry ( mainWindowGeometry );

				/** Create "console" area (on bottom) */
				dockWindowConsole_ = new QDockWindow ( workspace_, "dockWindowConsole" );
				dockWindowConsole_->setCloseMode ( QDockWindow::Undocked );
				dockWindowConsole_->setResizeEnabled ( true );
				dockWindowConsole_->setFixedExtentHeight ( 100 );
				dockWindowConsole_->setCaption ( tr ( "Console" ) );
				moveDockWindow ( dockWindowConsole_, DockBottom );

				console_ = new QConsole ( dockWindowConsole_, "console" );
				dockWindowConsole_->setWidget ( console_ );
				console_->setReadOnly ( true );
				//console_->setIostreamRedirect( true );
				console_->show();
				dockWindowConsole_->hide();

				createMenus();

				statusBar() ->message ( tr ( "Ready" ), 2000 );
			}

			/** Destructor */
			QOpenTURNS::~QOpenTURNS()
			{
				// main window geometry
				settings_.beginGroup ( "/QOpenTURNS/Geometry" );
				settings_.writeEntry ( "x", geometry().x() );
				settings_.writeEntry ( "y", geometry().y() );
				settings_.writeEntry ( "width", geometry().width() );
				settings_.writeEntry ( "height", geometry().height() );
				settings_.endGroup();

				/**
				 * @todo check memory leaks with valgrind
				 */
				delete workspace_;
			}

			/** Create a new study */
			void QOpenTURNS::newStudy()
			{
				bool ok;
				QString text = QInputDialog::getText (
				                   tr ( "Choose study name" ) + " - OpenTURNS", tr ( "Enter study name:" ), QLineEdit::Normal,
				                   tr ( "Untitled" ), &ok, this );
				if ( ok && !text.isEmpty() )
				{
					// user entered something and pressed OK

					study_ = new Study ( text.ascii() );
					createStudyWorkspace();

					/** Update application context */
					updateWindow();
				}
				else
				{
					// user entered nothing or pressed Cancel
				}
			}

			/** Display open dialog then open the selected study's file */
			void QOpenTURNS::openStudy()
			{
				QString filename = QFileDialog::getOpenFileName ( QString::null, QString::null, this );
				if ( !filename.isEmpty() )
				{
					study_ = new Study ( "Study" );
					if ( study_->load ( filename ) )
					{

						createStudyWorkspace();

						/** Update application context */
						updateWindow();
						createStudyMenu();
					}
					else
					{
						delete study_;
						study_ = NULL;
						delete studyMenu_;
						studyMenu_ = NULL;

						statusBar() ->message ( tr ( "Loading aborted" ), 2000 );
					}
				}
				else
					statusBar() ->message ( tr ( "Loading aborted" ), 2000 );
			}

			/** Close the current study */
			void QOpenTURNS::closeStudy()
			{
				bool canBeClosed = false;
				if ( study_->isModified() )
				{
					QCloseEvent * ce = new QCloseEvent();
					closeEvent ( ce );
					if ( ce->isAccepted() )
					{
						canBeClosed = true;
					}
				}
				else
				{
					canBeClosed = true;
				}

				if ( canBeClosed )
				{
					/** @todo delete all QWidgets.. */
					if ( dockWindowStudy_ != NULL )
					{
						delete dockWindowStudy_;
						dockWindowStudy_ = NULL;
					}

					Q_CHECK_PTR ( study_ );
					delete study_;
					study_ = NULL;
					delete studyMenu_;
					studyMenu_ = NULL;
					// Drawing...
					delete dockWindowStudy_;
					dockWindowStudy_ = NULL;
					if ( visualizationFrame_ != NULL )
					{
						delete visualizationFrame_;
						visualizationFrame_ = NULL;
					}
					setCaption ( "QOpenTURNS" );
					updateMenus();
				}
			}

			/** Save the current study */
			void QOpenTURNS::saveStudy()
			{
				if ( study_->getFilename().isEmpty() )
				{
					saveStudyAs();
					return ;
				}

				if ( study_->save() )
				{
					statusBar() ->message ( tr ( "File %1 saved" ).arg ( study_->getFilename() ), 2000 );
				}
			}

			/** Display save dialog then save the current study as selected filename */
			void QOpenTURNS::saveStudyAs()
			{
				QString fn = QFileDialog::getSaveFileName ( QString::null, QString::null,
				             this );
				if ( !fn.isEmpty() )
				{
					study_->setFilename ( fn );
					saveStudy();
				}
				else
				{
					statusBar() ->message ( tr ( "Saving aborted" ), 2000 );
				}
			}

			/** Display confirmation dialog if study can be saved */
			void QOpenTURNS::closeEvent ( QCloseEvent* ce )
			{
				if ( study_ == NULL || ( study_ != NULL && !study_->isModified() ) )
				{
					ce->accept();
					return ;
				}

				switch ( QMessageBox::information ( this, tr ( "Save modified study ?" ) + " - OpenTURNS",
				                                    tr ( "Do you want to save the changes"
				                                         " to the study?" ),
				                                    tr ( "Yes" ), tr ( "No" ), tr ( "Cancel" ),
				                                    0, 1 ) )
				{
					case 0:
						saveStudy();
						ce->accept();
						break;
					case 1:
						ce->accept();
						break;
					case 2:
					default:           // just for sanity
						ce->ignore();
						break;
				}
			}

			/** Display the QOpenTURNS's about box */
			void QOpenTURNS::about()
			{
				QMessageBox::about ( this, tr ( "About" ) + " - OpenTURNS",         //  GUI aka QOpenTURNS
				                     "EDF - EADS - Phimeca" );
			}

			/** Display the QT's about box */
			void QOpenTURNS::aboutQt()
			{
				QMessageBox::aboutQt ( this, tr ( "About Qt" ) + " - OpenTURNS" );
			}

			/** @todo doxygen me...*/
			void QOpenTURNS::updateWindow( )
			{
				updateCaption();
				updateMenus();
			}

			/** @todo doxygen me...*/
			void QOpenTURNS::updateCaption( )
			{
				QString title = study_->getName();
				if ( study_->isModified() )
					title += tr ( " [modified]" );
				setCaption ( title + " - OpenTURNS" );
			}

			/** Create menus (on top) */
			void QOpenTURNS::createMenus()
			{
				QPopupMenu * file = new QPopupMenu ( this );
				menuBar() ->insertItem ( tr ( "&File" ), file );
				actionNewStudy_ = new QAction ( QPixmap::fromMimeSource ( "filenew.png" ), tr ( "&New" ), CTRL + Key_N, this, "newStudy" );
				connect ( actionNewStudy_, SIGNAL ( activated() ) , this, SLOT ( newStudy() ) );
				actionNewStudy_->addTo ( file );
				actionOpenStudy_ = new QAction ( QPixmap::fromMimeSource ( "fileopen.png" ), tr ( "&Open..." ), CTRL + Key_O, this, "openStudy" );
				connect ( actionOpenStudy_, SIGNAL ( activated() ) , this, SLOT ( openStudy() ) );
				actionOpenStudy_->addTo ( file );
				actionCloseStudy_ = new QAction ( QPixmap::fromMimeSource ( "fileclose.png" ), tr ( "&Close" ), CTRL + Key_W, this, "closeStudy" );
				connect ( actionCloseStudy_, SIGNAL ( activated() ) , this, SLOT ( closeStudy() ) );
				actionCloseStudy_->addTo ( file );

				file->insertSeparator();
				actionSaveStudy_ = new QAction ( QPixmap::fromMimeSource ( "filesave.png" ), tr ( "&Save" ), CTRL + Key_S, this, "saveStudy" );
				connect ( actionSaveStudy_, SIGNAL ( activated() ) , this, SLOT ( saveStudy() ) );
				actionSaveStudy_->addTo ( file );
				actionSaveStudyAs_ = new QAction ( QPixmap::fromMimeSource ( "filesaveas.png" ), tr ( "Save &As..." ), 0, this, "saveStudyAs" );
				connect ( actionSaveStudyAs_, SIGNAL ( activated() ) , this, SLOT ( saveStudyAs() ) );
				actionSaveStudyAs_->addTo ( file );
				file->insertSeparator();
				file->insertItem ( QPixmap::fromMimeSource ( "exit.png" ), tr ( "&Quit" ), qApp, SLOT ( closeAllWindows() ), CTRL + Key_Q );
				menuBar() ->insertSeparator();
				/*
				  QPopupMenu * options = new QPopupMenu( this );
				  menuBar() ->insertItem( tr( "&Options" ), options );
							 
				  QAction* actionConsoleOutput = new QAction( tr( "&Use internal console output" ), 0, this, "consoleOuput" );
				  actionConsoleOutput->setToggleAction ( true );
				  connect( actionConsoleOutput, SIGNAL( toggled( bool ) ) , this, SLOT( setConsoleVisible( bool ) ) );
				  connect( actionConsoleOutput, SIGNAL( toggled( bool ) ) , console_, SLOT( setIostreamRedirect( bool ) ) );
				  actionConsoleOutput->addTo( options );
				*/
				QPopupMenu * help = new QPopupMenu ( this );
				menuBar() ->insertItem ( tr ( "&Help" ), help );
				help->insertItem ( QIconSet ( QPixmap::fromMimeSource ( "mainicon.png" ) ), tr ( "&About" ), this, SLOT ( about() ), Key_F1 );
				QAction* helpAboutQt = new QAction ( tr ( "About &Qt" ), 0, this );
				helpAboutQt->setToolTip ( tr ( "Show the Qt library's About box" ) );
				connect ( helpAboutQt, SIGNAL ( activated() ) , this, SLOT ( aboutQt() ) );
				helpAboutQt->addTo ( help );
				help->insertSeparator();
				help->insertItem ( tr ( "What's &This" ), this, SLOT ( whatsThis() ), SHIFT + Key_F1 );

				updateMenus();
			}

			/** Enable/disable each menu depending on context */
			void QOpenTURNS::updateMenus()
			{
				actionNewStudy_->setEnabled ( study_ == NULL );
				actionOpenStudy_->setEnabled ( study_ == NULL );
				actionCloseStudy_->setEnabled ( study_ != NULL );
				actionSaveStudy_->setEnabled ( study_ != NULL );
				actionSaveStudyAs_->setEnabled ( study_ != NULL );
				/*			actionSaveStudy_->setEnabled( study_ != NULL && study_->isModified() );
							actionSaveStudyAs_->setEnabled( study_ != NULL && study_->isModified() );*/
			}

			void QOpenTURNS::createStudyWorkspace()
			{
				/** Create study's area (on left side)*/
				dockWindowStudy_ = new QDockWindow ( workspace_, "dockWindowStudy" );
				dockWindowStudy_->setCloseMode ( QDockWindow::Never );
				dockWindowStudy_->setResizeEnabled ( true );
				dockWindowStudy_->setFixedExtentWidth ( 270 );
				dockWindowStudy_->setCaption ( tr ( "Study" ) );
				moveDockWindow ( dockWindowStudy_, DockLeft );

				createStudyMenu();
				connect ( study_, SIGNAL ( renamed() ) , this, SLOT ( updateCaption( ) ) );
				connect ( study_, SIGNAL ( modifiedStateChanged ( bool ) ) , this, SLOT ( updateWindow() ) );

				studyBrowser_ = new QStudyBrowser ( study_, dockWindowStudy_, "studyBrowser_" );
				connect ( studyBrowser_, SIGNAL ( clicked ( QOTObject* ) ), this, SLOT ( showInformationAbout ( QOTObject* ) ) );

				dockWindowStudy_->setWidget ( studyBrowser_ );
			}

			void QOpenTURNS::showInformationAbout ( QOTObject * otObject )
			{
				if ( visualizationFrame_ != NULL )
					delete visualizationFrame_;

				visualizationFrame_ = new QVisualizationFrame ( workspace_ );
				visualizationFrame_->visualize ( otObject );

				if ( !visualizationFrame_->isEmpty() )
				{
					visualizationFrame_->showMaximized();
				}
				else
				{
					delete visualizationFrame_;
					visualizationFrame_ = NULL;
				}
			}

			void QOpenTURNS::createStudyMenu()
			{
				if ( studyMenu_ != NULL )
				{
					delete studyMenu_;
				}
				if ( study_ != NULL )
				{
					studyMenu_ = new QPopupMenu ( this );
					menuBar() ->insertItem ( tr ( "&Study" ), studyMenu_ );
					studyMenu_->insertItem ( QIconSet ( QPixmap::fromMimeSource ( "StepA_newBlock.png" ) ), tr ( "&New block A..." ), study_->getStepA(), SLOT ( createBlock() ) );
					studyMenu_->insertItem ( QIconSet ( QPixmap::fromMimeSource ( "StepB_newBlock.png" ) ), tr ( "&New block B..." ), study_->getStepB(), SLOT ( createBlock() ) );
					studyMenu_->insertItem ( QIconSet ( QPixmap::fromMimeSource ( "StepC3_newBlock.png" ) ), tr ( "&New calculation block C3..." ), study_->getStepC3(), SLOT ( createBlock() ) );
				}

			}

			void QOpenTURNS::setConsoleVisible ( bool visible )
			{
				if ( visible )
				{
					dockWindowConsole_->show();
				}
				else
				{
					dockWindowConsole_->hide();
				}
			}

		} /* namespace GUI */
	} /* namespace UI */
} /* namespace OpenTURNS */


