//                                               -*- C++ -*-
/**
 * @file    QOTObject.cxx
 * @brief   QOTObject is the base class for all QOpenTURNS classes.
 *
 * @author  Romuald Conty
 * @date    2006-07-17 10:25:15
 *
 * @par Last change :
 *  $LastChangedBy: dutka $
 *  $LastChangedDate: 2007-11-08 17:13:40 +0100 (jeu 08 nov 2007) $
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */
// Header
#include "QOTObject.hxx"

// Qt
#include <qobject.h>
#include <qmetaobject.h>
#include <qvariant.h>
#include <qregexp.h>

// OT::UI::GUI
#include "Parameter.hxx"

#include "DescriptionProperty.hxx"

namespace OpenTURNS
{
	namespace UI
	{
		namespace GUI
		{
			/** Default constructor */
			QOTObject::QOTObject ( QOTObject *parent, const char *name ) : QObject ( parent, name )
			{
				uuid_ = QUuid::createUuid ();
				renameEnabled_ = false;
				connect ( this, SIGNAL ( updated() ), this, SLOT ( updatedExt() ) );

				nameProperty_ = new QOTNameProperty ( this );

				description_ = Description ( 1 );
				new DescriptionProperty ( &description_, 0, true, this, "description" );
			}

			/** Copy constructor */
			QOTObject::QOTObject ( const QOTObject & other, QOTObject* parent, const char* name ) : QObject ( parent, name )
			{
				assignToMe ( other );
			}

			/** Destructor */
			QOTObject::~QOTObject()
			{}

			/** Assignment */
			QOTObject & QOTObject::operator = ( const QOTObject & other )
			{
				if ( this != &other )
				{
					// Other is NOT me, so I can assign it to me
					assignToMe ( other );
				}
				return *this;
			}

			/** QOTObject name accessor */
			void QOTObject::setName ( const char * name )
			{
				setName ( QString ( name ) );
			}

			/** QOTObject name accessor */
			bool QOTObject::setName ( const QString& name )
			{
				if ( renameEnabled_ )
				{
					QString newName = name.stripWhiteSpace();
					newName.replace ( QChar ( ' ' ), "_" );
					newName.replace ( QChar ( '.' ), "_" );
					newName.replace ( QChar ( '/' ), "_" );
					newName.replace ( QChar ( '\\' ), "_" );
					newName.replace ( QChar ( '*' ), "_" );

					if ( !newName.isEmpty() && ( newName != getName() ) )
					{
						QObject::setName ( newName.ascii() );
						nameProperty_->setValue ( newName );
						emit renamed();
						emit updated();
						return true;
					}
				}
				return false;
			}

			/** QOTObject name accessor */
			QString QOTObject::getName() const
			{
				return name();
			}

			/** QOTObject renameEnabled accessor */
			void QOTObject::setRenameEnabled ( bool renameEnabled )
			{
				renameEnabled_ = renameEnabled;
			}

			/** QOTObject renameEnabled accessor */
			bool QOTObject::getRenameEnabled()
			{
				return renameEnabled_;
			}

			/** QOTObject description accessor */
			void QOTObject::setDescription ( const QString& description )
			{
				if ( !description.stripWhiteSpace().isEmpty() && ( description.stripWhiteSpace() != getDescription() ) )
				{
					description_[ 0 ] = description.stripWhiteSpace().ascii();
					emit updated();
				}
			}

			/** QOTObject description accessor */
			QString QOTObject::getDescription() const
			{
				return QString ( description_[ 0 ] );
			}

			/** QOTObject uuid accessor (read only)*/
			QString QOTObject::getUuid() const
			{
				return uuid_.toString();
			}

			/** QOTObject className accessor */
			QString QOTObject::getClassName() const
			{
				return className();
			}

			/** QOTObject short className accessor */
			QString QOTObject::getShortClassName() const
			{
				QString classname = className();
				return classname.remove ( "OpenTURNS::UI::GUI::" );
			}

			void QOTObject::getChildren ( const QObject* startPoint, QObjectList* resultList, const char * inheritsClass, bool recursiveSearch, bool inherits )
			{
				QObjectList * childrenList = ( QObjectList* ) startPoint->children();
				if ( childrenList != NULL )
				{
					for ( uint i = 0; i < childrenList->count(); i++ )
					{
						const QObject* object = childrenList->at ( i );
						if ( inherits && object->inherits ( inheritsClass ) )
						{
							resultList->append ( object );
						}
						else if ( !inherits && object->isA ( inheritsClass ) )
						{
							resultList->append ( object );
						}
						if ( recursiveSearch )
							getChildren ( object, resultList, inheritsClass, TRUE, inherits );
					}
				}
			}

			void QOTObject::getChildren ( QObjectList* resultList, const char * inheritsClass, bool recursiveSearch, bool inherits )
			{
				QOTObject::getChildren ( this, resultList, inheritsClass, recursiveSearch, inherits );
			}

			const QObjectList * QOTObject::getChildren () const
			{
				return QObject::children();
			}

			QOTActions QOTObject::getActions()
			{
				QOTActions actions;
				QObjectList * childrenList = ( QObjectList* ) children();
				if ( childrenList != NULL )
				{
					for ( uint i = 0; i < childrenList->count(); i++ )
					{
						const QObject* object = childrenList->at ( i );
						if ( object->isA ( "OpenTURNS::UI::GUI::QOTAction" ) )
						{
							actions.append ( ( QOTAction* ) object );
						}
					}
				}
				return actions;
			}

			/** The function that actually realizes assignment between objects */
			void QOTObject::assignToMe ( const QOTObject & other )
			{
				setName ( other.getName() );
				setDescription ( other.getDescription() );
				uuid_ = QUuid::createUuid();
			}

			void QOTObject::updatedExt()
			{
				emit updated ( this );
			}

			/** */
			QString QOTObject::getFullName() const
			{
				return tr ( getShortClassName() ) + " : " + getName();
			}

			QString QOTObject::giveUniqueChildName ( QString wantedName ) const
			{
				QObjectList * objectList = ( QObjectList* ) getChildren();
				if ( objectList != NULL )
				{
					for ( uint i = 0; i < objectList->count(); i++ )
					{
						if ( objectList->at ( i ) ->inherits ( "OpenTURNS::UI::GUI::QOTObject" ) )
						{
							QOTObject * otObject = ( QOTObject* ) objectList->at ( i );
							if ( otObject->getName() == wantedName )
							{
								// if(match regexp _d)
								QRegExp rx ( "(.+)_(\\d+)$" );
								int pos = rx.search ( wantedName );
								if ( pos > -1 )
								{
									QString str = rx.cap ( 1 ); // "MyName"
									QString id = rx.cap ( 2 ); // "12"
									wantedName = str + "_" + QString::number ( id.toInt() + 1 );
								}
								else
								{
									wantedName += "_1";
								}
								return giveUniqueChildName ( wantedName );
							}
						}
					}
				}
				return wantedName;
			}

			QDomElement QOTObject::toXmlNode ( const QOTObject* startPoint, QDomDocument &d )
			{
				QDomElement node = d.createElement ( startPoint->getShortClassName() );

				const QMetaObject* metaObject = startPoint->metaObject();
				uint propertiesCount = metaObject->numProperties ( true );

				for ( uint i = 0; i < propertiesCount; i++ )
				{
					const QString propertyName = metaObject->propertyNames ( true ).at ( i );
					const QVariant propertyValue = startPoint->property ( propertyName.ascii() );
					if ( propertyValue.isValid() )
					{
						if ( propertyValue.type() == QVariant::List )
						{
							QValueList<QVariant> parameters = propertyValue.toList();
							if ( parameters[ 0 ].type() == QVariant::Map )
							{
								// Subproperties
								QDomElement parameters_node = d.createElement ( "Property" );
								parameters_node.setAttribute ( "name", propertyName );
								for ( uint j = 0; j < parameters.count(); j++ )
								{
									Parameter parameter = parameters[ j ].asMap();
									QDomElement parameter_node = d.createElement ( "Subproperty" );
									Parameter::Iterator it;
									for ( it = parameter.begin(); it != parameter.end(); ++it )
									{
										const QString key = it.key();
										QVariant value = it.data();
										const QString valueType = value.typeName();
										//parameter_node.setAttribute( "property", propertyName );
										if ( value.type() == QVariant::String )
										{
											parameter_node.setAttribute ( key , value.toString() );
										}
										else if ( ( value.type() == QVariant::StringList ) || ( value.type() == QVariant::List ) )
										{
											parameter_node.setAttribute ( key, value.asList() [ 0 ].toString() );
										}
										else
										{
											qDebug ( QString ( "Unknown type : " + valueType + " for " + key + " in " + propertyName ).ascii() );
										}
									}
									parameters_node.appendChild ( parameter_node );
								}
								node.appendChild ( parameters_node );
								//node.setAttribute( propertyName, parameters_attributes );
							}
							else if ( parameters[ 0 ].type() == QVariant::String )
							{
								node.setAttribute ( propertyName, parameters[ 0 ].toString() );
							}
						}
						else
						{
							QString propertyValueStr = propertyValue.toString() ;
							const QMetaProperty* metaProperty = metaObject->property ( i, true );

							if ( metaProperty != NULL && propertyName != "classname" )
							{
								if ( metaProperty->writable() )
								{
									if ( metaProperty->isEnumType() )
									{
										QStrList c_strings = metaProperty->enumKeys();
										propertyValueStr = c_strings.at ( propertyValue.toInt() );
										node.setAttribute ( propertyName, propertyValueStr );
									}
									else
									{
										node.setAttribute ( propertyName, propertyValueStr );
									}
								}
							}
						}
					}
				}
				QString nodeStr = d.toString();

				QObjectList * childrenList = new QObjectList();
				( ( QOTObject* ) startPoint ) ->getChildren ( childrenList, "OpenTURNS::UI::GUI::QOTObject", false, true );
				for ( uint i = 0; i < childrenList->count(); i++ )
				{
					QOTObject * otChild = ( QOTObject* ) childrenList->at ( i );
					node.appendChild ( otChild->toXmlNode ( d ) );
				}
				return node;
			}

			QDomElement QOTObject::toXmlNode ( QDomDocument &d )
			{
				return toXmlNode ( this, d );
			}

			QObject* QOTObject::findParent ( const QString& className )
			{
				QObject * otParent = this->parent();
				while ( otParent != NULL && !otParent->isA ( className ) )
				{
					otParent = otParent->parent();
				}
				return otParent;
			}

			QValueList<QVariant> QOTObject::convertParameters ( const Parameters parameters )
			{
				QValueList<QVariant> tmp;
				for ( uint i = 0; i < parameters.count(); i++ )
					tmp.insert ( tmp.end(), QVariant ( parameters[ i ] ) );
				return tmp;
			}

			QOTProperties QOTObject::getProperties ( bool recursiveSearch )
			{
				QPtrList<QOTProperty> properties;
				QOTProperty::getProperties ( this, &properties, recursiveSearch );
				return properties;
			}

			void QOTObject::propertyUpdated ( QOTProperty* property )
			{
				emit updated ( property );
				emit updated();
			}
		} /* namespace GUI */
	} /* namespace UI */
} /* namespace OpenTURNS */

