//                                               -*- C++ -*-
/**
 * @file    DistributionProperty.cxx
 * @brief
 *
 * @author  Romuald Conty
 * @date    2007-01-12 11:40:42
 *
 * @par Last change :
 *  $LastChangedBy: dutka $
 *  $LastChangedDate: 2008-06-26 13:50:17 +0200 (jeu 26 jun 2008) $
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */
// Header
#include "DistributionProperty.hxx"

// STD
#include <cstdio>

// OT
#include "NumericalSample.hxx"
#include "NumericalPoint.hxx"
#include "Graph.hxx"
#include "Path.hxx"

#include "Distribution.hxx"
#include "Normal.hxx"
#include "LogNormal.hxx"
#include "Uniform.hxx"
#include "Beta.hxx"
#include "Exponential.hxx"
#include "Gamma.hxx"
#include "Gumbel.hxx"
#include "Logistic.hxx"
#include "Student.hxx"
#include "TruncatedNormal.hxx"
#include "Triangular.hxx"
#include "Weibull.hxx"

// Qt
#include <qwidget.h>
#include <qimage.h>
#include <qpixmap.h>
#include <qlabel.h>

// OT::UI::GUI
#include "NumericalScalarProperty.hxx"

namespace OpenTURNS
{

	namespace UI
	{

		namespace GUI
		{

			typedef OT::Uncertainty::Distribution::Normal          Normal;
			typedef OT::Uncertainty::Distribution::LogNormal       LogNormal;
			typedef OT::Uncertainty::Distribution::Uniform         Uniform;
			typedef OT::Uncertainty::Distribution::Beta            Beta;
			typedef OT::Uncertainty::Distribution::Exponential     Exponential;
			typedef OT::Uncertainty::Distribution::Gamma           Gamma;
			typedef OT::Uncertainty::Distribution::Gumbel          Gumbel;
			typedef OT::Uncertainty::Distribution::Logistic        Logistic;
			typedef OT::Uncertainty::Distribution::Student         Student;
			typedef OT::Uncertainty::Distribution::TruncatedNormal TruncatedNormal;
			typedef OT::Uncertainty::Distribution::Triangular      Triangular;
			typedef OT::Uncertainty::Distribution::Weibull         Weibull;

			typedef OT::Base::Type::NumericalPoint NumericalPoint;

			typedef OT::Base::Graph::Graph Graph;
			typedef OT::Base::Common::Path Path;
			typedef OT::Base::Stat::CorrelationMatrix CorrelationMatrix;

			DistributionProperty::DistributionProperty ( Distribution* distribution, QObject* parent, const char* name ) : QOTProperty ( QOTProperty::MultipleChoice, parent, name )
			{
				Q_CHECK_PTR ( distribution );
				distribution_ = distribution;

				QOTAction * actionDrawPDF = new QOTAction ( QPixmap::fromMimeSource ( "PDF.png" ), tr ( "Draw PDF" ), 0, this, "EntryVariableType_drawPDF" );
				connect ( actionDrawPDF, SIGNAL ( activated() ), this, SLOT ( drawPDF() ) );

				QOTAction* actionDrawCDF = new QOTAction ( QPixmap::fromMimeSource ( "CDF.png" ), tr ( "Draw CDF" ), 0, this, "EntryVariableType_drawCDF" );
				connect ( actionDrawCDF, SIGNAL ( activated() ), this, SLOT ( drawCDF() ) );
			}


			DistributionProperty::~DistributionProperty()
			{}


			QString DistributionProperty::getValue()
			{
				return distributionTitle_;
			}

			QStringList DistributionProperty::getChoices()
			{
				QStringList choices;
				choices << "normal" << "log normal" << "uniform" << "beta" << "exponential" << "gamma" << "gumbel" << "logistic" << "student" << "triangular" << "truncated normal" << "weibull";
				//<< "histogram"
				//<< "geometric" << "multinomial" << "poisson" ;
				return choices;
			}

			bool DistributionProperty::setValue ( QString value )
			{
				if ( distributionTitle_ != value )
				{
					distributionTitle_ = value;
					autoconnect ( this );
					QOTProperties subproperties = getSubproperties ( false );
					subproperties.setAutoDelete ( true );
					subproperties.clear();

					if ( distributionTitle_ == "normal" )
					{
						arguments_ = NumericalPoint ( 2 );
						arguments_[ 0 ] = 0.;
						arguments_[ 1 ] = 1.;
						autoconnect ( new NumericalScalarProperty ( & ( arguments_[ 0 ] ), true, this, "mean" ) );
						autoconnect ( new NumericalScalarProperty ( & ( arguments_[ 1 ] ), true, this, "sigma" ) );
					}
					else if ( distributionTitle_ == "log normal" )
					{
						arguments_ = NumericalPoint ( 3 );
						arguments_[ 0 ] = 0.;
						arguments_[ 1 ] = 1.;
						arguments_[ 2 ] = 0.;
						autoconnect ( new NumericalScalarProperty ( & ( arguments_[ 0 ] ), true, this, "mean log" ) );
						autoconnect ( new NumericalScalarProperty ( & ( arguments_[ 1 ] ), true, this, "sigma log" ) );
						autoconnect ( new NumericalScalarProperty ( & ( arguments_[ 2 ] ), true, this, "origin" ) );
					}
					else if ( distributionTitle_ == "uniform" )
					{
						arguments_ = NumericalPoint ( 2 );
						arguments_[ 0 ] = 0.;
						arguments_[ 1 ] = 1.;
						autoconnect ( new NumericalScalarProperty ( & ( arguments_[ 0 ] ), true, this, "lower bound" ) );
						autoconnect ( new NumericalScalarProperty ( & ( arguments_[ 1 ] ), true, this, "upper bound" ) );
					}
					else if ( distributionTitle_ == "beta" )
					{
						arguments_ = NumericalPoint ( 4 );
						arguments_[ 0 ] = 1.;
						arguments_[ 1 ] = 2.;
						arguments_[ 2 ] = 0.;
						arguments_[ 3 ] = 1.;
						autoconnect ( new NumericalScalarProperty ( & ( arguments_[ 0 ] ), true, this, "s shape parameter" ) );
						autoconnect ( new NumericalScalarProperty ( & ( arguments_[ 1 ] ), true, this, "t shape parameter" ) );
						autoconnect ( new NumericalScalarProperty ( & ( arguments_[ 2 ] ), true, this, "lower bound" ) );
						autoconnect ( new NumericalScalarProperty ( & ( arguments_[ 3 ] ), true, this, "upper bound" ) );
					}
					else if ( distributionTitle_ == "exponential" )
					{
						arguments_ = NumericalPoint ( 2 );
						arguments_[ 0 ] = 1.;
						arguments_[ 1 ] = 0.;
						autoconnect ( new NumericalScalarProperty ( & ( arguments_[ 0 ] ), true, this, "lamda" ) );
						autoconnect ( new NumericalScalarProperty ( & ( arguments_[ 1 ] ), true, this, "gamma" ) );
					}
					else if ( distributionTitle_ == "gamma" )
					{
						arguments_ = NumericalPoint ( 3 );
						arguments_[ 0 ] = 1.;
						arguments_[ 1 ] = 1.;
						arguments_[ 2 ] = 0.;
						autoconnect ( new NumericalScalarProperty ( & ( arguments_[ 0 ] ), true, this, "k shape parameter" ) );
						autoconnect ( new NumericalScalarProperty ( & ( arguments_[ 1 ] ), true, this, "lamda" ) );
						autoconnect ( new NumericalScalarProperty ( & ( arguments_[ 2 ] ), true, this, "gamma" ) );
					}
					else if ( distributionTitle_ == "gumbel" )
					{
						arguments_ = NumericalPoint ( 2 );
						arguments_[ 0 ] = 1.;
						arguments_[ 1 ] = 0.;
						autoconnect ( new NumericalScalarProperty ( & ( arguments_[ 0 ] ), true, this, "alpha" ) );
						autoconnect ( new NumericalScalarProperty ( & ( arguments_[ 1 ] ), true, this, "beta" ) );
					}
					else if ( distributionTitle_ == "logistic" )
					{
						arguments_ = NumericalPoint ( 2 );
						arguments_[ 0 ] = 0.;
						arguments_[ 1 ] = 1.;
						autoconnect ( new NumericalScalarProperty ( & ( arguments_[ 0 ] ), true, this, "alpha" ) );
						autoconnect ( new NumericalScalarProperty ( & ( arguments_[ 1 ] ), true, this, "beta" ) );
					}
					else if ( distributionTitle_ == "student" )
					{
						arguments_ = NumericalPoint ( 2 );
						arguments_[ 0 ] = 3.;
						arguments_[ 1 ] = 0.;
						autoconnect ( new NumericalScalarProperty ( & ( arguments_[ 0 ] ), true, this, "nu" ) );
						autoconnect ( new NumericalScalarProperty ( & ( arguments_[ 1 ] ), true, this, "mu" ) );
					}
					else if ( distributionTitle_ == "triangular" )
					{
						arguments_ = NumericalPoint ( 3 );
						arguments_[ 0 ] = 0.;
						arguments_[ 1 ] = 0.5;
						arguments_[ 2 ] = 1.;
						autoconnect ( new NumericalScalarProperty ( & ( arguments_[ 0 ] ), true, this, "a" ) );
						autoconnect ( new NumericalScalarProperty ( & ( arguments_[ 1 ] ), true, this, "m" ) );
						autoconnect ( new NumericalScalarProperty ( & ( arguments_[ 2 ] ), true, this, "b" ) );
					}
					else if ( distributionTitle_ == "truncated normal" )
					{
						arguments_ = NumericalPoint ( 4 );
						arguments_[ 0 ] = 0.;
						arguments_[ 1 ] = 1.;
						arguments_[ 2 ] = -1.;
						arguments_[ 3 ] = 1.;
						autoconnect ( new NumericalScalarProperty ( & ( arguments_[ 0 ] ), true, this, "mu" ) );
						autoconnect ( new NumericalScalarProperty ( & ( arguments_[ 1 ] ), true, this, "sigma" ) );
						autoconnect ( new NumericalScalarProperty ( & ( arguments_[ 2 ] ), true, this, "a" ) );
						autoconnect ( new NumericalScalarProperty ( & ( arguments_[ 3 ] ), true, this, "b" ) );
					}
					else if ( distributionTitle_ == "weibull" )
					{
						arguments_ = NumericalPoint ( 3 );
						arguments_[ 0 ] = 1.;
						arguments_[ 1 ] = 1.;
						arguments_[ 2 ] = 0.;
						autoconnect ( new NumericalScalarProperty ( & ( arguments_[ 0 ] ), true, this, "alpha" ) );
						autoconnect ( new NumericalScalarProperty ( & ( arguments_[ 1 ] ), true, this, "beta" ) );
						autoconnect ( new NumericalScalarProperty ( & ( arguments_[ 2 ] ), true, this, "gamma" ) );
					}


					generateDistribution();
					return true;
				}
				return false;
			}

			void DistributionProperty::autoconnect ( QOTProperty* property )
			{
				Q_CHECK_PTR ( property );
				connect ( property, SIGNAL ( updated() ), this, SLOT ( generateDistribution() ) );
				propertyUpdated ( property );
			}

			void DistributionProperty::generateDistribution()
			{
				try
				{
					if ( distributionTitle_ == "normal" )
					{
						( *distribution_ ) = Normal ( arguments_[ 0 ], arguments_[ 1 ] );
					}
					else if ( distributionTitle_ == "log normal" )
					{
						( *distribution_ ) = LogNormal ( arguments_[ 0 ], arguments_[ 1 ], arguments_[ 2 ] );
					}
					else if ( distributionTitle_ == "uniform" )
					{
						( *distribution_ ) = Uniform ( arguments_[ 0 ], arguments_[ 1 ] );
					}
					else if ( distributionTitle_ == "beta" )
					{
						( *distribution_ ) = Beta ( arguments_[ 0 ], arguments_[ 1 ], arguments_[ 2 ], arguments_[ 3 ] );
					}
					else if ( distributionTitle_ == "exponential" )
					{
						( *distribution_ ) = Exponential ( arguments_[ 0 ], arguments_[ 1 ] );
					}
					else if ( distributionTitle_ == "gamma" )
					{
						( *distribution_ ) = Gamma ( arguments_[ 0 ], arguments_[ 1 ], arguments_[ 2 ] );
					}
					else if ( distributionTitle_ == "gumbel" )
					{
						( *distribution_ ) = Gumbel ( arguments_[ 0 ], arguments_[ 1 ] );
					}
					else if ( distributionTitle_ == "logistic" )
					{
						( *distribution_ ) = Logistic ( arguments_[ 0 ], arguments_[ 1 ] );
					}
					else if ( distributionTitle_ == "student" )
					{
						( *distribution_ ) = Student ( arguments_[ 0 ], arguments_[ 1 ] );
					}
					else if ( distributionTitle_ == "triangular" )
					{
						( *distribution_ ) = Triangular ( arguments_[ 0 ], arguments_[ 1 ], arguments_[ 2 ] );
					}
					else if ( distributionTitle_ == "truncated normal" )
					{
						( *distribution_ ) = TruncatedNormal ( arguments_[ 0 ], arguments_[ 1 ], arguments_[ 2 ], arguments_[ 3 ] );
					}
					else if ( distributionTitle_ == "weibull" )
					{
						( *distribution_ ) = Weibull ( arguments_[ 0 ], arguments_[ 1 ], arguments_[ 2 ] );
					}
				}
				catch ( ... )
				{
					std::cerr << "Error creating distribution distributionTitle_, check parameters" << std::endl;
				}
			}

			void DistributionProperty::drawPDF()
			{
				const uint pointNumber = 601;

				qDebug ( distribution_->str().c_str() );
				NumericalScalar median = distribution_->computeQuantile ( 0.5 ) [ 0 ];
				NumericalScalar interQuartile = distribution_->computeQuantile ( 0.75 ) [ 0 ] - distribution_->computeQuantile ( 0.25 ) [ 0 ];

				Graph graphPDF ( distribution_->drawPDF ( median - 3. * interQuartile, median + 3. * interQuartile, pointNumber ) );
				graphPDF.draw ( "", Path::BuildTemporaryFileName ( "PDFXXXXXX" ), 640, 480 );

				QWidget* widget = new QWidget ( NULL, "Distribution PDF" );
				widget->setBackgroundPixmap ( QPixmap ( QImage ( graphPDF.getBitmap().c_str() ) ) );
				remove ( graphPDF.getBitmap().c_str() );
				remove ( graphPDF.getPostscript().c_str() );
				widget->setFixedSize ( 640, 480 );
				widget->show();
			}

			void DistributionProperty::drawCDF()
			{
				const uint pointNumber = 601;

				qDebug ( distribution_->str().c_str() );

				NumericalScalar median = distribution_->computeQuantile ( 0.5 ) [ 0 ];
				NumericalScalar interQuartile = distribution_->computeQuantile ( 0.75 ) [ 0 ] - distribution_->computeQuantile ( 0.25 ) [ 0 ];

				Graph graphCDF ( distribution_->drawCDF ( median - 3. * interQuartile, median + 3. * interQuartile, pointNumber ) );

				graphCDF.draw ( "", Path::BuildTemporaryFileName ( "CDFXXXXXX" ), 640, 480 );

				QWidget* widget = new QWidget ( NULL, "Distribution CDF" );
				widget->setBackgroundPixmap ( QPixmap ( QImage ( graphCDF.getBitmap().c_str() ) ) );
				remove ( graphCDF.getBitmap().c_str() );
				remove ( graphCDF.getPostscript().c_str() );
				widget->setFixedSize ( 640, 480 );
				widget->show();

				/*
				  #include "Cloud.hxx"
				  typedef OT::Base::Stat::NumericalSample NumericalSample;
				  typedef OT::Base::Graph::Cloud Cloud;

				  uint size = 100;
				  NumericalSample dataX = normal.getNumericalSample( size );
				  NumericalSample dataY = normal.getNumericalSample( size );
				  NumericalSample data( size, 2 );
				  for ( uint i = 0; i < size; i ++ ) {
				  data[ i ][ 0 ] = dataX[ i ][ 0 ];
				  data[ i ][ 1 ] = dataY[ i ][ 0 ];
				  }
				  Cloud cloud( data, "La chaine de caractere" );
				  bool axis = true;
				  Graph graph( "Titre du graph", "truc x", "truc y", axis, "topright" );
				  graph.addDrawable( cloud );
				  graph.draw( "/tmp", "cloud", 640, 480 );
				       
				  QWidget* widgetCloud = new QWidget();
				  widgetCloud->setBackgroundPixmap( QPixmap( QImage( "/tmp/cloud.png" ) ) );
				  widgetCloud->show();
				*/
			}

			void DistributionProperty::propertyUpdated ( QOTProperty* subproperty )
			{
				emit updated ( subproperty );
			}

		}

	}

}
