//                                               -*- C++ -*-
/**
 * @file		Calculation.cxx
 * @brief
 *
 * @author	Romuald Conty
 * @date		2006-09-07 12:25:37
 *
 * @par Last change :
 *  $LastChangedBy: dutka $
 *  $LastChangedDate: 2007-11-08 17:13:40 +0100 (jeu 08 nov 2007) $
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */
// Header
#include "Calculation.hxx"

// Qt
#include <qapplication.h>
#include <qmutex.h>
#include <qvariant.h>

// OT::UI::GUI
#include "ResponseSurfaceFunction.hxx"
#include "GradientCalculationProperty.hxx"
#include "HessianCalculationProperty.hxx"

// OT
#include <iostream>

#include "Matrix.hxx"

#include "Distribution.hxx"
#include "ComposedDistribution.hxx"
#include "IndependentCopula.hxx"

#include "NumericalMathFunction.hxx"
#include "LinearNumericalMathFunction.hxx"

namespace OpenTURNS
{
	namespace UI
	{
		namespace GUI
		{
			typedef OT::Bool Bool;
			typedef OT::UnsignedLong UnsignedLong;
			typedef OT::NumericalScalar NumericalScalar;
			typedef OT::Base::Type::Matrix Matrix;

			typedef OT::Uncertainty::Distribution::IndependentCopula IndependentCopula;
			typedef OT::Uncertainty::Distribution::ComposedDistribution ComposedDistribution;
			typedef OT::Uncertainty::Model::Distribution Distribution;

			typedef OT::Base::Type::Collection<Distribution> DistributionCollection;
			typedef OT::Base::Func::NumericalMathFunction NumericalMathFunction;
			typedef OT::Base::Func::LinearNumericalMathFunction LinearNumericalMathFunction;

			// We define a Pair structure, which should be available in OT::Base::Common as a generic class
			struct Pair
			{
				QString name_;
				UnsignedLong index_;
				Pair() : name_(), index_()
				{}
				Pair ( QString name, UnsignedLong index ) : name_ ( name ), index_ ( index )
				{}
			}
			;
			typedef OT::Base::Type::Collection<Pair> PairCollection;

			Calculation::Calculation ( QOTObject *parent, const char *name ) : Chapter ( parent, name, "" ), QThread()
			{
				errorCode_ = 0;

				actionRun_ = new QOTAction ( tr ( "Run this calculation" ), 0, this, "Calculation_run" );
				connect ( actionRun_, SIGNAL ( activated() ), this, SLOT ( process() ) );
			}


			Calculation::~Calculation()
			{}

			void Calculation::process()
			{
				emit preprocessing();
				QThread::start();
			}

			void Calculation::run()
			{
				actionRun_->setEnabled ( false );
				errorCodeMutex_.lock();
				try
				{
					errorCode_ = processing();
				}
				catch ( ... )
				{
					errorCode_ = -1;
				}
				errorCodeMutex_.unlock();
				actionRun_->setEnabled ( true );
				//				Q_CHECK_PTR( eventUpdate );
				eventUpdate = new QCustomEvent ( 65000 );
				QApplication::postEvent ( ( QObject* ) this, eventUpdate );
			}

			int Calculation::getErrorCode()
			{
				int errorCode;
				errorCodeMutex_.lock();
				errorCode = errorCode_;
				errorCodeMutex_.unlock();
				return errorCode;
			}

			/**
			 *
			 * @param finalVariable
			 * @param blockB
			 * @return
			 * @note
			 * Here, we do all the steps in one step, but it could have been done in several
			 *       steps, each of them being done as soon as possible:
			 *       + At the end of step A1.2, we could have built a NumericalMathFunction
			 *         corresponding to the given function.
			 *       + At the end of step A1.3, we could have built a first matrix M of size n by p,
			 *         where p is the number of different entry variables connected to a component of
			 *         the input vector of the function. The element Mij equals 1 if and only if the
			 *         ith component of the input vector is connected to the jth unique entry variable.
			 *       + At the end of step A2.2.2, we have parts of the event already available: the
			 *         ComparisonOperator and the threshold.
			 *       + At the end of step B, we partition the matrix in two parts:
			 *         - the part associated to the deterministic entry variables
			 *         - the part associated to the stochastic entry variables
			 *         For the part associated to the deterministic variables, we build a vector of
			 *         size n by multiplying all the columns of M associated with deterministic entry
			 *         variables by their deterministic value, then adding these columns to form the
			 *         required vector.
			 *         For the part associated to the stochastic entry variables, we just remove the
			 *         columns of M associated with the deterministic entry variables.
			 *
			 * Example: Assume that you defined 5 variables E0, E1, E2, E3, E4, and a function taking
			 *          an input vector of dimension n = 6.
			 *          You connect the entry variables to the input vector components according to
			 *          the following pattern:
			 *          p0 = E1
			 *          p1 = E3
			 *          p2 = E0
			 *          p3 = E1
			 *          p4 = E2
			 *          p5 = E3
			 *
			 *          The entry variable E4 is unused, and the unique entry variables are E1, E3, E0
			 *          E2 IN THIS ORDER. In a loop over p0, ..., p5, for each entry variable we check
			 *          if it has already been used or not. In the first case we add a 1 at the proper
			 *          place in the proper column of M (which has a dynamic number of columns). In
			 *          the second case, we just add a column containing only one nonzero component,
			 *          which is equal to 1. Here is the evolution of M during the loop:
			 *          p0: M=[1] p1: M=[1 0] p2: M=[1 0 0] p3: M=[1 0 0] p4: M=[1 0 0 0]
			 *                [0]       [0 1]       [0 1 0]       [0 1 0]       [0 1 0 0]
			 *                [0]       [0 0]       [0 0 1]       [0 0 1]       [0 0 1 0]
			 *                [0]       [0 0]       [0 0 0]       [1 0 0]       [1 0 0 0]
			 *                [0]       [0 0]       [0 0 0]       [0 0 0]       [0 0 0 1]
			 *                [0]       [0 0]       [0 0 0]       [0 0 0]       [0 0 0 0]
			 *
			 *          p5: M=[1 0 0 0]
			 *                [0 1 0 0]
			 *                [0 0 1 0]
			 *                [1 0 0 0]
			 *                [0 0 0 1]
			 *                [0 1 0 0]
			 *
			 *          and the unique entry vector is (E1, E3, E0, E2). These two information are
			 *          computed in step A, and are transmitted to step B.
			 *
			 *          In step B, we decide that E0 and E1 are deterministic with value e0 and e1,
			 *          the other entry variables being stochastic. The partition reads:
			 *          linear=[0 0] constant=e1*[1] + e0*[0] = [e1]
			 *                 [1 0]             [0]      [0]   [ 0]
			 *                 [0 0]             [0]      [1]   [e0]
			 *                 [0 0]             [1]      [0]   [e1]
			 *                 [0 1]             [0]      [0]   [ 0]
			 *                 [1 0]             [0]      [0]   [ 0]
			 *
			 *          and the unique random vector is (E3, E2). As we know its marginal
			 *          distributions, we can build its distribution using a ComposedDistribution
			 *          with the independent copula of dimension 2, then the associated RandomVector.
			 *          With this RandomVector and the NumericalMathFunction, we define a RandomVector
			 *          corresponding to the final variable. With this last RandomVector, the
			 *          ComparisonOperator and the threshold, we define an Event.
			 *          We are ready to go to step C, to build a MonteCarlo algorithm with this Event,
			 *          then to modify the value of its parameters and to perform the computation.
			 *
			 */

			RandomVector Calculation::createOutputRandomVector ( FinalVariable* finalVariable, BlockB* blockB )
			{
				// Step A.1.2:
				// It is time to load the function associated to the final variable. For now, we just
				// load our old friend "poutre".
				//
				NumericalMathFunction numericalMathFunction = finalVariable->getNumericalMathFunction();

				/* code "temporaire" */

				// Step A.1.3 & B
				// The function has an input vector of dimension n = functionParameters.count().
				// For each component of this input vector, we must identify:
				//   1) If it is connected to a random variable or a deterministic variable
				//   2) In the case of a random variable, if this random variable has already been used
				//      for a previous component or not:
				//      + if it has already been used, to which of these unique variables it corresponds
				//      + if it has not already been used, it is a new unique variable and it is appened
				//        to the list of unique variables.
				//

				// Loop over the function parameters, i.e. the components of the input vector
				UnsignedLong inputDimension ( numericalMathFunction.getInputNumericalPointDimension() );

				NumericalPoint constant ( inputDimension, 0. );

				// As the number of columns of the transfer matrix is not known in advance, we store it in
				// a NumericalSample, which is a collection of NumericalPoints
				// size = 0.
				NumericalSample dynamicTransferMatrix ( 0, inputDimension );

				// List of the unique variables. We use OpenTURNS types on purpose, to give an example of
				// their use. A much cleaver (or at lest cleaner) implementation is possible using QT
				// containers.
				PairCollection uniqueList;
				// Marginal distributions of the input random vector
				DistributionCollection marginalDistributions;

				Description inputVariables = finalVariable->getInputVariables();

				for ( UnsignedLong indexParameter = 0; indexParameter < inputDimension; indexParameter++ )
				{
					const QString Schoice ( inputVariables[ indexParameter ] );

					EntryVariableType* entryVariableType = blockB->findEntryVariableType_ptr ( Schoice );

					Q_CHECK_PTR ( entryVariableType );
					if ( !entryVariableType )
					{
						qFatal ( "Can not find entry variables in the current block B." );
					}
					QString Stype ( "" );
					// If the entry is deterministic, it contributes to the constant term of the
					// transfer function
					if ( entryVariableType->isDeterministic() )
					{
						Stype = "deterministic";
						// Get the deterministic value.
						constant[ indexParameter ] = entryVariableType->getEntryVariable_ptr() ->getValue();
					} // if entryVariableType->isDeterministic()
					// If the entry is stochastic, check if it has already been used or not.
					// In the first case, we update the corresponding column of dynamicTransferMatrix
					// In the second case, add a column to dynamicTransferMatrix and add the entry
					// to the unique list.
					else
					{
						Stype = "stochastic";
						// Check if the entry has already been used
						UnsignedLong indexEntry;
						for ( indexEntry = 0; indexEntry < uniqueList.getSize(); indexEntry++ )
						{
							if ( Schoice == uniqueList[ indexEntry ].name_ )
							{
								break;
							}
						} // for indexEntry
						// If the entry has already been used
						if ( indexEntry < uniqueList.getSize() )
						{
							// Update the corresponding column of dynamicTransferMatrix
							dynamicTransferMatrix[ indexEntry ][ indexParameter ] = 1.;
						} // if indexEntry
						else
							// It is a new stochastic entry
						{
							// Add a column to dynamicTransferMatrix
							NumericalPoint column ( inputDimension, 0. );
							column[ indexParameter ] = 1.;
							dynamicTransferMatrix.add ( column );
							// Add the entry to the unique list
							uniqueList.add ( Pair ( Schoice, indexParameter ) );
							// Add the marginal distribution to the collection
							marginalDistributions.add ( entryVariableType->getDistribution() );
						} // else indexEntry
					} // else entryVariableType->isDeterministic()
					//std::cout << "parameter = " << Skey << ", value = " << Schoice << ", type = " << Stype << std::endl;
				} // for indexParameter
				// We can now convert dynamicTransferMatrix into a Matrix
				UnsignedLong stochasticDimension ( dynamicTransferMatrix.getSize() );
				Matrix linear ( inputDimension, stochasticDimension );
				for ( UnsignedLong i = 0; i < linear.getNbRows(); i++ )
				{
					for ( UnsignedLong j = 0; j < linear.getNbColumns(); j++ )
					{
						linear ( i, j ) = dynamicTransferMatrix[ j ][ i ];
					}
				}
				// Then, we can build a linear NumericalMathFunction from constant and linear. We do
				// it the long way, as OpenTURNS doesn't have a LinearNumericalMathFunction class yet.
				NumericalPoint zero ( linear.getNbColumns(), 0 );
				LinearNumericalMathFunction vectorize ( zero, constant, linear );
				// We are now ready to build the resulting code finalCode = externalCode(connect())
				NumericalMathFunction finalCode ( numericalMathFunction, vectorize );

				ComposedDistribution inputDistribution ( marginalDistributions, IndependentCopula ( stochasticDimension ) );
				inputDistribution.setName ( "inputDistribution" );
				qDebug ( ( std::string ( "inputDistribution=" ) + inputDistribution.str() ).c_str() );

				// We build the stochastic input vector form this distribution
				RandomVector inputVector ( inputDistribution, "inputVector" );
				qDebug ( ( std::string ( "inputVector=" ) + inputVector.str() ).c_str() );
				qDebug ( ( QString ( "inputVector.getDimension()=" ) + QString::number ( inputVector.getDimension() ) ).ascii() );

				// We build the output vector from the input vector and the function
				RandomVector outputVector ( finalCode, inputVector, "outputVector" );
				qDebug ( ( std::string ( "outputVector=" ) + outputVector.str() ).c_str() );
				qDebug ( ( QString ( "outputVector.getDimension()=" ) + QString::number ( outputVector.getDimension() ) ).ascii() );

				return outputVector;
			}

			void Calculation::customEvent ( QCustomEvent * event )
			{
				Q_CHECK_PTR ( eventUpdate );
				Q_CHECK_PTR ( event );
				if ( event->type() == eventUpdate->type() )
				{
					emit postprocessing();
				}
			}

		} /* namespace GUI */
	} /* namespace UI */
} /* namespace OpenTURNS */

