#! /usr/bin/env python
# -*- coding: utf-8 -*-

#	Copyright 2011-2012, Marten de Vries
#	Copyright 2011, Milan Boers
#
#	This file is part of OpenTeacher.
#
#	OpenTeacher is free software: you can redistribute it and/or modify
#	it under the terms of the GNU General Public License as published by
#	the Free Software Foundation, either version 3 of the License, or
#	(at your option) any later version.
#
#	OpenTeacher is distributed in the hope that it will be useful,
#	but WITHOUT ANY WARRANTY; without even the implied warranty of
#	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#	GNU General Public License for more details.
#
#	You should have received a copy of the GNU General Public License
#	along with OpenTeacher.  If not, see <http://www.gnu.org/licenses/>.

import os
import sys

class Loader(object):
	def __init__(self, loadModule, guiModule, path, addToRecentlyOpened=None, *args, **kwargs):
		super(Loader, self).__init__(*args, **kwargs)

		self.loadModule = loadModule
		self.guiModule = guiModule
		self.path = path
		self.addToRecentlyOpened = addToRecentlyOpened

	def load(self):
		if isinstance(self.path, unicode):
			#recently opened case
			self.path = self.path.encode(sys.getfilesystemencoding())
		lesson = self.loadModule.load(self.path)
		lesson.update({
			"changed": False,
			"path": self.path,
		})

		if self.addToRecentlyOpened:
			# Add to recently opened
			self.addToRecentlyOpened(**{
				"label": lesson["list"].get("title", "") or os.path.basename(self.path),
				"args": {},
				"kwargs": {"path": unicode(self.path, sys.getfilesystemencoding())},
				"method": "load",
				"moduleArgsSelectors": ["active"],
				"moduleKwargsSelectors": {"type": "loader"},
			})

		self.guiModule.loadFromLesson(lesson)

class LoaderModule(object):
	def __init__(self, moduleManager, *args, **kwargs):
		super(LoaderModule, self).__init__(*args, **kwargs)
		self._mm = moduleManager

		self.type = "loader"

		self.uses = (
			self._mm.mods(type="lesson"),
			self._mm.mods(type="load"),
			self._mm.mods(type="recentlyOpened"),
		)

	@property
	def _supportedFileTypes(self):
		types = set()
		for lesson in self._mm.mods("active", type="lesson"):
			try:
				types.add(lesson.dataType)
			except AttributeError:
				pass
		return types

	@property
	def usableExtensions(self):
		"""Returns a alphabetically sorted list of tuples. The tuples
		   have this form: (file_format_name, ext). For example:
		   ("OpenTeaching words file", "otwd"). The list includes all
		   extensions that can be loaded with the modules currently in
		   use by OpenTeacher

		"""
		exts = set()

		#Collect exts the loader modules support, if there is a gui
		#module for the type(s) they can provide
		for module in self._modules.sort("active", type="load"):
			for ext, fileTypes in module.loads.iteritems():
				for fileType in fileTypes:
					if fileType in self._supportedFileTypes:
						exts.add((ext, module.name))
		return sorted(exts)

	@property
	def openSupport(self):
		return len(self.usableExtensions) != 0

	@property
	def _addToRecentlyOpened(self):
		try:
			recentlyOpenedModule = self._modules.default("active", type="recentlyOpened")
		except IndexError:
			return
		return recentlyOpenedModule.add

	def load(self, path):
		loaders = []

		#Checks if loader modules can open it, and which type they would
		#return if they would load it only adds it as a possibility if
		#there also is a gui module for that type

		for loadModule in self._modules.sort("active", type="load"):
			fileType = loadModule.getFileTypeOf(path)
			for guiModule in self._modules.sort("active", type="lesson"):
				if not hasattr(guiModule, "loadFromLesson"):
					continue
				if guiModule.dataType == fileType:
					loaders.append(Loader(loadModule, guiModule, path, self._addToRecentlyOpened))

		if len(loaders) == 0:
			raise NotImplementedError()
		loader = loaders[0]

		loader.load()

	def loadFromLesson(self, dataType, lesson):
		loaders = []
		for loader in self._modules.sort("active", type="lesson"):
			if loader.dataType == dataType and hasattr(loader, "loadFromLesson"):
				loaders.append(loader)
		if len(loaders) == 0:
			raise NotImplementedError()
		loader = loaders[0]
		loader.loadFromLesson(lesson)

	def enable(self):
		self._modules = set(self._mm.mods(type="modules")).pop()

		self.active = True

	def disable(self):
		self.active = False

		del self._modules

def init(moduleManager):
	return LoaderModule(moduleManager)

