/*      -*- OpenSAF  -*-
 *
 * (C) Copyright 2008 The OpenSAF Foundation
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE. This file and program are licensed
 * under the GNU Lesser General Public License Version 2.1, February 1999.
 * The complete license can be accessed from the following location:
 * http://opensource.org/licenses/lgpl-license.php
 * See the Copying file included with the OpenSAF distribution for full
 * licensing terms.
 *
 * Author(s): Emerson Network Power
 *
 */

/*****************************************************************************
..............................................................................

   Vinay Khanna

..............................................................................

  DESCRIPTION:
   This module deals with the creation, accessing and deletion of
   the SU database on the AVND.

..............................................................................

  FUNCTIONS INCLUDED in this module:

  
******************************************************************************
*/

#include "avnd.h"

/****************************************************************************
  Name          : avnd_sudb_init
 
  Description   : This routine initializes the SU database.
 
  Arguments     : cb  - ptr to the AvND control block
 
  Return Values : NCSCC_RC_SUCCESS/NCSCC_RC_FAILURE
 
  Notes         : None.
******************************************************************************/
uns32 avnd_sudb_init(AVND_CB *cb)
{
	NCS_PATRICIA_PARAMS params;
	uns32 rc = NCSCC_RC_SUCCESS;

	memset(&params, 0, sizeof(NCS_PATRICIA_PARAMS));

	params.key_size = sizeof(SaNameT);
	rc = ncs_patricia_tree_init(&cb->sudb, &params);
	if (NCSCC_RC_SUCCESS == rc)
		m_AVND_LOG_SU_DB(AVND_LOG_SU_DB_CREATE, AVND_LOG_SU_DB_SUCCESS, 0, 0, NCSFL_SEV_INFO);
	else
		m_AVND_LOG_SU_DB(AVND_LOG_SU_DB_CREATE, AVND_LOG_SU_DB_FAILURE, 0, 0, NCSFL_SEV_CRITICAL);

	return rc;
}

/****************************************************************************
  Name          : avnd_sudb_destroy
 
  Description   : This routine destroys the SU database. It deletes all the 
                  SU records in the database.
 
  Arguments     : cb  - ptr to the AvND control block
 
  Return Values : NCSCC_RC_SUCCESS/NCSCC_RC_FAILURE
 
  Notes         : None.
******************************************************************************/
uns32 avnd_sudb_destroy(AVND_CB *cb)
{
	AVND_SU *su = 0;
	uns32 rc = NCSCC_RC_SUCCESS;

	/* scan & delete each su */
	while (0 != (su = (AVND_SU *)ncs_patricia_tree_getnext(&cb->sudb, (uns8 *)0))) {
		/* delete the record 
		   m_AVND_SEND_CKPT_UPDT_ASYNC_RMV(cb, su, AVND_CKPT_SU_CONFIG);
		   AvND is going down, but don't send any async update even for 
		   external components, otherwise external components will be deleted
		   from ACT. */
		rc = avnd_sudb_rec_del(cb, &su->name);
		if (NCSCC_RC_SUCCESS != rc)
			goto err;
	}

	/* finally destroy patricia tree */
	rc = ncs_patricia_tree_destroy(&cb->sudb);
	if (NCSCC_RC_SUCCESS != rc)
		goto err;

	m_AVND_LOG_SU_DB(AVND_LOG_SU_DB_DESTROY, AVND_LOG_SU_DB_SUCCESS, 0, 0, NCSFL_SEV_INFO);
	return rc;

 err:
	m_AVND_LOG_SU_DB(AVND_LOG_SU_DB_DESTROY, AVND_LOG_SU_DB_FAILURE, 0, 0, NCSFL_SEV_CRITICAL);
	return rc;
}

/****************************************************************************
  Name          : avnd_sudb_rec_add
 
  Description   : This routine adds a SU record to the SU database. If a SU 
                  is already present, nothing is done.
 
  Arguments     : cb   - ptr to the AvND control block
                  info - ptr to the SU parameters
                  rc   - ptr to the operation result
 
  Return Values : ptr to the SU record, if success
                  0, otherwise
 
  Notes         : None.
******************************************************************************/
AVND_SU *avnd_sudb_rec_add(AVND_CB *cb, AVND_SU_PARAM *info, uns32 *rc)
{
	AVND_SU *su = 0;

	/* verify if this su is already present in the db */
	if (0 != m_AVND_SUDB_REC_GET(cb->sudb, info->name)) {
		*rc = AVND_ERR_DUP_SU;
		goto err;
	}

	/* a fresh su... */
	su = calloc(1, sizeof(AVND_SU));
	if (!su) {
		*rc = AVND_ERR_NO_MEMORY;
		goto err;
	}

	/*
	 * Update the config parameters.
	 */
	/* update the su-name (patricia key) */
	memcpy(&su->name, &info->name, sizeof(SaNameT));

	/* update error recovery escalation parameters */
	su->comp_restart_prob = info->comp_restart_prob;
	su->comp_restart_max = info->comp_restart_max;
	su->su_restart_prob = info->su_restart_prob;
	su->su_restart_max = info->su_restart_max;
	su->su_err_esc_level = AVND_ERR_ESC_LEVEL_0;

	/* update the NCS flag */
	su->is_ncs = info->is_ncs;
	su->su_is_external = info->su_is_external;

	/*
	 * Update the rest of the parameters with default values.
	 */
	m_AVND_SU_OPER_STATE_SET_AND_SEND_NTF(cb, su, SA_AMF_OPERATIONAL_ENABLED);
	su->pres = SA_AMF_PRESENCE_UNINSTANTIATED;
	su->avd_updt_flag = FALSE;

	/* 
	 * Initialize the comp-list.
	 */
	su->comp_list.order = NCS_DBLIST_ASSCEND_ORDER;
	su->comp_list.cmp_cookie = avsv_dblist_uns32_cmp;
	su->comp_list.free_cookie = 0;

	/* 
	 * Initialize the si-list.
	 */
	su->si_list.order = NCS_DBLIST_ASSCEND_ORDER;
	su->si_list.cmp_cookie = avsv_dblist_saname_cmp;
	su->si_list.free_cookie = 0;

	/* 
	 * Initialize the siq.
	 */
	su->siq.order = NCS_DBLIST_ANY_ORDER;
	su->siq.cmp_cookie = 0;
	su->siq.free_cookie = 0;

	/* create the association with hdl-mngr */
	if ((0 == (su->su_hdl = ncshm_create_hdl(cb->pool_id, NCS_SERVICE_ID_AVND, (NCSCONTEXT)su)))) {
		*rc = AVND_ERR_HDL;
		goto err;
	}

	/* 
	 * Add to the patricia tree.
	 */
	su->tree_node.bit = 0;
	su->tree_node.key_info = (uns8 *)&su->name;
	*rc = ncs_patricia_tree_add(&cb->sudb, &su->tree_node);
	if (NCSCC_RC_SUCCESS != *rc) {
		*rc = AVND_ERR_TREE;
		goto err;
	}

	m_AVND_LOG_SU_DB(AVND_LOG_SU_DB_REC_ADD, AVND_LOG_SU_DB_SUCCESS, &info->name, 0, NCSFL_SEV_INFO);
	return su;

 err:
	if (su) {
		if (su->su_hdl)
			ncshm_destroy_hdl(NCS_SERVICE_ID_AVND, su->su_hdl);
		free(su);
	}

	m_AVND_LOG_SU_DB(AVND_LOG_SU_DB_REC_ADD, AVND_LOG_SU_DB_FAILURE, &info->name, 0, NCSFL_SEV_CRITICAL);
	return 0;
}

/****************************************************************************
  Name          : avnd_sudb_rec_del
 
  Description   : This routine deletes a SU record from the SU database.
 
  Arguments     : cb       - ptr to the AvND control block
                  name - ptr to the su-name (in n/w order)
 
  Return Values : NCSCC_RC_SUCCESS/NCSCC_RC_FAILURE
 
  Notes         : All the components belonging to this SU should have been 
                  deleted before deleting the SU record. Also no SIs should be
                  attached to this record.
******************************************************************************/
uns32 avnd_sudb_rec_del(AVND_CB *cb, SaNameT *name)
{
	AVND_SU *su = 0;
	uns32 rc = NCSCC_RC_SUCCESS;

	/* get the su record */
	su = m_AVND_SUDB_REC_GET(cb->sudb, *name);
	if (!su) {
		rc = AVND_ERR_NO_SU;
		goto err;
	}

	/* su should not have any comp or si attached to it */
	assert((!su->comp_list.n_nodes) && (!su->si_list.n_nodes));

	/* remove from the patricia tree */
	rc = ncs_patricia_tree_del(&cb->sudb, &su->tree_node);
	if (NCSCC_RC_SUCCESS != rc) {
		rc = AVND_ERR_TREE;
		goto err;
	}

	/* remove the association with hdl mngr */
	ncshm_destroy_hdl(NCS_SERVICE_ID_AVND, su->su_hdl);

	m_AVND_LOG_SU_DB(AVND_LOG_SU_DB_REC_DEL, AVND_LOG_SU_DB_SUCCESS, name, 0, NCSFL_SEV_INFO);

	/* free the memory */
	free(su);

	return rc;

 err:
	m_AVND_LOG_SU_DB(AVND_LOG_SU_DB_REC_DEL, AVND_LOG_SU_DB_FAILURE, name, 0, NCSFL_SEV_CRITICAL);
	return rc;
}

uns32 avnd_su_oper_req(AVND_CB *cb, AVSV_PARAM_INFO *param)
{
	AVND_SU *su;
	uns32 rc = NCSCC_RC_FAILURE;

	avnd_log(NCSFL_SEV_NOTICE, "Op %u, '%s'", param->act, param->name.value);

	su = m_AVND_SUDB_REC_GET(cb->sudb, param->name);
	if (!su) {
		avnd_log(LOG_ERR, "failed to get %s", param->name.value);
		goto done;
	}

	switch (param->act) {
	case AVSV_OBJ_OPR_MOD:
		switch (param->attr_id) {
		default:
			assert(0);
		}
		break;
	case AVSV_OBJ_OPR_DEL:
		/* delete the record */
		m_AVND_SEND_CKPT_UPDT_ASYNC_RMV(cb, su, AVND_CKPT_SU_CONFIG);
		rc = avnd_sudb_rec_del(cb, &param->name);
		break;
	default:
		assert(0);
	}

	rc = NCSCC_RC_SUCCESS;
done:
	return rc;
}
