# Copyright (C) 2000-2001 The OpenRPG Project
#
#   openrpg-dev@lists.sourceforge.net
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
# --
#
# File: scriptkit.py
# Author: Ted Berg
# Maintainer:
# Version:
#   $Id: scriptkit.py,v 1.5 2005/06/12 03:47:41 digitalxero Exp $
#
# Description: Class that contains convenience methods for various operations.  Was created with the purpose
#	of exposing a simple API to users of an as yet uncreated scripting interface.
#


import time
from orpg.orpg_windows import *
from orpg.orpg_xml import *
import orpg.chat.chat_msg

class scriptkit:
    """
    """
    __openrpg = None
    def __init__( self, openrpg ):
        """Simple constructor.  It currently only assigns the openrpg reference to a local variable.
        <ul>
            <li>openrpg - a reference to the application openrpg object.
        </ul>
        """
        self.__openrpg = openrpg
        self.chat = self.__openrpg.get_component( 'chat' )
        self.map = self.__openrpg.get_component( 'map' )
        self.settings = self.__openrpg.get_component( 'settings' )
        self.session = self.__openrpg.get_component('session')

    def addMiniatureToMap( self, min_label, min_url, unique=0 ):
        """Adds a new miniature icon to the map.  Miniature <em>will</em> be labeled unless autolabel is
        turned off on the map <em>and</em> the min_label field is blank.  Miniature will be numbered unless
        the 'unique' argument evaluates to true ( i.e. nonzero or a non-empty string ).
        <ul>
            <li>min_label - text string to be used as a label for the miniature
            <li>min_url - the URL for the image to be displayed on the map
            <li>unique - the mini will be numbered if this evaluates to false.
        </ul>
        """

        if min_url[:7] != "http://" :
            min_url = "http://"+min_url

        if self.map.canvas.auto_label:
            if min_label == '':
                start = min_url.rfind('/') + 1
                min_label = min_url[ start : len( min_url ) - 4 ]

            try:
                unique = eval( unique )
            except:
                pass

            if not unique:
                min_label = '%s %d' % ( min_label, self.map.canvas.layers['miniatures'].next_serial() )
        self.map.canvas.add_miniature( min_url, label, unique )

    def become( self, name ):
        self.sendToChat( '/name '+name )

    def sendToChat( self, text, name=false ):
        """Broadcasts the specified text to the chatbuffer.
        <ul>
            <li>text - the text to send.
        </ul>
        """
        # chat = self.__openrpg.get_component( 'chat' )
        if len( text ) == 0:
            return
        if text[0] == '/':
            self.chat.chat_cmds.docmd( text )
        else:
            self.chat.ParsePost( self.chat.colorize( self.chat.mytextcolor, text ), send=1, myself=1, alias=name )

    def sendToChatAs( self, name, text ):
        """Broadcasts the specified text to the chatbuffer as the specified alias
        <ul>
            <li>name - The player's name will be temporarily changed to this value
            <li>text - The text to broadcast to the chatbuffer
        </ul>
        """
        chat_cmds = self.chat.chat_cmds
        msg = orpg.chat.chat_msg.chat_msg()
        playername = self.settings.get_setting( 'player' )
        self.become( name )
        self.sendToChat( text, name )
        self.become( playername )

    def emoteToChat( self, text, alias=false ):
        if text[0] != '/':
            text = '/me ' + text
        self.sendToChat( text, name=alias )

    def emoteToChatAs( self, name, text ):
        playername = self.settings.get_setting( 'player' )
        self.become( name )
        self.emoteToChat( text, name )
        self.become( playername )

    def whisperToChat( self, who, text, alias=false ):
        if text[0] != '/':
            text = '/w %s=%s' % ( who, text )
        self.sendToChat( text, name=alias )

    def whisperToChatAs( self, who, alias, text ):
        playername = self.settings.get_setting( 'player' )
        self.become( alias )
        self.whisperToChat( who, text, name )
        self.become( playername )

    def chatMessage( self, message ):
        self.chat.Post( self.chat.colorize( self.chat.syscolor, message ) )

    def get_input( self, message, title, default ):
        dlg = rpgutils.wxTextEntryDialog( self, message, title, default )
        if dlg.ShowModal() == rpgutils.wxID_OK:
            return dlg.GetValue()
        dlg.Destroy()
        return None

    def show_info( self, title, message ):
        dlg.rpgutils.wxMessageDialog( None, message, title, rpgutils.wxOK | rpgutils.wxICON_INFORMATION )
        dlg.ShowModal()
        dlg.Destroy()
