/*************************************************************************
 *
 *  $RCSfile: layermanager.hxx,v $
 *
 *  $Revision: 1.2 $
 *
 *  last change: $Author: rt $ $Date: 2004/11/26 19:17:50 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/

#ifndef _SLIDESHOW_LAYERMANAGER_HXX
#define _SLIDESHOW_LAYERMANAGER_HXX

#ifndef BOOST_SHARED_PTR_HPP_INCLUDED
#include <boost/shared_ptr.hpp>
#endif

#ifndef _BGFX_RANGE_B2DRECTANGLE_HXX
#include <basegfx/range/b2drectangle.hxx>
#endif
#ifndef _BGFX_RANGE_B2DMULTIRANGE_HXX
#include <basegfx/range/b2dmultirange.hxx>
#endif

#include <vector>
#include <hash_map>
#include <algorithm>
#include <functional>

#include <view.hxx>
#include <layer.hxx>
#include <attributableshape.hxx>


namespace presentation
{
    namespace internal
    {

		/* Definition of Layermanager class */

        /** This class manages all of a slide's layers (and shapes)

			Since layer content changes when animations start or end,
			the layer manager keeps track of this and also handles
			starting/stopping of Shape animations. Note that none of
			the methods actually perform a screen update, this is
			always delayed until the ActivitiesQueue explicitely
			performs it.

            @see Layer
            @see Shape
         */
        class LayerManager
        {
        private:
            // defined here already, to be available for forEachShape
            // member template below.

            /** A set which contains all shapes for which a notifyAnimationUpdate
                has been issued.
            */
            typedef ::std::set< ShapeSharedPtr, Shape::lessThanShape > 	ShapeSet;

            // A hash map which maps the XShape to the corresponding Shape object
            typedef ::std::hash_map< ::com::sun::star::uno::Reference< ::com::sun::star::drawing::XShape >,
                                     ShapeSharedPtr,
                                     ::std::size_t (*)(const ::com::sun::star::uno::Reference< ::com::sun::star::drawing::XShape >&) > XShapeHash;

        public:
            /** Create a new layer manager for the given page bounds
             */
            LayerManager( const ::basegfx::B2DRectangle& rPageBounds );

            /** Query the bounds of the page, in user coordinate space.
             */
            ::basegfx::B2DRectangle getPageBounds() const;

            /** Add a new view to this object

            	All shapes will be made visible on the new view
             */
            void addView( const ViewSharedPtr& rView );

            /** Revoke a view
                
            	All subordinate objects associated to this view are
            	destroyed, and all references to this view are
            	dropped.
             */
            bool removeView( const ViewSharedPtr& rView );

            /** Add the shape to this object

            	This method adds a shape to the page.
             */
            void addShape( const ShapeSharedPtr& rShape );

            /** Remove shape from this object

            	This method removes a shape from the shape.
             */
            bool removeShape( const ShapeSharedPtr& rShape );
            
            /** Lookup a Shape from an XShape model object

            	This method looks up the internal shape map for one
            	representing the given XShape.

                @param xShape
                The XShape object, for which the representing Shape
                should be looked up.
             */
            ShapeSharedPtr lookupShape( const ::com::sun::star::uno::Reference< 
                                           ::com::sun::star::drawing::XShape >& xShape ) const;

            /** Query a subset of the given original shape

				This method queries a new (but not necessarily unique)
				shape, which displays only the given subset of the
				original one.
             */
            AttributableShapeSharedPtr getSubsetShape( const AttributableShapeSharedPtr& 	rOrigShape,
                                                       const DocTreeNode&					rTreeNode );

            /** Revoke a previously queried subset shape.

            	With this method, a previously requested subset shape
            	is revoked again. If the last client revokes a given
            	subset, it will cease to be displayed, and the
            	original shape will again show the subset data.

                @param rOrigShape
                The shape the subset was created from

                @param rSubsetShape
                The subset created from rOrigShape
             */
            void revokeSubset( const AttributableShapeSharedPtr& rOrigShape,
                               const AttributableShapeSharedPtr& rSubsetShape );

            /** Notify the LayerManager that the given Shape starts an
                animation now.

				This method enters animation mode for the Shape on all
				registered views.
             */
            void enterAnimationMode( const AnimatableShapeSharedPtr& rShape );

            /** Notify the LayerManager that the given Shape is no
                longer animated.

				This methods ends animation mode for the given Shape
				on all registered views.
             */
            void leaveAnimationMode( const AnimatableShapeSharedPtr& rShape );

            /** Notify that a shape needs an update

            	This method notifies the layer manager that a shape
            	update is necessary. This is useful if, during
            	animation playback, changes occur to shapes which make
            	an update necessary on an update() call. Otherwise,
            	update() will not render anything, which is not
            	triggered by calling one of the other LayerManager
            	methods.

                @param rShape
                Shape which needs an update
             */
            void notifyShapeUpdate( const ShapeSharedPtr& rShape);

            /** Check whether any update operations  are pending.

            	@return true, if this LayerManager has any updates
            	pending, i.e. needs to repaint something for the next
            	frame.
             */
            bool isUpdatePending() const;
            
            /** Flush any pending update operations.

            	This method clears all internally stored pending
            	updates. Use this method if you've refreshed the
            	screen content by other means.
             */
            void clearPendingUpdates();
            
            /** Update the content

				This method updates the content on all layers on all
				registered views. It does not issues a
				View::updateScreen() call on registered views. Please
				note that this method only takes into account changes
				to shapes induced directly by calling methods of the
				LayerManager. If a shape needs an update, because of
				some external event unknown to the LayerManager (most
				notably running animations), you have to notify the
				LayerManager via notifyShapeUpdate().

                @see LayerManager::updateScreen()

	            @return whether the update finished successfully.
            */
            bool update();

            /** Render the content.

				This method renders all shapes on all Layers on all
				registered views unconditionally, regardless of
				whether their attributes have changed or not. It does
				not issues a View::updateScreen() call on registered
				views.

                @see LayerManager::updateScreen()

	            @return whether the rendering finished successfully.
            */
            bool render();

            /** Call updateScreen() on all registered views.

            	This method calls updateScreen() on all registered
            	views, causing the canvases to copy their backbuffer
            	to front.
             */
            void updateScreen() const;

            /** Apply given functor to each added shape.

            	This member template applies the given functor to each
            	of the shapes added to this object (the order of
            	functor application is undefined), and returns a copy
            	of the functor in the state as it was for the last
            	functor application.

                @tpl Functor
                Functor object, must provide an unary operator() that
                accepts a const ShapeSharedPtr as its sole parameter.

                @param rFunctor
                Functor to apply
             */
            template< typename Functor> void forEachShape( const Functor& rFunctor ) const
            {
                // TODO(Q2): Use for_each here. Unfortunately, the straight-forward 
                // ::std::for_each( maXShapeHash.begin(),
                //                  maXShapeHash.end(),
                //                  ::boost::bind( rFunctor,
                //                                 ::boost::bind( ::std::select2nd<XShapeHash::value_type>(),
                //                                                _1 ) ) );
                // does not currentl compile with gcc. D'oh.

                XShapeHash::const_iterator 			aCurr( maXShapeHash.begin() );
                const XShapeHash::const_iterator	aEnd ( maXShapeHash.end() );
                while( aCurr != aEnd )
                {
                    rFunctor( aCurr->second );
                    ++aCurr;
                }
            }

            /** Apply given functor to each added shape.

            	This member template applies the given functor to each
            	of the shapes added to this object (the order of
            	functor application is undefined), and returns a copy
            	of the functor in the state as it was for the last
            	functor application.

                @tpl Functor
                Functor object, must provide an unary operator() that
                accepts a ShapeSharedPtr as its sole parameter.

                @param rFunctor
                Functor to apply
             */
            template< typename Functor> void forEachShape( const Functor& rFunctor )
            {
                // TODO(Q2): Use for_each here. Unfortunately, the straight-forward 
                // ::std::for_each( maXShapeHash.begin(),
                //                  maXShapeHash.end(),
                //                  ::boost::bind<bool>( rFunctor,
                //                                       ::boost::bind( ::std::select2nd<XShapeHash::value_type>(),
                //                                                      _1 ) ) );
                // does not currentl compile with gcc. D'oh.

                XShapeHash::iterator 		aCurr( maXShapeHash.begin() );
                const XShapeHash::iterator	aEnd ( maXShapeHash.end() );
                while( aCurr != aEnd )
                {
                    rFunctor( aCurr->second );
                    ++aCurr;
                }
            }
            
        private:
            // default: disabled copy/assignment
            LayerManager(const LayerManager&);
            LayerManager& operator=( const LayerManager& );

            void updateShapeArea( const ShapeSharedPtr& rShape ) const;

            struct ViewEntry
            {
                typedef ::std::vector< ViewLayerSharedPtr >		ViewLayerVector;

                // for generic algo access (which needs actual functions)
                ViewSharedPtr getView() const { return mpView; }

                ViewSharedPtr	mpView;
                ViewLayerVector	maLayerVector;
            };

            typedef ::std::vector< ViewEntry > 							ViewVector;
            typedef ::std::vector< LayerSharedPtr > 					LayerVector;

            /// Registered views on this object
            ViewVector							maViews;

            /// Active layers 
            LayerVector							maLayers;

            /// Contains all shapes with their XShape reference as the key
            XShapeHash							maXShapeHash;

            /// Set containing all shapes that have requested an update via notifyShapeUpdate()
            mutable ShapeSet					maUpdateShapes;
            mutable ::basegfx::B2DMultiRange	maUpdateAreas;

            const ::basegfx::B2DRectangle		maPageBounds;
        };

        typedef ::boost::shared_ptr< LayerManager > LayerManagerSharedPtr;
    }
}

#endif /* _SLIDESHOW_LAYERMANAGER_HXX */
