/*************************************************************************
 *
 *  $RCSfile: dx_canvashelper.cxx,v $
 *
 *  $Revision: 1.2 $
 *
 *  last change: $Author: rt $ $Date: 2004/11/26 17:19:00 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/

#include <canvas/debug.hxx>

#ifndef _RTL_LOGFILE_HXX_
#include <rtl/logfile.hxx>
#endif

#ifndef _DRAFTS_COM_SUN_STAR_RENDERING_COMPOSITEOPERATION_HPP_
#include <drafts/com/sun/star/rendering/CompositeOperation.hpp>
#endif

#ifndef BOOST_SCOPED_ARRAY_HPP_INCLUDED
#include <boost/scoped_array.hpp>
#endif

#ifndef _BGFX_MATRIX_B2DHOMMATRIX_HXX
#include <basegfx/matrix/b2dhommatrix.hxx>
#endif
#ifndef _BGFX_POINT_B2DPOINT_HXX
#include <basegfx/point/b2dpoint.hxx>
#endif
#ifndef _BGFX_TOOLS_CANVASTOOLS_HXX
#include <basegfx/tools/canvastools.hxx>
#endif

#include <algorithm>

#include <canvas/canvastools.hxx>

#include <dx_impltools.hxx>
#include <dx_parametricpolypolygon.hxx>
#include <dx_gdisurface.hxx>
#include <dx_canvasfont.hxx>
#include <dx_textlayout.hxx>
#include <dx_canvashelper.hxx>


using namespace ::com::sun::star;
using namespace ::drafts::com::sun::star;

namespace dxcanvas
{
    CanvasHelper::CanvasHelper() :
        mpGdiPlusUser( GDIPlusUser::createInstance() ),
        mxDevice(),
        mpSurface(),
        mpBitmap(),
        maSize(),
        maOutputOffset()
    {
    }

    void CanvasHelper::disposing()
    {
        mpGdiPlusUser.reset();
        mxDevice.reset();
        mpSurface.reset();
        mpBitmap.reset();
    }

    void CanvasHelper::setGraphicDevice( const WindowGraphicDevice::ImplRef& rDevice )
    {
        mxDevice = rDevice;
    }

    void CanvasHelper::setBitmap( const BitmapSharedPtr& rBitmap )
    {
        setBitmap( rBitmap,
                   ::basegfx::B2ISize() );
    }

    void CanvasHelper::setBitmap( const BitmapSharedPtr& 		rBitmap,
                                  const ::basegfx::B2ISize& 	rOutputOffset )
    {
        ENSURE_AND_THROW( rBitmap.get(),
                          "CanvasHelper::setBitmap(): Invalid bitmap" );
        ENSURE_AND_THROW( !mpSurface.get(),
                          "CanvasHelper::setBitmap(): surface set, old surface would be overwritten" );

        mpBitmap = rBitmap;

        mpSurface.reset( new GdiSurface(
                             SurfaceGraphicsSharedPtr(
                                 new SurfaceGraphics(
                                     Gdiplus::Graphics::FromImage(
                                         mpBitmap.get() ) ) ) ) );

        tools::setupGraphics( *mpSurface->getGraphics()->get() );

        maSize.setX( rBitmap->GetWidth() );
        maSize.setY( rBitmap->GetHeight() );

        maOutputOffset = rOutputOffset;
    }

    void CanvasHelper::setSurface( const SurfaceSharedPtr& 		rSurface,
                                   const ::basegfx::B2ISize& 	rSize,
                                   const ::basegfx::B2ISize& 	rOutputOffset )
    {
        ENSURE_AND_THROW( !mpBitmap.get(),
                          "CanvasHelper::setSurface(): bitmap set, surface would be overwritten" );

        mpSurface = rSurface;
        maSize = rSize;
        maOutputOffset = rOutputOffset;
    }

    void SAL_CALL CanvasHelper::drawPoint( const geometry::RealPoint2D& 	aPoint, 
                                           const rendering::ViewState& 		viewState, 
                                           const rendering::RenderState&	renderState )
    {
        if( needOutput() )
        {
            SurfaceGraphicsSharedPtr aGraphics( mpSurface->getGraphics() );

            setupGraphicsState( aGraphics, viewState, renderState );
        
            Gdiplus::SolidBrush aBrush( 
                Gdiplus::Color( 
                    tools::sequenceToArgb( mxDevice.getRef(),
                                           renderState.DeviceColor ) ) );

            // paint a one-by-one circle, with the given point
            // in the middle (rounded to float)
            (*aGraphics)->FillEllipse( &aBrush, 
                                       static_cast<Gdiplus::REAL>(aPoint.X), 			// disambiguate call
                                       static_cast<Gdiplus::REAL>(aPoint.Y), 
                                       static_cast<Gdiplus::REAL>(aPoint.X + 1.0), 
                                       static_cast<Gdiplus::REAL>(aPoint.Y + 1.0) );
        }
    }

    void SAL_CALL CanvasHelper::drawLine( const geometry::RealPoint2D& 	aStartPoint, 
                                          const geometry::RealPoint2D& 	aEndPoint, 
                                          const rendering::ViewState& 	viewState, 
                                          const rendering::RenderState& renderState )
    {
        if( needOutput() )
        {
            SurfaceGraphicsSharedPtr aGraphics( mpSurface->getGraphics() );

            setupGraphicsState( aGraphics, viewState, renderState );
        
            Gdiplus::Pen aPen( 
                Gdiplus::Color( 
                    tools::sequenceToArgb( mxDevice.getRef(),
                                           renderState.DeviceColor ) ) );

            (*aGraphics)->DrawLine( &aPen, 
                                    static_cast<Gdiplus::REAL>(aStartPoint.X), // disambiguate call
                                    static_cast<Gdiplus::REAL>(aStartPoint.Y), 
                                    static_cast<Gdiplus::REAL>(aEndPoint.X), 
                                    static_cast<Gdiplus::REAL>(aEndPoint.Y) );
        }
    }

    void SAL_CALL CanvasHelper::drawBezier( const geometry::RealBezierSegment2D&	aBezierSegment, 
                                            const geometry::RealPoint2D& 			aEndPoint,
                                            const rendering::ViewState& 			viewState, 
                                            const rendering::RenderState& 			renderState )
    {
        if( needOutput() )
        {
            SurfaceGraphicsSharedPtr aGraphics( mpSurface->getGraphics() );

            setupGraphicsState( aGraphics, viewState, renderState );
        
            Gdiplus::Pen aPen( 
                Gdiplus::Color( 
                    tools::sequenceToArgb( mxDevice.getRef(),
                                           renderState.DeviceColor ) ) );

            (*aGraphics)->DrawBezier( &aPen,
                                      static_cast<Gdiplus::REAL>(aBezierSegment.Px), // disambiguate call
                                      static_cast<Gdiplus::REAL>(aBezierSegment.Py),
                                      static_cast<Gdiplus::REAL>(aBezierSegment.C1x),
                                      static_cast<Gdiplus::REAL>(aBezierSegment.C1y),
                                      static_cast<Gdiplus::REAL>(aEndPoint.X),
                                      static_cast<Gdiplus::REAL>(aEndPoint.Y),
                                      static_cast<Gdiplus::REAL>(aBezierSegment.C2x),
                                      static_cast<Gdiplus::REAL>(aBezierSegment.C2y) );
        }
    }

    uno::Reference< rendering::XCachedPrimitive > SAL_CALL CanvasHelper::drawPolyPolygon( const uno::Reference< rendering::XPolyPolygon2D >& 	xPolyPolygon, 
                                                                                          const rendering::ViewState& 							viewState, 
                                                                                          const rendering::RenderState& 						renderState )
    {
        ENSURE_AND_THROW( xPolyPolygon.is(), 
                          "CanvasHelper::drawPolyPolygon: polygon is NULL");

        if( needOutput() )
        {
            SurfaceGraphicsSharedPtr aGraphics( mpSurface->getGraphics() );

            setupGraphicsState( aGraphics, viewState, renderState );
        
            Gdiplus::Pen aPen( 
                Gdiplus::Color( 
                    tools::sequenceToArgb( mxDevice.getRef(),
                                           renderState.DeviceColor ) ) );
        
            GraphicsPathSharedPtr pPath( tools::graphicsPathFromXPolyPolygon2D( xPolyPolygon ) );

            // TODO(E1): Return value
            (*aGraphics)->DrawPath( &aPen, pPath.get() );
        }

        // TODO(P1): Provide caching here.
        return uno::Reference< rendering::XCachedPrimitive >(NULL);
    }

    uno::Reference< rendering::XCachedPrimitive > SAL_CALL CanvasHelper::strokePolyPolygon( const uno::Reference< rendering::XPolyPolygon2D >& 	xPolyPolygon, 
                                                                                            const rendering::ViewState& 						viewState, 
                                                                                            const rendering::RenderState& 						renderState, 
                                                                                            const rendering::StrokeAttributes& 					strokeAttributes )
    {
        // TODO
        return uno::Reference< rendering::XCachedPrimitive >(NULL);
    }

    uno::Reference< rendering::XCachedPrimitive > SAL_CALL CanvasHelper::strokeTexturedPolyPolygon( const uno::Reference< rendering::XPolyPolygon2D >& 	xPolyPolygon, 
                                                                                                    const rendering::ViewState& 						viewState, 
                                                                                                    const rendering::RenderState& 						renderState, 
                                                                                                    const uno::Sequence< rendering::Texture >& 			textures, 
                                                                                                    const rendering::StrokeAttributes& 					strokeAttributes )
    {
        // TODO
        return uno::Reference< rendering::XCachedPrimitive >(NULL);
    }

    uno::Reference< rendering::XCachedPrimitive > SAL_CALL CanvasHelper::strokeTextureMappedPolyPolygon( const uno::Reference< rendering::XPolyPolygon2D >& xPolyPolygon, 
                                                                                                         const rendering::ViewState& 						viewState, 
                                                                                                         const rendering::RenderState& 						renderState, 
                                                                                                         const uno::Sequence< rendering::Texture >& 		textures, 
                                                                                                         const uno::Reference< geometry::XMapping2D >& 		xMapping, 
                                                                                                         const rendering::StrokeAttributes& 				strokeAttributes )
    {
        // TODO
        return uno::Reference< rendering::XCachedPrimitive >(NULL);
    }

    uno::Reference< rendering::XPolyPolygon2D >   SAL_CALL CanvasHelper::queryStrokeShapes( const uno::Reference< rendering::XPolyPolygon2D >& 	xPolyPolygon, 
                                                                                            const rendering::ViewState& 						viewState, 
                                                                                            const rendering::RenderState& 						renderState, 
                                                                                            const rendering::StrokeAttributes& 					strokeAttributes )
    {
        // TODO
        return uno::Reference< rendering::XPolyPolygon2D >(NULL);
    }

    uno::Reference< rendering::XCachedPrimitive > SAL_CALL CanvasHelper::fillPolyPolygon( const uno::Reference< rendering::XPolyPolygon2D >& 	xPolyPolygon, 
                                                                                          const rendering::ViewState& 							viewState, 
                                                                                          const rendering::RenderState& 						renderState )
    {
        ENSURE_AND_THROW( xPolyPolygon.is(), 
                          "CanvasHelper::fillPolyPolygon: polygon is NULL");

        if( needOutput() )
        {
            SurfaceGraphicsSharedPtr aGraphics( mpSurface->getGraphics() );

            setupGraphicsState( aGraphics, viewState, renderState );
        
            Gdiplus::SolidBrush aBrush( 
                Gdiplus::Color( 
                    tools::sequenceToArgb( mxDevice.getRef(),
                                           renderState.DeviceColor ) ) );
        
            GraphicsPathSharedPtr pPath( tools::graphicsPathFromXPolyPolygon2D( xPolyPolygon ) );

            // TODO(E1): Return value
            // TODO(F1): FillRule
            (*aGraphics)->FillPath( &aBrush, pPath.get() );
        }

        // TODO(P1): Provide caching here.
        return uno::Reference< rendering::XCachedPrimitive >(NULL);
    }

    uno::Reference< rendering::XCachedPrimitive > SAL_CALL CanvasHelper::fillTexturedPolyPolygon( const uno::Reference< rendering::XPolyPolygon2D >& 	xPolyPolygon, 
                                                                                                  const rendering::ViewState& 							viewState, 
                                                                                                  const rendering::RenderState& 						renderState, 
                                                                                                  const uno::Sequence< rendering::Texture >& 			textures )
    {
        ENSURE_AND_THROW( xPolyPolygon.is(), 
                          "CanvasHelper::fillTexturedPolyPolygon: polygon is NULL");
        ENSURE_AND_THROW( textures.getLength(), 
                          "CanvasHelper::fillTexturedPolyPolygon: empty texture sequence");

        if( needOutput() )
        {
            SurfaceGraphicsSharedPtr aGraphics( mpSurface->getGraphics() );

            setupGraphicsState( aGraphics, viewState, renderState );
        
            // TODO(F1): Multi-texturing
            if( textures[0].Gradient.is() )
            {
                uno::Reference< lang::XServiceInfo > xRef( textures[0].Gradient, 
                                                           uno::UNO_QUERY );

                if( xRef.is() && 
                    xRef->getImplementationName().equals( ::rtl::OUString(RTL_CONSTASCII_USTRINGPARAM(PARAMETRICPOLYPOLYGON_IMPLEMENTATION_NAME))) )
                {
                    // TODO(Q1): Maybe use dynamic_cast here

                    // TODO(E1): Return value
                    // TODO(F1): FillRule
                    static_cast<ParametricPolyPolygon*>(textures[0].Gradient.get())->fill(
                        aGraphics,
                        tools::graphicsPathFromXPolyPolygon2D( xPolyPolygon ),
                        viewState,
                        renderState,
                        textures[0] );
                }
            }
        }

        // TODO(P1): Provide caching here.
        return uno::Reference< rendering::XCachedPrimitive >(NULL);
    }

    uno::Reference< rendering::XCachedPrimitive > SAL_CALL CanvasHelper::fillTextureMappedPolyPolygon( const uno::Reference< rendering::XPolyPolygon2D >& 	xPolyPolygon, 
                                                                                                       const rendering::ViewState& 							viewState, 
                                                                                                       const rendering::RenderState& 						renderState, 
                                                                                                       const uno::Sequence< rendering::Texture >& 			textures, 
                                                                                                       const uno::Reference< geometry::XMapping2D >& 		xMapping )
    {
        // TODO
        return uno::Reference< rendering::XCachedPrimitive >(NULL);
    }

    uno::Reference< rendering::XCanvasFont > SAL_CALL CanvasHelper::createFont( const rendering::FontRequest& 					fontRequest, 
                                                                                const uno::Sequence< beans::PropertyValue >& 	extraFontProperties, 
                                                                                const geometry::Matrix2D& 						fontMatrix )
    {
        if( needOutput() )
        {
            return uno::Reference< rendering::XCanvasFont >( 
                    new CanvasFont(fontRequest, extraFontProperties, fontMatrix ) );
        }

        return uno::Reference< rendering::XCanvasFont >();
    }

    uno::Sequence< rendering::FontInfo > SAL_CALL CanvasHelper::queryAvailableFonts( const rendering::FontInfo& 					aFilter, 
                                                                                     const uno::Sequence< beans::PropertyValue >& 	aFontProperties )
    {
        // TODO
        return uno::Sequence< rendering::FontInfo >();
    }

    uno::Reference< rendering::XCachedPrimitive > SAL_CALL CanvasHelper::drawText( const rendering::StringContext& 					text, 
                                                                                   const uno::Reference< rendering::XCanvasFont >& 	xFont, 
                                                                                   const rendering::ViewState& 						viewState, 
                                                                                   const rendering::RenderState& 					renderState, 
                                                                                   sal_Int8				 							textDirection )
    {
        ENSURE_AND_THROW( xFont.is(), 
                          "CanvasHelper::drawText: font is NULL");

        if( needOutput() )
        {
            SurfaceGraphicsSharedPtr aGraphics( mpSurface->getGraphics() );

            setupGraphicsState( aGraphics, viewState, renderState );
        
            Gdiplus::SolidBrush aBrush( 
                Gdiplus::Color( 
                    tools::sequenceToArgb( mxDevice.getRef(),
                                           renderState.DeviceColor ) ) );

            CanvasFont::ImplRef pFont( 
                tools::canvasFontFromXFont(xFont) );

            // Move glyphs up, such that output happens at the font
            // baseline.
            Gdiplus::PointF aPoint( 0.0,
                                    static_cast<Gdiplus::REAL>(-(pFont->getFont()->GetSize()*
                                                                 pFont->getCellAscent() /
                                                                 pFont->getEmHeight())) );
            
            // TODO(F1): According to
            // http://support.microsoft.com/default.aspx?scid=kb;EN-US;Q307208,
            // we might have to revert to GDI and ExTextOut here,
            // since GDI+ takes the scalability a little bit too
            // far...

            // TODO(E1): Return value
            // TODO(F2): Proper layout (BiDi, CTL)! IMHO must use
            // DrawDriverString here, and perform layouting myself...
            (*aGraphics)->DrawString( text.Text.copy( text.StartPosition,
                                                      text.Length ).getStr(),
                                      text.Length,
                                      pFont->getFont().get(),
                                      aPoint,
                                      &aBrush );
        }

        return uno::Reference< rendering::XCachedPrimitive >(NULL);
    }

    uno::Reference< rendering::XCachedPrimitive > SAL_CALL CanvasHelper::drawTextLayout( const uno::Reference< rendering::XTextLayout >& 	xLayoutetText, 
                                                                                         const rendering::ViewState& 						viewState, 
                                                                                         const rendering::RenderState& 						renderState )
    {
        ENSURE_AND_THROW( xLayoutetText.is(), 
                          "CanvasHelper::drawTextLayout: layout is NULL");

        if( needOutput() )
        {
            SurfaceGraphicsSharedPtr aGraphics( mpSurface->getGraphics() );

            setupGraphicsState( aGraphics, viewState, renderState );
        
            Gdiplus::SolidBrush aBrush( 
                Gdiplus::Color( 
                    tools::sequenceToArgb( mxDevice.getRef(),
                                           renderState.DeviceColor ) ) );

            TextLayout* pTextLayout = NULL;

            uno::Reference< lang::XServiceInfo > xRef( xLayoutetText, 
                                                       uno::UNO_QUERY );

            if( xRef.is() && 
                xRef->getImplementationName().equals( ::rtl::OUString(RTL_CONSTASCII_USTRINGPARAM(TEXTLAYOUT_IMPLEMENTATION_NAME))) )
            {
                // TODO(Q1): Maybe use dynamic_cast here
                
                pTextLayout = static_cast<TextLayout*>(xLayoutetText.get());
            }
            else
            {
                OSL_TRACE("CanvasHelper::drawTextLayout(): TextLayout not compatible with this canvas");
                throw lang::IllegalArgumentException();
            }

            pTextLayout->draw( aGraphics, aBrush, viewState, renderState, maOutputOffset, mxDevice.getRef() );
        }

        return uno::Reference< rendering::XCachedPrimitive >(NULL);
    }

    uno::Reference< rendering::XCachedPrimitive > SAL_CALL CanvasHelper::drawBitmap( const uno::Reference< rendering::XBitmap >& 	xBitmap, 
                                                                                     const rendering::ViewState& 					viewState, 
                                                                                     const rendering::RenderState& 					renderState )
    {
        ENSURE_AND_THROW( xBitmap.is(), 
                          "CanvasHelper::drawBitmap: bitmap is NULL");

        if( needOutput() )
        {
            SurfaceGraphicsSharedPtr aGraphics( mpSurface->getGraphics() );

            setupGraphicsState( aGraphics, viewState, renderState );
        
            // first try to paint cached device-dependent bitmap
            Gdiplus::PointF	aPoint;
            BitmapSharedPtr pBitmap( tools::bitmapFromXBitmap( xBitmap ) );
            
            // TODO(E1): Return value
            (*aGraphics)->DrawImage( pBitmap.get(),
                                     aPoint );

            // TODO(P2): Consider using CachedBitmap here for certain
            // situations (but note that Graphics::DrawCachedBitmap()
            // does not work for any transformation, except pure
            // translations!)
        }

        // TODO(P1): Provide caching here.
        return uno::Reference< rendering::XCachedPrimitive >(NULL);
    }

    uno::Reference< rendering::XCachedPrimitive > SAL_CALL CanvasHelper::drawBitmapModulated( const uno::Reference< rendering::XBitmap >& 	xBitmap, 
                                                                                              const rendering::ViewState& 					viewState, 
                                                                                              const rendering::RenderState& 				renderState )
    {
        ENSURE_AND_THROW( xBitmap.is(), 
                          "CanvasHelper::drawBitmap: bitmap is NULL");

        // no color set -> this is equivalent to a plain drawBitmap(), then
        if( renderState.DeviceColor.getLength() < 3 )
            return drawBitmap( xBitmap, viewState, renderState );

        if( needOutput() )
        {            
            SurfaceGraphicsSharedPtr aGraphics( mpSurface->getGraphics() );

            setupGraphicsState( aGraphics, viewState, renderState );
        
            BitmapSharedPtr pBitmap( tools::bitmapFromXBitmap( xBitmap ) );
            Gdiplus::Rect aRect( 0, 0, 
                                 pBitmap->GetWidth(),
                                 pBitmap->GetHeight() );

            // Setup an ImageAttributes with an alpha-modulating
            // color matrix.
            const double nRed( renderState.DeviceColor[0] );
            const double nGreen( renderState.DeviceColor[1] );
            const double nBlue( renderState.DeviceColor[2] );
            const double nAlpha( renderState.DeviceColor.getLength() > 3 ? 
                                 renderState.DeviceColor[3] : 1.0 );

            Gdiplus::ImageAttributes aImgAttr;
            tools::setModulateImageAttributes( aImgAttr,
                                               nRed, nGreen, nBlue, nAlpha );

            // TODO(E1): Return value
            (*aGraphics)->DrawImage( pBitmap.get(), 
                                     aRect,
                                     0, 0,
                                     pBitmap->GetWidth(),
                                     pBitmap->GetHeight(),
                                     Gdiplus::UnitPixel,
                                     &aImgAttr,
                                     NULL,
                                     NULL );
        }

        // TODO(P1): Provide caching here.
        return uno::Reference< rendering::XCachedPrimitive >(NULL);
    }

    uno::Reference< rendering::XGraphicDevice > SAL_CALL CanvasHelper::getDevice()
    {
        return mxDevice.getRef();
    }

    void SAL_CALL CanvasHelper::copyRect( const uno::Reference< rendering::XBitmapCanvas >& sourceCanvas, 
                                          const geometry::RealRectangle2D& 					sourceRect, 
                                          const rendering::ViewState& 						sourceViewState, 
                                          const rendering::RenderState& 					sourceRenderState, 
                                          const geometry::RealRectangle2D& 					destRect, 
                                          const rendering::ViewState& 						destViewState, 
                                          const rendering::RenderState& 					destRenderState )
    {
        // TODO(F2): copyRect NYI
    }

    geometry::IntegerSize2D SAL_CALL CanvasHelper::getSize()
    {
        if( !mxDevice.get() )
            geometry::IntegerSize2D(1, 1); // we're disposed

        return ::basegfx::unotools::integerSize2DFromB2ISize( maSize );
    }

    uno::Reference< rendering::XBitmap > SAL_CALL CanvasHelper::getScaledBitmap( const geometry::RealSize2D&	newSize, 
                                                                                 sal_Bool 						beFast )
    {
        // TODO(F1):
        return uno::Reference< rendering::XBitmap >();
    }

    uno::Sequence< sal_Int8 > SAL_CALL CanvasHelper::getData( const geometry::IntegerRectangle2D& rect )
    {
        RTL_LOGFILE_CONTEXT( aLog, "::dxcanvas::CanvasHelper::getData()" );

        // TODO(F2): getData() for non-bitmaps NYI
        if( !mpBitmap.get() )
            return uno::Sequence< sal_Int8 >();

        uno::Sequence< sal_Int8 > aRes( (rect.X2-rect.X1)*(rect.Y2-rect.Y1)*4 ); // TODO(F1): Be format-agnostic here

        const Gdiplus::Rect aRect( tools::gdiPlusRectFromIntegerRectangle2D( rect ) );

        Gdiplus::BitmapData aBmpData;
        aBmpData.Width		 = rect.X2-rect.X1;
        aBmpData.Height		 = rect.Y2-rect.Y1;
        aBmpData.Stride 	 = 4*aBmpData.Width;
        aBmpData.PixelFormat = PixelFormat32bppARGB;
        aBmpData.Scan0		 = aRes.getArray(); 

        // TODO(F1): Support more pixel formats natively

        // read data from bitmap
        if( Gdiplus::Ok != mpBitmap->LockBits( &aRect,
                                               Gdiplus::ImageLockModeRead | Gdiplus::ImageLockModeUserInputBuf,
                                               PixelFormat32bppARGB, // TODO(F1): Adapt to
                                               						 // Graphics native
				                    	                             // format/change
                					                                 // getMemoryLayout
                                               &aBmpData ) )
        {
            // failed to lock, bail out
            return uno::Sequence< sal_Int8 >();
        }

        mpBitmap->UnlockBits( &aBmpData );

        return aRes;
    }

    void SAL_CALL CanvasHelper::setData( const uno::Sequence< sal_Int8 >& 		data, 
                                         const geometry::IntegerRectangle2D& 	rect )
    {
        RTL_LOGFILE_CONTEXT( aLog, "::dxcanvas::CanvasHelper::setData()" );

        // TODO(F2): setData() for non-bitmaps NYI
        if( !mpBitmap.get() )
            return;

        const Gdiplus::Rect aRect( tools::gdiPlusRectFromIntegerRectangle2D( rect ) );

        Gdiplus::BitmapData aBmpData;
        aBmpData.Width		 = rect.X2-rect.X1;
        aBmpData.Height		 = rect.Y2-rect.Y1;
        aBmpData.Stride 	 = 4*aBmpData.Width;
        aBmpData.PixelFormat = PixelFormat32bppARGB;
        aBmpData.Scan0		 = const_cast< uno::Sequence< sal_Int8 >& >(data).getArray(); 

        // TODO(F1): Support more pixel formats natively

        if( Gdiplus::Ok != mpBitmap->LockBits( &aRect,
                                               Gdiplus::ImageLockModeWrite | Gdiplus::ImageLockModeUserInputBuf,
                                               PixelFormat32bppARGB, // TODO: Adapt to
                                               						 // Graphics native
                                                  					 // format/change
                                                  					 // getMemoryLayout
                                               &aBmpData ) )
        {
            throw uno::RuntimeException();
        }

        // commit data to bitmap
        mpBitmap->UnlockBits( &aBmpData );
    }

    void SAL_CALL CanvasHelper::setPixel( const uno::Sequence< sal_Int8 >& 	color, 
                                          const geometry::IntegerPoint2D& 	pos )
    {
        RTL_LOGFILE_CONTEXT( aLog, "::dxcanvas::CanvasHelper::setPixel()" );

        // TODO(F2): setPixel() for non-bitmaps NYI
        if( !mpBitmap.get() )
            return;

        const geometry::IntegerSize2D aSize( getSize() );

        CHECK_AND_THROW( pos.X >= 0 && pos.X < aSize.Width, 
                         "CanvasHelper::setPixel: X coordinate out of bounds" );
        CHECK_AND_THROW( pos.Y >= 0 && pos.Y < aSize.Height, 
                         "CanvasHelper::setPixel: Y coordinate out of bounds" );
        CHECK_AND_THROW( color.getLength() > 3, 
                         "CanvasHelper::setPixel: not enough color components" );

        if( Gdiplus::Ok != mpBitmap->SetPixel( pos.X, pos.Y, 
                                               Gdiplus::Color( tools::sequenceToArgb( mxDevice.getRef(),
                                                                                      color ) ) ) )
        {
            throw uno::RuntimeException();
        }
    }

    uno::Sequence< sal_Int8 > SAL_CALL CanvasHelper::getPixel( const geometry::IntegerPoint2D& pos )
    {
        RTL_LOGFILE_CONTEXT( aLog, "::dxcanvas::CanvasHelper::getPixel()" );

        // TODO(F2): setPixel() for non-bitmaps NYI
        if( !mpBitmap.get() )
            return uno::Sequence< sal_Int8 >();

        const geometry::IntegerSize2D aSize( getSize() );

        CHECK_AND_THROW( pos.X >= 0 && pos.X < aSize.Width, 
                         "CanvasHelper::getPixel: X coordinate out of bounds" );
        CHECK_AND_THROW( pos.Y >= 0 && pos.Y < aSize.Height, 
                         "CanvasHelper::getPixel: Y coordinate out of bounds" );

        Gdiplus::Color aColor;

        if( Gdiplus::Ok != mpBitmap->GetPixel( pos.X, pos.Y, &aColor ) )
            return uno::Sequence< sal_Int8 >();
        
        return tools::argbToIntSequence( mxDevice.getRef(),
                                         aColor.GetValue() );
    }

    uno::Reference< rendering::XBitmapPalette > SAL_CALL CanvasHelper::getPalette()
    {
        // TODO(F1): Palette bitmaps NYI
        return uno::Reference< rendering::XBitmapPalette >();
    }

    rendering::IntegerBitmapLayout SAL_CALL CanvasHelper::getMemoryLayout()
    {
        // TODO(F1): finish memory layout initialization
        rendering::IntegerBitmapLayout aLayout;

        const geometry::IntegerSize2D aSize( getSize() );

        aLayout.ScanLines = aSize.Width;
        aLayout.ScanLineBytes = aSize.Height * 4;
        aLayout.ScanLineStride = aLayout.ScanLineBytes;
        aLayout.Format = 0;
        aLayout.NumComponents = 4;
        aLayout.ComponentMasks = uno::Sequence<sal_Int64>();
        aLayout.Endianness = 0;
        aLayout.IsPseudoColor = false;

        return aLayout;
    }


    // private helper
    // --------------------------------------------------

    Gdiplus::CompositingMode CanvasHelper::calcCompositingMode( sal_Int8 nMode )
    {
        Gdiplus::CompositingMode aRet( Gdiplus::CompositingModeSourceOver );

        switch( nMode )
        {
            case rendering::CompositeOperation::OVER:
                aRet = Gdiplus::CompositingModeSourceOver;
                // FALLTHROUGH intended

            case rendering::CompositeOperation::CLEAR:
                // FALLTHROUGH intended
            case rendering::CompositeOperation::SOURCE:
                // FALLTHROUGH intended
            case rendering::CompositeOperation::DESTINATION:
                // FALLTHROUGH intended
            case rendering::CompositeOperation::UNDER:
                // FALLTHROUGH intended
            case rendering::CompositeOperation::INSIDE:
                // FALLTHROUGH intended
            case rendering::CompositeOperation::INSIDE_REVERSE:
                // FALLTHROUGH intended
            case rendering::CompositeOperation::OUTSIDE:
                // FALLTHROUGH intended
            case rendering::CompositeOperation::OUTSIDE_REVERSE:
                // FALLTHROUGH intended
            case rendering::CompositeOperation::ATOP:
                // FALLTHROUGH intended
            case rendering::CompositeOperation::ATOP_REVERSE:
                // FALLTHROUGH intended
            case rendering::CompositeOperation::XOR:
                // FALLTHROUGH intended
            case rendering::CompositeOperation::ADD:
                // FALLTHROUGH intended
            case rendering::CompositeOperation::SATURATE:
                // TODO(F2): Problem, because GDI+ only knows about two compositing modes
                aRet = Gdiplus::CompositingModeSourceOver;
                break;
                
            default:
                ENSURE_AND_THROW( false, "CanvasHelper::calcCompositingMode: unexpected mode" );
                break;
        }

        return aRet;
    }

    void CanvasHelper::setupGraphicsState( SurfaceGraphicsSharedPtr&		rGraphics,
                                           const rendering::ViewState& 		viewState, 
                                           const rendering::RenderState&	renderState )
    {
        ENSURE_AND_THROW( needOutput(), 
                          "CanvasHelper::setupGraphicsState: primary graphics invalid" );
        ENSURE_AND_THROW( mxDevice.get(), 
                          "CanvasHelper::setupGraphicsState: reference device invalid" );

        // setup view transform first. Clipping e.g. depends on it
        ::basegfx::B2DHomMatrix aTransform;
        ::canvas::tools::getViewStateTransform(aTransform, viewState);

        // add output offset
        if( !maOutputOffset.equalZero() )
        {
            ::basegfx::B2DHomMatrix aOutputOffset;
            aOutputOffset.translate( maOutputOffset.getX(), 
                                     maOutputOffset.getY() );
            
            aTransform = aOutputOffset * aTransform;
        }

        Gdiplus::Matrix aMatrix;
        tools::gdiPlusMatrixFromB2DHomMatrix( aMatrix, aTransform );

        if( Gdiplus::Ok != (*rGraphics)->SetTransform( &aMatrix ) )
            throw uno::RuntimeException();

        // setup view and render state clipping
        if( Gdiplus::Ok != (*rGraphics)->ResetClip() )
        {
            throw uno::RuntimeException();
        }

        if( viewState.Clip.is() )
        {
            GraphicsPathSharedPtr aClipPath( tools::graphicsPathFromXPolyPolygon2D( viewState.Clip ) );

            // TODO(P3): Cache clip. SetClip( GraphicsPath ) performs abyssmally on GDI+. 
            // Try SetClip( Rect ) or similar for simple clip paths (need some support in 
            // LinePolyPolygon, then)
            if( Gdiplus::Ok != (*rGraphics)->SetClip( aClipPath.get(), 
                                                      Gdiplus::CombineModeIntersect ) )
            {
                throw uno::RuntimeException();
            }
        }

        // setup overall transform only now. View clip above was relative to 
        // view transform
        ::canvas::tools::mergeViewAndRenderTransform(aTransform,
                                                     viewState,
                                                     renderState);

        // add output offset
        if( !maOutputOffset.equalZero() )
        {
            ::basegfx::B2DHomMatrix aOutputOffset;
            aOutputOffset.translate( maOutputOffset.getX(), 
                                     maOutputOffset.getY() );
            
            aTransform = aOutputOffset * aTransform;
        }

        tools::gdiPlusMatrixFromB2DHomMatrix( aMatrix, aTransform );

        if( Gdiplus::Ok != (*rGraphics)->SetTransform( &aMatrix ) )
            throw uno::RuntimeException();

        if( renderState.Clip.is() )
        {
            GraphicsPathSharedPtr aClipPath( tools::graphicsPathFromXPolyPolygon2D( renderState.Clip ) );

            // TODO(P3): Cache clip. SetClip( GraphicsPath ) performs abyssmally on GDI+. 
            // Try SetClip( Rect ) or similar for simple clip paths (need some support in 
            // LinePolyPolygon, then)
            if( Gdiplus::Ok != (*rGraphics)->SetClip( aClipPath.get(), 
                                                      Gdiplus::CombineModeIntersect ) )
            {
                throw uno::RuntimeException();
            }
        }

        // setup compositing
        const Gdiplus::CompositingMode eCompositing( calcCompositingMode( renderState.CompositeOperation ) );
        if( Gdiplus::Ok != (*rGraphics)->SetCompositingMode( eCompositing ) )
            throw uno::RuntimeException();
    }

    void CanvasHelper::flush() const
    {
        if( needOutput() )
            (*mpSurface->getGraphics())->Flush( Gdiplus::FlushIntentionSync );
    }

    SurfaceSharedPtr CanvasHelper::getSurface() const
    {
        return mpSurface;
    }

    bool CanvasHelper::needOutput() const
    {
        return mpSurface.get() != NULL;
    }

}
