/*************************************************************************
 *
 *  $RCSfile: redrawmanagement.hxx,v $
 *
 *  $Revision: 1.2 $
 *
 *  last change: $Author: rt $ $Date: 2004/11/26 17:03:02 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/

#ifndef _CANVAS_INTERNAL_REDRAWMANAGEMENT_HXX
#define _CANVAS_INTERNAL_REDRAWMANAGEMENT_HXX

#include <canvas/debug.hxx>

#ifndef _BGFX_POINT_B2DPOINT_HXX
#include <basegfx/point/b2dpoint.hxx>
#endif
#ifndef _BGFX_VECTOR_B2DSIZE_HXX
#include <basegfx/vector/b2dsize.hxx>
#endif
#ifndef _BGFX_RANGE_B2DRECTANGLE_HXX
#include <basegfx/range/b2drectangle.hxx>
#endif

#include <algorithm>
#include <vector>


namespace canvas
{

    namespace internal
    {
        /** Helper struct for SpriteTracer template

        	This templated struct stores change information to a
        	sprite's visual appearance (move, content updated, and the like).

            @tpl SpriteRefType
            Reference to sprite. Simply plug in the corresponding
            (smart) pointer for your sprite class type here.
         */
        template< typename SpriteRefType > struct SpriteChangeRecord
        {
            typedef enum{ none=0, move, update } ChangeType;

            SpriteChangeRecord() :
                meChangeType( none ),
                mpAffectedSprite(),
                maOldPos(),
                maNewPos(),
                maUpdateArea()
            {
            }

            SpriteChangeRecord( const SpriteRefType& 		rSprite, 
                                const ::basegfx::B2DPoint& 	rOldPos,
                                const ::basegfx::B2DPoint& 	rNewPos ) :
                meChangeType( move ),
                mpAffectedSprite( rSprite ),
                maOldPos( rOldPos ),
                maNewPos( rNewPos ),
                maUpdateArea()
            {
            }

            SpriteChangeRecord( const SpriteRefType& 			rSprite, 
                                const ::basegfx::B2DPoint& 		rPos,
                                const ::basegfx::B2DRectangle& 	rUpdateArea ) :
                meChangeType( update ),
                mpAffectedSprite( rSprite ),
                maOldPos( rPos ),
                maNewPos(),
                maUpdateArea( rUpdateArea )
            {
            }

            SpriteRefType getSprite() const { return mpAffectedSprite; }

            ChangeType				meChangeType;
            SpriteRefType			mpAffectedSprite;
            ::basegfx::B2DPoint		maOldPos;
            ::basegfx::B2DPoint		maNewPos;
            ::basegfx::B2DRectangle	maUpdateArea;
        };

        /** Helper class to condense sprite updates into a single action

        	This template class tracks the sprite changes over the
        	recorded change list, and generates a single update action
        	from that (note that per screen update, several moves,
        	visibility changes and content updates might happen)

            @tpl SpriteRefType
            Reference to sprite. Simply plug in the corresponding
            (smart) pointer for your sprite class type here. The
            dereferenced SpriteRefType object must provide a getSize()
            method, returning something that is convertible to
            ::basegfx::B2DSize.

            @tpl UpdateCollector
            This is the sprite update range collector type to operate
            with. The type given here must provide an addRange( const
            ::basegfx::B2DRange& rRect, const SpriteRefType& rSprite)
            method, where rRect denotes the area that needs update,
            and rSprite the sprite that is to be drawn.
         */
        template< typename SpriteRefType, typename UpdateCollector > class SpriteTracer
        {
        public:
            typedef SpriteChangeRecord< SpriteRefType > SpriteChangeRecordType;

            SpriteTracer( const SpriteRefType& rAffectedSprite ) :
                mpAffectedSprite(rAffectedSprite),
                maMoveStart(),
                maMoveEnd(),
                maUpdateArea(),
                mbFirstMove( true ),
                mbIsMove( false )
            {
            }

            void operator()( const SpriteChangeRecordType& rSpriteRecord )
            {
                // only deal with change events from the currently
                // affected sprite
                if( rSpriteRecord.mpAffectedSprite == mpAffectedSprite )
                {
                    switch( rSpriteRecord.meChangeType )
                    {
                        case SpriteChangeRecordType::move:
                            if( mbFirstMove )
                            {
                                maMoveStart = rSpriteRecord.maOldPos;
                                mbFirstMove = false;
                            }

                            maMoveEnd = rSpriteRecord.maNewPos;
                            mbIsMove = true;
                            break;

                        case SpriteChangeRecordType::update:
                            // union new update area with already
                            // existing one. Should the sprite be
                            // moved, this will of course generate
                            // grossly wrong results, which we luckily
                            // ignore below in commit() (a move always
                            // causes the whole sprite to need redraw,
                            // anyway. Thus, maUpdateArea is not used
                            // at all, then)
                            maUpdateArea.expand( rSpriteRecord.maUpdateArea );
                            break;

                        default:
                            ENSURE_AND_THROW( false, 
                                              "Unexpected case in SpriteUpdater::operator()" );
                            break;
                    }
                }
            }

            void commit( UpdateCollector& rUpdateCollector ) const
            {
                if( mbIsMove )
                {
                    const ::basegfx::B2DSize aSpriteSize( mpAffectedSprite->getSize() );
                    const ::basegfx::B2DRectangle aOldArea( 
                        maMoveStart.getX(),
                        maMoveStart.getY(),
                        maMoveStart.getX() + aSpriteSize.getX(), 
                        maMoveStart.getY() + aSpriteSize.getY() );
                    const ::basegfx::B2DRectangle aNewArea( 
                        maMoveEnd.getX(),
                        maMoveEnd.getY(),
                        maMoveEnd.getX() + aSpriteSize.getX(), 
                        maMoveEnd.getY() + aSpriteSize.getY() );

                    // do new and old sprite area overlap? if yes, we have
                    // to prepare the whole area
                    if( aOldArea.overlaps( aNewArea ) )
                    {
                        ::basegfx::B2DRectangle aTemp( aNewArea );
                        aTemp.expand( aOldArea );

                        rUpdateCollector.addRange( aTemp,
                                                   mpAffectedSprite );
                    }
                    else
                    {
                        // we can update new and old position separately

                        // first, draw the new sprite position
                        rUpdateCollector.addRange( aNewArea, 
                                                   mpAffectedSprite );

                        // then, clear the old place (looks smoother
                        // this way)
                        rUpdateCollector.addRange( aOldArea, 
                                                   SpriteRefType() );
                    }
                }
                else if( !maUpdateArea.isEmpty() )
                {
                    rUpdateCollector.addRange( maUpdateArea, 
                                               mpAffectedSprite );
                }
            }

        private:
            SpriteRefType			mpAffectedSprite;
            ::basegfx::B2DPoint		maMoveStart;
            ::basegfx::B2DPoint		maMoveEnd;
            ::basegfx::B2DRectangle	maUpdateArea;
            bool					mbFirstMove;
            bool					mbIsMove;
        
        };


        /** SpriteChecker functor, which for every sprite checks the
            given update vector for necessary screen updates

            @tpl SpriteRefType
            Reference to sprite. Simply plug in the corresponding
            (smart) pointer for your sprite class type here. The
            dereferenced SpriteRefType object must provide a getSize()
            method, returning something that is convertible to
            ::basegfx::B2DSize.

            @tpl RecordContainerType
            An STL container (or something concept-compatible), which
            contains the SpriteChangeRecord entries

            @tpl UpdaterType
            Update target type. This class must provide an addRange()
            method, see SpriteTracer template.
            
         */
        template< typename SpriteRefType, class RecordContainerType, class UpdateCollector > class SpriteUpdater
        {
        public:
            typedef SpriteTracer< SpriteRefType, UpdateCollector >	SpriteTracerType;

            /** Generate update area list
                
            	@param rUpdater
                Reference to an updater object, which will receive the
                update areas.

                @param rChangeContainer
                Container with all sprite change requests

             */
            SpriteUpdater( UpdateCollector&				rUpdater,
                           const RecordContainerType&	rChangeContainer ) :
                mrUpdater( rUpdater ),
                mrChangeContainer( rChangeContainer )
            {
            }

            /** Call this method for every sprite on your screen

            	This method scans the change container, collecting all
            	update info for the given sprite into one or two
            	update operations, which in turn are inserted into the
            	connected ranges processor.

                @param rSprite
                Current sprite to collect update info for.
             */
            void operator()( const SpriteRefType& rSprite )
            {
                const SpriteTracerType aSpriteTracer( 
                    ::std::for_each( mrChangeContainer.begin(), 
                                     mrChangeContainer.end(), 
                                     SpriteTracerType( rSprite ) ) );

                aSpriteTracer.commit( mrUpdater );
            }

        private:
            UpdateCollector&			mrUpdater;
            const RecordContainerType&	mrChangeContainer;
        };

    }
}

#endif /* _CANVAS_INTERNAL_REDRAWMANAGEMENT_HXX */
