/*************************************************************************
 *
 *  $RCSfile: cntrange.cxx,v $
 *
 *  $Revision: 1.3 $
 *
 *  last change: $Author: kso $ $Date: 2001/07/25 15:09:44 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/

#ifndef _STREAM_HXX //autogen
#include <tools/stream.hxx>
#endif

#ifndef _CNTRANGE_HXX
#include <cntrange.hxx>
#endif
#include <string>
#include <algorithm>

using namespace chaos;

//============================================================================
//
//	CntRangesItem
//
//============================================================================

TYPEINIT1_AUTOFACTORY(CntRangesItem, SfxPoolItem);

//============================================================================
CntRangesItem::CntRangesItem(USHORT nWhich, SvStream & rStream):
	SfxPoolItem(nWhich),
	m_pFirstRange(0),
	m_nRangeCount(0),
	m_nTotalCount(0)
{
	USHORT nCount = 0;
	rStream >> nCount;
	while (nCount--)
	{
		ULONG nFrom = 0;
		ULONG nTo = 0;
		rStream >> nFrom >> nTo;
		InsertRange(nFrom, nTo);
	}
}

//============================================================================
CntRangesItem::CntRangesItem(const CntRangesItem & rRanges):
	SfxPoolItem(rRanges),
	m_nRangeCount(rRanges.m_nRangeCount),
	m_nTotalCount(rRanges.m_nTotalCount)
{
	Range * pSource = rRanges.m_pFirstRange;
	Range ** ppDest = &m_pFirstRange;
	while (pSource)
	{
		*ppDest = new Range;
		(*ppDest)->m_nFrom = pSource->m_nFrom;
		(*ppDest)->m_nTo = pSource->m_nTo;
		pSource = pSource->m_pNext;
		ppDest = &(*ppDest)->m_pNext;
	}
	*ppDest = 0;
}

//============================================================================
// virtual
CntRangesItem::~CntRangesItem()
{
	while (m_pFirstRange)
	{
		Range * pNext = m_pFirstRange->m_pNext;
		delete m_pFirstRange;
		m_pFirstRange = pNext;
	}
}

//============================================================================
CntRangesItem & CntRangesItem::operator =(const CntRangesItem & rRanges)
{
	if (this != &rRanges)
	{
		SetWhich(rRanges.Which());
		while (m_pFirstRange)
		{
			Range * pNext = m_pFirstRange->m_pNext;
			delete m_pFirstRange;
			m_pFirstRange = pNext;
		}
		Range * pSource = rRanges.m_pFirstRange;
		Range ** ppDest = &m_pFirstRange;
		while (pSource)
		{
			*ppDest = new Range;
			(*ppDest)->m_nFrom = pSource->m_nFrom;
			(*ppDest)->m_nTo = pSource->m_nTo;
			pSource = pSource->m_pNext;
			ppDest = &(*ppDest)->m_pNext;
		}
		*ppDest = 0;
		m_nRangeCount = rRanges.m_nRangeCount;
		m_nTotalCount = rRanges.m_nTotalCount;
	}
	return *this;
}

//============================================================================
// virtual
int CntRangesItem::operator ==(const SfxPoolItem & rItem) const
{
	const CntRangesItem * pRanges = PTR_CAST(CntRangesItem, &rItem);
	if (!pRanges || m_nRangeCount != pRanges->m_nRangeCount
		|| m_nTotalCount != pRanges->m_nTotalCount)
		return FALSE;
	Range * pRange1 = m_pFirstRange;
	Range * pRange2 = pRanges->m_pFirstRange;
	while (pRange1)
	{
		if (!pRange2 || pRange1->m_nFrom != pRange2->m_nFrom
			|| pRange1->m_nTo != pRange2->m_nTo)
			return FALSE;
		pRange1 = pRange1->m_pNext;
		pRange2 = pRange2->m_pNext;
	}
	return !pRange2;
}

//============================================================================
// virtual
SfxPoolItem * CntRangesItem::Clone(SfxItemPool *) const
{
	return new CntRangesItem(*this);
}

//============================================================================
// virtual
SfxPoolItem * CntRangesItem::Create(SvStream & rStream, USHORT) const
{
	return new CntRangesItem(Which(), rStream);
}

//============================================================================
// virtual
SvStream & CntRangesItem::Store(SvStream & rStream, USHORT) const
{
	rStream << m_nRangeCount;
	for (Range * pRange = m_pFirstRange; pRange; pRange = pRange->m_pNext)
		rStream << pRange->m_nFrom << pRange->m_nTo;
	return rStream;
}

//============================================================================
void CntRangesItem::GetRange(USHORT nIndex, ULONG & rFrom, ULONG & rTo) const
{
	for (Range * pRange = m_pFirstRange; pRange; pRange = pRange->m_pNext)
		if (!nIndex--)
		{
			rFrom = pRange->m_nFrom;
			rTo = pRange->m_nTo;
			break;
		}
}

//============================================================================
void CntRangesItem::RemoveRange(USHORT nIndex)
{
	for (Range ** ppRange = &m_pFirstRange; *ppRange;
		 ppRange = &(*ppRange)->m_pNext)
		if (!nIndex--)
		{
			Range * pRemovedRange = *ppRange;
			*ppRange = pRemovedRange->m_pNext;
			m_nTotalCount
			 -= pRemovedRange->m_nTo - pRemovedRange->m_nFrom + 1;
			delete pRemovedRange;
			--m_nRangeCount;
			break;
		}
}

//============================================================================
ULONG CntRangesItem::TotalMax() const
{
	if (m_pFirstRange)
	{
		Range * pRange = m_pFirstRange;
		while (pRange->m_pNext)
			pRange = pRange->m_pNext;
		return pRange->m_nTo;
	}
	else
		return 0;
}

//============================================================================
BOOL CntRangesItem::IsWithinRanges(ULONG nValue) const
{
	for (Range * pRange = m_pFirstRange; pRange; pRange = pRange->m_pNext)
		if (nValue <= pRange->m_nTo)
			return nValue >= pRange->m_nFrom;
	return FALSE;
}

//============================================================================
BOOL CntRangesItem::InsertRange(ULONG nFrom, ULONG nTo)
{
	if (nFrom > nTo)
		return FALSE;

	// Skip ranges that completely lie below and do not touch [nFrom..nTo]:
	Range ** ppRange = &m_pFirstRange;
	while (*ppRange && (*ppRange)->m_nTo + 1 < nFrom)
		ppRange = &(*ppRange)->m_pNext;

	// If there are no ranges left, or if the next range starts above and does
	// not touch [nFrom..nTo], no ranges intersect with or touch [nFrom..nTo]:
	if (!*ppRange || (*ppRange)->m_nFrom > nTo + 1)
	{
		Range * pNewRange = new Range;
		pNewRange->m_nFrom = nFrom;
		pNewRange->m_nTo = nTo;
		pNewRange->m_pNext = *ppRange;
		*ppRange = pNewRange;
		++m_nRangeCount;
		m_nTotalCount += nTo - nFrom + 1;
		return TRUE;
	}

	// The first range that intersects with or touches [nFrom..nTo] is found.
	// First, decrease its lower bound if necessary:
	BOOL bAdded = FALSE;
	if (nFrom < (*ppRange)->m_nFrom)
	{
		m_nTotalCount += (*ppRange)->m_nFrom - nFrom;
		(*ppRange)->m_nFrom = nFrom;
		bAdded = TRUE;
	}

	// Second, remove all following ranges that at least touch nTo, adjusting
	// nTo if necessary:
	Range * pNextRange;
	while ((pNextRange = (*ppRange)->m_pNext)
		   && pNextRange->m_nFrom <= nTo + 1)
	{
		if (nTo < pNextRange->m_nTo)
			nTo = pNextRange->m_nTo;
		--m_nRangeCount;
		m_nTotalCount -= pNextRange->m_nTo - pNextRange->m_nFrom + 1;
		(*ppRange)->m_pNext = pNextRange->m_pNext;
		delete pNextRange;
	}

	// Third, increase the upper bound if necessary:
	if (nTo > (*ppRange)->m_nTo)
	{
		m_nTotalCount += nTo - (*ppRange)->m_nTo;
		(*ppRange)->m_nTo = nTo;
		bAdded = TRUE;
	}
	return bAdded;
}

//============================================================================
BOOL CntRangesItem::RemoveRange(ULONG nFrom, ULONG nTo)
{
	if (nFrom > nTo)
		return FALSE;

	// Skip ranges that completely lie below [nFrom..nTo]:
	Range ** ppRange = &m_pFirstRange;
	while (*ppRange && (*ppRange)->m_nTo < nFrom)
		ppRange = &(*ppRange)->m_pNext;

	// If there are no ranges left, or if the next range starts above
	// [nFrom..nTo], no ranges intersect with [nFrom..nTo]:
	if (!*ppRange || (*ppRange)->m_nFrom > nTo)
		return FALSE;

	// The first range that intersects with [nFrom..nTo] is found.  There are
	// tree cases:
	// 1  Its right end lies within [nFrom..nTo].
	// 2.1  Its right end lies above [nFrom..nto] and its left end lies below
	//      [nFrom..nTo].
	// 2.2  Its right end lies above [nFrom..nTo] and its left end lies within
	//      [nFrom..nTo].
	if ((*ppRange)->m_nTo <= nTo)  // Case 1
	{
		// If the range starts below [nFrom..nTo], keep the 'left end' of it:
		BOOL bRemoved = FALSE;
		if ((*ppRange)->m_nFrom < nFrom)
		{
			m_nTotalCount -= (*ppRange)->m_nTo - nFrom + 1;
			(*ppRange)->m_nTo = nFrom - 1;
			bRemoved = TRUE;
			ppRange = &(*ppRange)->m_pNext;
		}

		// Remove ranges that completely lie within [nFrom..nTo]:
		while (*ppRange && (*ppRange)->m_nTo <= nTo)
		{
			Range * pRemove = *ppRange;
			*ppRange = (*ppRange)->m_pNext;
			--m_nRangeCount;
			m_nTotalCount -= pRemove->m_nTo - pRemove->m_nFrom + 1;
			delete pRemove;
			bRemoved = TRUE;
		}

		// If there is a range that intersects with [nFrom..nTo] and ends
		// above [nFrom..nTo], keep the 'right end' of it:
		if (*ppRange && (*ppRange)->m_nFrom <= nTo && (*ppRange)->m_nTo > nTo)
		{
			m_nTotalCount -= nTo - (*ppRange)->m_nFrom + 1;
			(*ppRange)->m_nFrom = nTo + 1;
			bRemoved = TRUE;
		}
		return bRemoved;
	}
	else if ((*ppRange)->m_nFrom < nFrom) // Case 2.1
	{
		// The range encompasses [nFrom..nTo]; it is replaced with two ranges
		// (the 'left and right ends'):
		Range * pRight = new Range;
		pRight->m_nFrom = nTo + 1;
		pRight->m_nTo = (*ppRange)->m_nTo;
		pRight->m_pNext = (*ppRange)->m_pNext;
		(*ppRange)->m_nTo = nFrom - 1;
		(*ppRange)->m_pNext = pRight;
		++m_nRangeCount;
		m_nTotalCount -= nTo - nFrom + 1;
		return TRUE;
	}
	else // Case 2.2
	{
		// The range starts within [nFrom..nTo] and ends above [nFrom..nTo],
		// so keep its 'right end':
		m_nTotalCount -= nTo - (*ppRange)->m_nFrom + 1;
		(*ppRange)->m_nFrom = nTo + 1;
		return TRUE;
	}
}

//============================================================================
void CntRangesItem::AddRanges(const CntRangesItem & rRanges)
{
	for (Range * pRange = rRanges.m_pFirstRange; pRange;
		 pRange = pRange->m_pNext)
		InsertRange(pRange->m_nFrom, pRange->m_nTo);
}

//============================================================================
void CntRangesItem::SubtractRanges(const CntRangesItem & rRanges)
{
	for (Range * pRange = rRanges.m_pFirstRange; pRange;
		 pRange = pRange->m_pNext)
		RemoveRange(pRange->m_nFrom, pRange->m_nTo);
}

//============================================================================
CntRangesItem * CntRangesItem::GetIntersectRanges(const CntRangesItem &
												   rRanges) const
{
	CntRangesItem * pResultRanges = 0;
	Range ** ppLastResultRange;
	Range * pRange1 = m_pFirstRange;
	Range * pRange2 = rRanges.m_pFirstRange;
	while (pRange1 && pRange2)
		if (pRange1->m_nTo < pRange2->m_nFrom)
			pRange1 = pRange1->m_pNext;
		else if (pRange2->m_nTo < pRange1->m_nFrom)
			pRange2 = pRange2->m_pNext;
		else
		{
			ULONG nFrom = std::max(pRange1->m_nFrom, pRange2->m_nFrom);
			ULONG nTo;
			if (pRange1->m_nTo < pRange2->m_nTo)
			{
				nTo = pRange1->m_nTo;
				pRange1 = pRange1->m_pNext;
			}
			else
			{
				nTo = pRange2->m_nTo;
				if (pRange1->m_nTo == pRange2->m_nTo)
					pRange1 = pRange1->m_pNext;
				pRange2 = pRange2->m_pNext;
			}
			if (!pResultRanges)
			{
				pResultRanges = new CntRangesItem(Which());
				ppLastResultRange = &pResultRanges->m_pFirstRange;
			}
			*ppLastResultRange = new Range;
			(*ppLastResultRange)->m_nFrom = nFrom;
			(*ppLastResultRange)->m_nTo = nTo;
			ppLastResultRange = &(*ppLastResultRange)->m_pNext;
			++pResultRanges->m_nRangeCount;
			pResultRanges->m_nTotalCount += nTo - nFrom + 1;
		}
	if (pResultRanges)
		*ppLastResultRange = 0;
	return pResultRanges;
}

//============================================================================
CntRangesItem * CntRangesItem::GetSubtractRanges(const CntRangesItem &
												  rRanges) const
{
	CntRangesItem * pResultRanges = (CntRangesItem *) Clone();
	pResultRanges->SubtractRanges(rRanges);
	if (pResultRanges->m_nRangeCount)
		return pResultRanges;
	else
	{
		delete pResultRanges;
		return 0;
	}
}

//============================================================================
CntRangesItem * CntRangesItem::GetDifferenceRanges(ULONG nFrom, ULONG nTo)
	const
{
	CntRangesItem aRanges(Which());
	aRanges.InsertRange(nFrom, nTo);
	return aRanges.GetSubtractRanges(*this);
}

