// -*- mode: C++; tab-width: 2; -*-
// vi: set ts=2:
//
// --------------------------------------------------------------------------
//                   OpenMS Mass Spectrometry Framework 
// --------------------------------------------------------------------------
//  Copyright (C) 2003-2011 -- Oliver Kohlbacher, Knut Reinert
//
//  This library is free software; you can redistribute it and/or
//  modify it under the terms of the GNU Lesser General Public
//  License as published by the Free Software Foundation; either
//  version 2.1 of the License, or (at your option) any later version.
//
//  This library is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//  Lesser General Public License for more details.
//
//  You should have received a copy of the GNU Lesser General Public
//  License along with this library; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
// --------------------------------------------------------------------------
// $Maintainer: Chris Bielow $
// $Authors:  Clemens Groepl, Chris Bielow $
// --------------------------------------------------------------------------
#include <OpenMS/CONCEPT/VersionInfo.h>
#include <OpenMS/DATASTRUCTURES/String.h>
#include <OpenMS/CONCEPT/Exception.h>

#include <cstdlib>
#include <fstream>


// these headers are generated by the build system
// and therefore intentionally break the naming convention (tagging them as automatically build)
#include <OpenMS/openms_package_version.h>
#include <OpenMS/openms_svn_revision.h>

using namespace std;

namespace OpenMS
{
	const VersionInfo::VersionDetails VersionInfo::VersionDetails::EMPTY;
  
  bool VersionInfo::VersionDetails::operator<(const VersionInfo::VersionDetails& rhs) const
  {
    return (    (this->version_major  < rhs.version_major)
             || (this->version_major == rhs.version_major && this->version_minor  < rhs.version_minor)
             || (this->version_major == rhs.version_major && this->version_minor == rhs.version_minor && this->version_patch < rhs.version_patch));
  }
  
  bool VersionInfo::VersionDetails::operator==(const VersionInfo::VersionDetails& rhs) const
  {
    return (this->version_major == rhs.version_major && this->version_minor == rhs.version_minor && this->version_patch == rhs.version_patch);
  }

  bool VersionInfo::VersionDetails::operator>(const VersionInfo::VersionDetails& rhs) const
  {
    return !(*this < rhs || *this == rhs);
  }


  VersionInfo::VersionDetails VersionInfo::VersionDetails::create(const String& version)
  {
    VersionInfo::VersionDetails result;

    size_t first_dot = version.find('.');
    // we demand at least one "."
    if (first_dot == string::npos) return VersionInfo::VersionDetails::EMPTY;
    try
    {
      result.version_major = String(version.substr(0, first_dot)).toInt();
    }
    catch (Exception::ConversionError& /*e*/)
    {
      return VersionInfo::VersionDetails::EMPTY;    	
    }

    // returns npos if no second "." is found - which does not hurt
    size_t second_dot = version.find('.', first_dot + 1);
    try
    {
      result.version_minor = String(version.substr(first_dot+1, second_dot)).toInt();
    }
    catch (Exception::ConversionError& /*e*/)
    {
      return VersionInfo::VersionDetails::EMPTY;    	
    }

    // if there is no second dot: return
    if (second_dot == string::npos) return result;
    
    // returns npos if no third "." is found - which does not hurt
    size_t third_dot = version.find('.', second_dot + 1);
    try
    {
      result.version_patch = String(version.substr(second_dot+1, third_dot)).toInt();
    }
    catch (Exception::ConversionError& /*e*/)
    {
      return VersionInfo::VersionDetails::EMPTY;    	
    }

    return result;
  }

	String VersionInfo::getTime()
	{
		static bool is_initialized = false;
		static String result;
		if ( !is_initialized )
		{
			result = String(__DATE__) + ", " + __TIME__;
			is_initialized = true;
		}
		return result;
	}

	String VersionInfo::getVersion()
	{
		static bool is_initialized = false;
		static String result;
		if ( !is_initialized )
		{
			result = OPENMS_PACKAGE_VERSION;
			result.trim();
			is_initialized = true;
		}
		return result;
	}


  VersionInfo::VersionDetails VersionInfo::getVersionStruct()
  {
    static bool is_initialized = false;
    static VersionDetails result;
    if ( !is_initialized )
    {
      result = VersionDetails::create(getVersion());
      is_initialized = true;
    }
    return result;
  }


	String VersionInfo::getRevision()
	{
		return String(OPENMS_SVN_REVISION); // defined in svn_revision.h
	}

}
