/*
   Copyright (C) 1994-2001 Digitool, Inc
;;;   This file is part of OpenMCL.  

   OpenMCL is licensed under the terms of the Lisp Lesser GNU Public
   License , known as the LLGPL and distributed with OpenMCL as the
   file "LICENSE".  The LLGPL consists of a preamble and the LGPL,
   which is distributed with OpenMCL as the file "LGPL".  Where these
   conflict, the preamble takes precedence.  

   OpenMCL is referenced in the preamble as the "LIBRARY."

   The LLGPL is also available online at
   http://opensource.franz.com/preamble.html
*/

#include <vxWorks.h>
#include <signal.h>
#include <excLib.h>
#include <intLib.h>
#include <arch/ppc/excPpcLib.h>
#include <arch/ppc/esfPpc.h>
#include <arch/ppc/dbgPpcLib.h>
#include <arch/ppc/mmuPpcLib.h>
#include <arch/ppc/vxPpcLib.h>
#include <arch/ppc/archPpc.h>
#include "sigcontext.h"

#define PROT_READ 0x1
#define PROT_WRITE 0x2

FUNCPTR signal_handlers[_NSIGS];


FUNCPTR
vx_signal(int signum, FUNCPTR new)
{
  FUNCPTR old = signal_handlers[signum];
  signal_handlers[signum] = new;
  return old;
}

void
raise_signal(ESFPPC *esf, int signum, FUNCPTR handler)
{
  struct sigcontext sc;
  int regbuf[48+(33*2)];
  struct pt_regs *regs = (struct pt_regs *)regbuf;
  int i;
  UINT32 entry_msr = vxMsrGet();

  vxMsrSet(entry_msr | _PPC_MSR_EE |_PPC_MSR_FP);
  for (i = 0 ; i < 32; i++) {
    regs->gpr[i] = esf->regSet.gpr[i];
  }
  regs->nip = esf->regSet.pc;
  regs->msr = esf->regSet.msr;
  regs->ctr = esf->regSet.ctr;
  regs->link = esf->regSet.lr;
  regs->xer = esf->regSet.xer;
  regs->ccr = esf->regSet.cr;
  regs->trap = esf->vecOffset;
  regs->dar = esf->dar;
  regs->dsisr = esf->dsisr;
  save_fp_context(&regbuf[PT_FPR0]);
  sc.regs = regs;
  sc.oldmask = 0;
  sc.handler = (unsigned long) handler;
  sc.signal = signum;
  handler(signum, &sc);
  restore_fp_context(&regbuf[PT_FPR0]);
  esf->regSet.pc = regs->nip;
  esf->regSet.msr = regs->msr;
  esf->regSet.ctr = regs->ctr;
  esf->regSet.lr = regs->link;
  esf->regSet.xer = regs->xer;
  esf->regSet.cr = regs->ccr;
  for (i = 0 ; i < 32; i++) {
    esf->regSet.gpr[i] = regs->gpr[i];
  }
  vxMsrSet(entry_msr);
}

int
maybe_raise_signal(ESFPPC *esf, int signum)
{
  FUNCPTR handler = signal_handlers[signum];
  
  if (handler) {
    raise_signal(esf, signum, handler);
    return 1;
  }
  return 0;
}


FUNCPTR old_prog_handler;
FUNCPTR old_data_handler;

/*
  Right now, the only cases of "program exceptions" that PPCCL
  cares about are conditional traps and UUOs (certain illegal
  instructions.)  It might someday want to enable FP exceptions,
  so that it doesn't have to poll after each FP operation.
  
  A breakpoint opcode is a particular "unconditional" conditional
  trap (all PPC trap instructions are conditional, but some - like
  "twle r0,r0" - are less conditional than others.)  We have to pick
  off the breakpoint case here and feed it to the old handler, which
  knows how to invoke a debugger.
*/

void
new_prog_handler(ESFPPC *esf)
{
  int msr = (int) esf->regSet.msr, handled = 0;

  if (msr & (0x80000000 >> 12) || /* if an illegal instruction (e.g., UUO) */
      ((msr & (0x80000000 >> 14)) && /* or a trap other than a breakpoint */
       (*((int *) (esf->regSet.pc)) != DBG_BREAK_INST))) {
    handled = maybe_raise_signal(esf, SIGILL);
  }
  if (! handled) {
    old_prog_handler(esf);
  }
}

/*
  Any kind of data access exception can be handled (or
  at least reported better) by a higher-level mechanism. */

int
new_data_handler(ESFPPC *esf)
{
  return maybe_raise_signal(esf, SIGSEGV);
}

/*
  There should probably be a task switch hook that juggles
  the (global) exception handlers.  Of course, task switch
  hooks are global themselves.
*/

void
enable_new_handlers()
{
  old_prog_handler = excVecGet((FUNCPTR *) _EXC_OFF_PROG);
  old_data_handler = excVecGet((FUNCPTR *) _EXC_OFF_DATA);
  excVecSet((FUNCPTR *) _EXC_OFF_PROG, (FUNCPTR)new_prog_handler);
  excVecSet((FUNCPTR *) _EXC_OFF_DATA, (FUNCPTR)new_data_handler);
}

void
disable_new_handlers()
{
  excVecSet((FUNCPTR *) _EXC_OFF_PROG, old_prog_handler); 
  excVecSet((FUNCPTR *) _EXC_OFF_DATA, old_data_handler);
}

void
init_vxlow()
{
  int i;
  
  for (i = 0; i < _NSIGS; i++) {
    signal_handlers[i] = NULL;
  }
  enable_new_handlers();
}

void
vxlow_cleanup()
{
  disable_new_handlers();
}

