/*
 * Copyright (c) 1998 Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *       This product includes software developed by the Computer Systems
 *       Engineering Group at Lawrence Berkeley Laboratory.
 * 4. Neither the name of the University nor of the Laboratory may be used
 *    to endorse or promote products derived from this software without
 *    specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * ---------------------------
 *
 * Filename: session-srmv2.cc
 *   -- Author: Suchitra Raman <suchi@cs.berkeley.edu>
 *
 * @(#) $Header: /usr/mash/src/repository/srmv2/srmv2/session-srmv2.cc,v 1.37 2001/11/21 17:58:22 lim Exp $
 * 
 */


#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <math.h>
#include <limits.h>
#include <fcntl.h>

#include <sys/time.h>


#include "srmv2/supp-srmv2.h"
#include "srmv2/ns-srmv2.h"
#include "session-srmv2.h"

//#define SRM_LOG_ALL_PACKETS

SRMv2_Session::SRMv2_Session(u_int32_t addr, u_int32_t sport, u_int32_t rport,
			     u_int32_t ttl, int bps, int *status=0)
	: SRMv2_Transmitter(bps), SRMv2_SessionHandler(),
	  badproto_(0),
#ifndef NDEBUG
	  dropProb_(0.20),
#endif
	  c1_(1.0), c2_(1.0),
	  logfile_(NULL),
	  uport_(0), representative_lsrc_(NULL),
	  manager_(NULL), pool_(NULL),
	  srcs_(NULL), delayinfo_(NULL),
	  lport_(0), laddr_(0),
	  delay_until_full_pkt_(0)
{
#ifndef NDEBUG
	// seed the random number generator for simulated packet drops
	srandom(time(NULL));
#endif
	
	srcs_      = new HashTable;
	delayinfo_ = new HashTable;
	SrmInitHashTable(srcs_, 4);
	SrmInitHashTable(delayinfo_, HASH_ONE_WORD_KEYS);

	pool_      = new SRMv2_BufferPool;
	manager_   = new SRMv2_Manager;
	manager_->session(this);
	pool_->sttl(ttl);

	int ret = reset(addr, sport, rport, ttl);
	if (status) *status = ret;
	srm_callbacks tmp = { NULL, NULL, NULL, NULL };
	callbacks_ = tmp;
	SRMv2_Transmitter::init_hfsc(bps/1000);
}


SRMv2_Session::~SRMv2_Session()
{
	/* Send out buffered packet. */
	send_delayed_pkt();

	if (manager_)   delete manager_;
	if (srcs_)      delete srcs_;
	if (delayinfo_) delete delayinfo_;
	if (pool_)      delete pool_;
}


int
SRMv2_Session::reset(u_int32_t addr, u_int32_t sport, u_int32_t rport,
		     u_int32_t ttl)
{
	SRMv2_IPNetwork *net;
	int status;
	
	net = (SRMv2_IPNetwork*)uh_.net();
	uh_.net(0);
	if (net) delete net;

	net = (SRMv2_IPNetwork*)dh_.net();
	dh_.net(0);
	if (net) delete net;

	dh_.manager(this);
 	uh_.manager(this);

	/* Open an IP multicast channel. */
	if (addr != 0 && sport != 0 && rport != 0 && ttl != 0) {
		net = new SRMv2_IPNetwork();
		status = net->open(addr, htons(sport), htons(rport), ttl);
		if (status < 0) {
			srm_trace(srmSRM, ("Error opening multicast socket."));
			delete net;
			return (-1);
		}
		dh_.net(net);
		lport_ = net->ssock_port(); /* send port */
		laddr_ = LookupLocalAddr();
	}

#ifdef SRMv2_LOC_REC
	/* Open a unicast channel, useful in 2-step recovery. */
	net = new SRMv2_IPNetwork;
	uport_ = (u_short) rport + 2; /* Unicast port. */
	status = net->open(htons(uport_));
	if (status < 0) {
		srm_trace(srmSRM, ("Error opening unicast socket."));
		delete net;
		return (-1);
	} 
	uh_.net(net);
#endif
	return 0;
}


SRMv2_Source *
SRMv2_Session::create_localsrc(const char *srcid)
{
	/* 
	 * Create a source whose ID is derived from the local IP address. 
	 * This is a common case optimization. 
	 */

	SRMv2_Source *src = 0;
	int created;
	src = new SRMv2_Source(srcid, 1);
	HashEntry *eptr = CreateHashEntry(srcs_, (char *)src->source_id(), 
					  &created);
	if (!created) {
		// a source with that id already exists
		delete src;
		return NULL;
	}
	
	SetHashValue(eptr, src);
	/* 
	 * Since the IPv4 address may change during the 
	 * lifetime of a session, this information must 
	 * be updated each time. 
	 */
	src->ipaddr(LookupLocalAddr());
	src->session(this);
	if (!representative_lsrc_) representative_lsrc_ = src;
	return src;	
}


SRMv2_Source *
SRMv2_Session::source(int *srcID, u_int32_t addr, int *created)
{
	HashEntry *eptr = FindHashEntry(srcs_, (char *)srcID);
	SRMv2_Source *src;
	
	if (!eptr) {
		src = new SRMv2_Source(srcID, 0);
		eptr = CreateHashEntry(srcs_, (char *)srcID, created);
		SetHashValue(eptr, src);
		src->ipaddr(addr);
		src->session(this);

		/* notify the application that we received a new source
		 */
		srm_source_update_proc proc;
		proc = callbacks()->source_update_proc;
		if (proc) {
			(*proc)((srm_source_t)src, NULL, 0);
		}
	} else {
		src = (SRMv2_Source *) GetHashValue(eptr);
		/* 
		 * Since the IPv4 address may change during the 
		 * lifetime of a session, this information must 
		 * be updated each time. 
		 */
		src->ipaddr(addr);
	}
	return src;
}


SRMv2_Source *
SRMv2_Session::source(SRMv2_Source *src)
{
	int *srcID = src->source_id();
	int created = 0;

	HashEntry *eptr = FindHashEntry(srcs_, (char *)srcID);
	
	if (!eptr) {
		eptr = CreateHashEntry(srcs_, (char *)srcID, &created);
		SetHashValue(eptr, src);
	}
	return (SRMv2_Source *) GetHashValue(eptr);
}


void
SRMv2_Session::send_delayed_pkt()
{
	if (!delayed_pkt_.pb_) return;
	srm_trace(srmSRM, ("S DELAYED"));
	if (delayed_pkt_.last_aduhdr_) {
		// set the L flag in the last ADU header
		delayed_pkt_.last_aduhdr_->flags |= (1 << 6);
	}
	delayed_pkt_.pb_->dp   = delayed_pkt_.pb_->data;
	SRMv2_Transmitter::recv(delayed_pkt_.pb_);
	delayed_pkt_.pb_ = NULL;
}


/* 
 * Protocol currently supports only 64 KB long ADUs.
 * For ADUs larger than 64 KB< we need a library that
 * handles fragmentation into pieces <= 64 KB 
 *
 * NTP time should be given by the application! 
 */
unsigned int
SRMv2_Session::send_adu(SRMv2_Source *src, u_char *data, u_int32_t seqno,
			u_int32_t sbytes, u_int32_t ebytes, u_int32_t adulbyte,
			u_int32_t cid, u_char type, u_int32_t addr, u_char f,
			const srm_adu_info *info)
{
	u_int32_t mss;
	u_char more = 0;
	u_char frag = 0;
	u_int32_t sentbytes = sbytes;
	u_short incr;
	SRMv2_pktbuf *pb;
	u_int8_t atype = (info ? info->type : 0);
	ntp64 ntp = { 0, 0 };
	if (info) ntp = info->timestamp;
	u_char *dp = data;
	dp += sbytes;

	/*
	 * aggressively dirty the namespace signature
	 */

	src->name_space()->dirty_signature();

	/*
	 * srm_send will set seqno to SRMv2_ALLOC; we should allocate a
	 * sequence number in that case
	 */
	if (seqno==SRMv2_ALLOC) {
		/* make sure this is actually a local source */
		if (!src->islocal()) return SRMv2_ALLOC;
		seqno = src->name_space()->next_seqno(cid);
		if (seqno==SRMv2_ALLOC)
			return SRMv2_ALLOC;
	}
	
	if (ebytes - sbytes + 1 <= 0) return SRMv2_ALLOC;
	if (!pool_) {
		printf("send failed. No buffer pool.\n");
		abort();
	}


	/*
	 * Check if we have a delayed packet; if the delayed packet has
	 * a different type or addr field, or if we can't fit the new ADU
	 * in the delayed packet, send out the delayed packet first
	 */
	
	if (delay_until_full_pkt_ && delayed_pkt_.pb_) {
		mss = NET_MTU - delayed_pkt_.pb_->len;
		u_int32_t adusize = sizeof(srmv2_aduhdr);
		if (delayed_pkt_.cid_ != cid || delayed_pkt_.src_ != src)
			adusize += sizeof(srmv2_conhdr);
		if (f) adusize += sizeof(srmv2_fraghdr);
		adusize += ebytes - sbytes + 1;

		if (adusize > mss || delayed_pkt_.type_!=type ||
		    delayed_pkt_.addr_!=addr) {
			// can't fit the packet, or this is a different packet
			// send out the previously buffered packet
			send_delayed_pkt();
		}		
	}
	
	/* 
	 * Check for fragmentation 
	 * Fragement header consumes 4B 
	 */
	mss = pool_->ip_mtu();
	if (ebytes - sbytes + 1 > mss) {
		mss -= sizeof(srmv2_fraghdr);
		frag = 1;
	}
	frag |= f;
	while (sentbytes <= ebytes) {
		if (delay_until_full_pkt_) {
			if (!delayed_pkt_.pb_) {
				pb = pool_->alloc(SRMv2_DATA_TCID, type, addr);
				delayed_pkt_.pb_   = pb;
				delayed_pkt_.type_ = type;
				delayed_pkt_.addr_ = addr;
				delayed_pkt_.src_  = src;
				delayed_pkt_.cid_  = cid;
				delayed_pkt_.last_aduhdr_ = NULL;
				
				pb->port = uport_;
				pool_->build_conhdr(pb, src->source_id(), cid);
			} else {
				pb = delayed_pkt_.pb_;
				if (delayed_pkt_.cid_ != cid ||
				    delayed_pkt_.src_ != src) {
					/* set the L bit in the aduhdr of the
					 * previous ADU
					 */
					if (delayed_pkt_.last_aduhdr_) {
						delayed_pkt_.last_aduhdr_->
							flags |= (1 << 6);
					}
					
					pool_->build_conhdr(pb,
							    src->source_id(),
							    cid);
					delayed_pkt_.src_ = src;
					delayed_pkt_.cid_ = cid;
				}
			}
		} else {
			pb = pool_->alloc(SRMv2_DATA_TCID, type, addr);
			pb->port = uport_;
			pool_->build_conhdr(pb, src->source_id(), cid);
		}
		incr = MIN(ebytes - sentbytes + 1, mss);
		if (delay_until_full_pkt_) {
			delayed_pkt_.last_aduhdr_ = (srmv2_aduhdr *)pb->dp;
			pool_->build_aduhdr(pb, atype, incr, seqno, ntp, frag);
			delayed_pkt_.last_aduhdr_->flags &=
				(unsigned char)(~(1 << 6));
		} else {
			pool_->build_aduhdr(pb, atype, incr, seqno, ntp, frag);
		}
		
		if (frag) {
			more = ((sentbytes + incr - 1) < ebytes) ? 1 : 0;
			more |= (adulbyte != ebytes);
			pool_->build_fraghdr(pb, more, sentbytes);
		}
		memcpy(pb->dp, dp, incr);
		pb->dp  += incr;
		pb->len += incr;
		// make the length be a multiple of 4
		if (incr % 4) {
			pb->dp  += (4 - incr % 4);
			pb->len += (4 - incr % 4);
		}
		srm_trace(srmSRM, ("S %d -- %d", sentbytes, sentbytes + incr));
		if (delay_until_full_pkt_) {
			// check if we have no more room for more packets
			if (NET_MTU - pb->len <= (int) sizeof(srmv2_aduhdr)) {
				send_delayed_pkt();
			}
		} else {
			srm_trace(srmSRM, ("S NOT DELAYED"));
			pb->dp   = pb->data;
			SRMv2_Transmitter::recv(pb);
		}
		sentbytes += incr;
		dp += incr;
	}
	if (src->islocal() && type==SRMv2_DATA) {
		// this is a local source and it is not a retransmission
		src->name_space()->update_edge(cid, seqno, ebytes);
		src->name_space()->update_prio(cid);
		NS_Node *node = src->name_space()->getnode(cid);
		node->set_lastseen(seqno);
	}

	return seqno;
}

void 
SRMv2_Session::send(SRMv2_pktbuf *pb)
{
	pb->dp = pb->data;
	SRMv2_Transmitter::recv(pb);
}


#ifdef SRM_LOG_ALL_PACKETS
void
srmwrite(unsigned char *buf, int len, FILE *fp)
{
	for (int i=0; i<len; i+=20) {
		fprintf(fp, "%05d ", i);
		for (int j=i; j<i+20 && j<len; j++, buf++) {
			fprintf(fp, "%02x ", *buf);
		}
		fprintf(fp, "\n");
	}
	
	fprintf(fp, "=====================================================\n");
	fflush(fp);
}
#endif /* SRM_LOG_ALL_PACKETS */


void
SRMv2_Session::transmit(SRMv2_pktbuf* pb)
{
#ifndef NDEBUG
	if (dropProb_ > 0.0 && ((double)random())/INT_MAX < dropProb_) {
		srm_trace(srmSRM, ("dropped...\n"));
		return;
	}
#endif
	
	SRMv2_Network* n = 0;

#ifdef SRM_LOG_ALL_PACKETS	
	static FILE *fp=NULL;
	if (!fp) {
		char name[80];
		sprintf(name, "/tmp/out%d.srm", getpid());
		fp = fopen(name, "w");
	}
	srmwrite(pb->dp, pb->len, fp);
#endif /* SRM_LOG_ALL_PACKETS */
	
#ifdef SRMv2_LOC_REC
	if (pb->mode == SRMv2_UNICAST) {
		n = uh_.net();
		if (n != 0) {
			n->sendto(pb);
		}
		return;
	}

#endif
	//	printf("In transmit...\n");
	n = dh_.net();
	if (n != 0) {
		u_short old = n->ttl();
		n->ttl(pb->ttl);
		n->send(pb);
		n->ttl(old);
	}
}

void 
SRMv2_Session::recv(SRMv2_CtrlHandler* /*ch*/)
{
	printf("SRMv2_Session::recv(SRMv2_CtrlHandler *) should never be "
	       "called\n");
	abort();
}


void 
SRMv2_Session::recv(SRMv2_DataHandler* dh)
{
	SRMv2_pktbuf* pb;
	u_int32_t addr, port;

#ifdef SRM_PROCESS_ALL_MESSAGES
	while(1) {
#endif
		pb = pool_->alloc(0);
		memset(pb->data, 0, PKTBUF_SIZE);
		int cc = dh->recv(pb->data, sizeof(pb->data), addr, port);
		if (cc <= 0) {
		        pb->release();
			return;
		}
		/* Discard local loopback packets.*/
		if (addr==laddr_ && port==lport_) {
			pb->release();
		} else {
#ifdef SRM_LOG_ALL_PACKETS
			static FILE *fp=NULL;
			if (!fp) {
				char name[80];
				sprintf(name, "/tmp/in%d.srm", getpid());
				fp = fopen(name, "w");
			}
			srmwrite(pb->data, cc, fp);
#endif /* SRM_LOG_ALL_PACKETS */
			pb->len = cc;
			parse_header(pb);
		}

#ifdef SRM_PROCESS_ALL_MESSAGES
	}
#endif

}


void
#ifndef NDEBUG
SRMv2_Session::dump_cmnhdr(SRMv2_pktbuf *pb)
#else
SRMv2_Session::dump_cmnhdr(SRMv2_pktbuf * /* pb */)
#endif
{

#ifndef NDEBUG
        /* Parse common header */
	srmv2_cmnhdr *sh  = (srmv2_cmnhdr *)pb->data;
	u_char type       = sh->type;
	u_char version    = sh->flags >> 6;
	u_char uflag      = (sh->flags >> 4) & 1;
	u_int32_t ipaddr  = htonl(sh->ipaddr);
#endif
	srm_trace(srmSRM|srmVerbose, ("V %d, T %d, U %d, addr %s", version,
				      type, uflag, intoa(ipaddr)));
}

void
SRMv2_Session::parse_header(SRMv2_pktbuf *pb)
{

	SRMv2_pktbuf *cpy=0;
        /* Parse common header */
	srmv2_cmnhdr *sh  = (srmv2_cmnhdr *)pb->data;
	u_char type       = sh->type;
	u_char version    = sh->flags >> 6;
#ifdef SRMv2_LOC_REC
	u_char uflag      = (sh->flags >> 4) & 1;
#endif
	u_int32_t ipaddr  = htonl(sh->ipaddr);

	pb->dp = pb->data + sizeof(srmv2_cmnhdr);	
	pb->len -= sizeof(srmv2_cmnhdr);
	
	if (version != SRMv2_VERSION) {
		srm_trace(srmSRM, ("%dB with bad version %d", 
				   pb->len + sizeof(srmv2_cmnhdr), version));
		badproto_++;
		pb->release();
		return;
	}

	switch (type) {
	case SRMv2_REXMIT: 
	case SRMv2_DATA:
		/*
		 * Duplicate packet before we modify its contents
		 * before unicast replies are multicast "locally".
		 */
#ifdef SRMv2_LOC_REC
		if (uflag == SRMv2_UNICAST) {
			cpy = (SRMv2_pktbuf *)pb->copy(SRMv2_REXMIT);
			cpy->dp  = cpy->data;
			cpy->len = pb->len + sizeof(srmv2_cmnhdr);
		}
#endif
		handle_data(pb, cpy, ipaddr);
		break;
	default:
		handle_ctrl(pb, type, ipaddr);
		break;
	}
	pb->release();
}

void
SRMv2_Session::handle_data(SRMv2_pktbuf *pb, SRMv2_pktbuf *cp,u_int32_t ipaddr)
{
	srmv2_aduhdr *ahdr = 0;
	srmv2_conhdr *chdr = 0;
	SRMv2_Source *src=0;
	srm_adu_info adu_info={0,{0,0}};
	int i, created;
	u_int32_t cid=0;
	u_int32_t seqno, offset=0;
	int len;
	int more=0;
	u_short ttl;
	int srcID[4];
	

	len = pb->len;
	while (len > 0) {
		if (!chdr || (ahdr && (ahdr->flags & (0x1 << 6)))) {
			/* we have a con hdr next. consume it. */
			if (len < (int) sizeof(srmv2_conhdr)) break;
			chdr = (srmv2_conhdr *) pb->dp;
			for (i=0; i<4; i++) 
				srcID[i] = ntohl(chdr->srcid[i]);
			src = source(&srcID[0], ipaddr, &created);
			cid = ntohl(chdr->cid);
			pb->dp  += sizeof(srmv2_conhdr);
			len -= sizeof(srmv2_conhdr);
		}
		
		if (len < (int)sizeof(srmv2_aduhdr)) break;
		ahdr  = (srmv2_aduhdr *) pb->dp;
		pb->dp  += sizeof(srmv2_aduhdr);
		len -= sizeof(srmv2_aduhdr);
		if (ahdr->flags >> 7) {
			if (len < (int)sizeof(srmv2_fraghdr)) break;
			offset = ntohl(*((srmv2_fraghdr *)pb->dp));
			more   = offset >> 31;
			offset &= 0x7fffffff;
			pb->dp  += sizeof(srmv2_fraghdr);
			len -= sizeof(srmv2_fraghdr);
		}
		pb->offset = offset;
		pb->len = ntohs(ahdr->alen);
		seqno = ntohl(ahdr->seqno);
		adu_info.type = ahdr->atype;
		adu_info.timestamp.lower = ntohl(ahdr->ts[0]);
		adu_info.timestamp.upper = ntohl(ahdr->ts[1]);
		if (len < pb->len) {
			srm_trace(srmSRM, ("Runt ADU\n"));
			break;
		}
		manager_->handle_data(src, pb, cid, seqno, more, &ttl,
				      &adu_info);
		/*
		 * If padded to word boundary, consume extra bytes.
		 */
		if (pb->len % 4) pb->len += (4 - pb->len % 4);
		pb->dp += pb->len;
		len    -= pb->len;
	}

	/*
	 * In 2-step rec., multicast a copy with the
	 * request's ttl.
	 */
	if (cp != 0) {
		cp->ttl = ttl;
		SRMv2_Transmitter::recv(cp);
	}
}

void
SRMv2_Session::handle_ctrl(SRMv2_pktbuf *pb, u_char type, u_int32_t sndr)
{
	srmv2_conhdr *chdr = 0;
	int srcID[4];
	int i, cid, created;
	SRMv2_Source *src;

	if (pb->len < (int) sizeof(srmv2_conhdr)) return;
	chdr = (srmv2_conhdr *) pb->dp;
	for (i=0; i<4; i++) 
		srcID[i] = ntohl(chdr->srcid[i]);
	src = source(&srcID[0], sndr, &created);
	cid = ntohl(chdr->cid);
	pb->dp  += sizeof(srmv2_conhdr);
	pb->len -= sizeof(srmv2_conhdr);

	switch(type) {

	case SRMv2_RREQ:
		handle_rreq(pb, src, cid, sndr);
		break;

	case SRMv2_DELAYS:
		handle_dvec(pb, sndr);
		break;

	case SRMv2_NSREQUEST:
		break;

	case SRMv2_ANNOUNCEMENT:
		handle_announce(pb, src);
		break;
	}
}


void
SRMv2_Session::handle_rreq(SRMv2_pktbuf *pb, SRMv2_Source *src,
			   u_int32_t cid, u_int32_t addr)
{
	u_int32_t ss, es, sbytes, ebytes;
	srmv2_rreqhdr *rhdr;

	if (manager_) {
		rhdr = (srmv2_rreqhdr *) pb->dp;
		ss = ntohl(rhdr->ss);
		es = ntohl(rhdr->es);
		sbytes = ntohl(rhdr->sbytes);
		ebytes = ntohl(rhdr->ebytes);
		pb->dp += sizeof(srmv2_rreqhdr);
		pb->len += sizeof(srmv2_rreqhdr);
		manager_->handle_rreq(src, ss, sbytes, es, 
				      ebytes, cid, addr);
	}
}

void
SRMv2_Session::handle_dvec(SRMv2_pktbuf *pb, u_int32_t addr)
{
	int n;
	srmv2_delayhdr *dvec=0;
	u_int32_t tsend;
		
	tsend = ntohl(*(u_int32_t*)(pb->dp));
	pb->dp  += sizeof(u_int32_t);
	pb->len -= sizeof(u_int32_t);
	n = pb->len/sizeof(srmv2_delayhdr);
	dvec = (srmv2_delayhdr *) pb->dp;
	update_delay(addr, tsend, n, dvec);
}

void
SRMv2_Session::handle_announce(SRMv2_pktbuf *pb, SRMv2_Source *src)
{
	/* extract the src's app info */
	int skip = src->update_appinfo((const u_char*)pb->dp);
	pb->dp  += skip;
	pb->len -= skip;

	src->update_namespace((naddr_plus_name*)pb->dp, pb->len);
}

void 
SRMv2_Session::send_rreq(SRMv2_Request *request, unsigned char type)
{
	if (!pool_) {
		printf("send failed. No buffer pool.\n");
		abort();
	}
	u_int16_t t       = request->ttl_;
	u_int32_t cid     = request->cid_;
	u_int32_t ss      = request->left.adu;
	u_int32_t sbytes  = request->left.byte;
	u_int32_t es      = request->right.adu;
	u_int32_t ebytes  = request->right.byte;
	srm_trace(srmSRM, ("send Q %d: (%x.%d -- %x.%d) [ttl=%x]\n", cid, ss,
			   sbytes, es, ebytes, t));

	SRMv2_Source *src = request->source();
	SRMv2_pktbuf* pb = pool_->alloc(SRMv2_NACK_TCID, type);
	pb->ttl    = t;
	pool_->build_conhdr(pb, src->source_id(), cid);
	pool_->build_rreqhdr(pb, ss, sbytes, es, ebytes);
	pb->dp  = pb->data;
	SRMv2_Transmitter::recv(pb);
	return;
}


void 
SRMv2_Session::send_announcement(SRMv2_Source *src, int include_srcinfo,
				 naddr* comps, int count)
{
	if (!pool_) {
		printf("send failed. No buffer pool.\n");
		abort();
	}
	SRMv2_pktbuf* pb = pool_->alloc(SRMv2_ANNOUNCE_TCID, SRMv2_ANNOUNCEMENT);
	pool_->build_conhdr(pb, src->source_id(), SRMv2_ERROR);
	pool_->build_srcinfohdr(pb, src, include_srcinfo);
	pool_->build_nshdr(pb, src, comps, count);
	pb->dp  = pb->data;

#ifdef NDEBUG
	srm_trace_start(srmSRM);
	for (int i=0; i<count; i++) 
		srm_trace_out(srmSRM, ("%d.%d ",comps[i].cid,comps[i].offset));
	srm_trace_end(srmSRM);
#endif
	
	src->name_space()->display(0);
	SRMv2_Transmitter::recv(pb);
	return;
}

/*
 * Multicast a delay information vector. 
 */
void 
SRMv2_Session::send_delayinfo()
{
	if (!representative_lsrc_) return;
	
	if (!pool_) {
		printf("send failed. No buffer pool.\n");
		abort();
	}
	SRMv2_pktbuf* pb = pool_->alloc(SRMv2_ANNOUNCE_TCID, SRMv2_DELAYS);
	pool_->build_conhdr(pb, representative_lsrc_->source_id(), 0);
	fill_delays(pb);
	pb->dp  = pb->data;
	SRMv2_Transmitter::recv(pb);
	return;
}

void
SRMv2_Session::announce(SRMv2_CtrlHandler* ch)
{
	// announce may get called even if we don't have a network
	// in that case, just return
	if (! dh_.net()) return;
	
	SRMv2_Source *src = SRMv2_Source::source(ch);

	if (src) {
		src->announce();
		if (src==representative_lsrc_) {
			/* Send another packet with delay info */
			send_delayinfo();
		}
	}
}


float
SRMv2_Session::delay(u_int32_t address)
{
	float delay=0.0;

	HashEntry *eptr = FindHashEntry(delayinfo_, (char*) address);
	if (eptr) {
		srmv2_delays *record = (srmv2_delays *) GetHashValue(eptr);
		delay = ntp2msec((int) (0.5 * record->rtt_est));
	} 
	return MAX(MIN_BACKOFF_DELAY, delay);
}

/* 
 * Compute backoff after reading the estimated delay.  We should be
 * using more conservative constants d1_ and d2_ for repairs,
 * but for now, use the same constants regardless of packet type. 
 */
double
SRMv2_Session::backoff(SRMv2_Source *src, u_int32_t level)
{
	double d = (double) delay(src->ipaddr());
	double r = (double)random()/double(INT_MAX);
	double p = pow(2, level);
	double b = p * (c1_ + c2_ * r) * d;
	return b;
}

/*
 * SRMv2_DELAYS :
 *                      (32)
 * +--------+---------+---------------------------+
 * |              tsend (NTP timestamp)           | 
 * +--------+---------+---------------------------+
 * |                 IPv4 addr           (0)      | 
 * +--------+---------+---------------------------+
 * |               one-way difference    (0)      | 
 * +--------+---------+---------------------------+
 * |                 IPv4 addr           (1)      | 
 * +--------+---------+---------------------------+
 * |               one-way difference    (1)      | 
 * +--------+---------+---------------------------+
                      .....
 * +--------+---------+---------------------------+
 * |                 IPv4 addr           (n-1)    | 
 * +--------+---------+---------------------------+
 * |               one-way difference    (n-1)    | 
 * +--------+---------+---------------------------+
 */

void
SRMv2_Session::fill_delays(SRMv2_pktbuf *pb)
{

	u_int32_t tsend = ntptime();
	*(u_int32_t *) pb->dp = htonl(tsend);
	pb->dp  += 4;	
	pb->len += 4;
	srmv2_delayhdr* di = (srmv2_delayhdr *)pb->dp;
	
	HashSearch hs;
	u_int32_t addr;
	int j = 0;
	srmv2_delays *pDI;

	for (HashEntry* pEntry = FirstHashEntry(delayinfo_, &hs);
	     pEntry; pEntry = NextHashEntry(&hs)) 
	{
		addr = (u_int32_t)GetHashKey(delayinfo_, pEntry);
		pDI  = (srmv2_delays *)GetHashValue(pEntry);

		/* Ignore estimates that don't provide any information. */
		if (pDI->ow_diff == 0)
			continue;      
		di[j].ip_addr = htonl(addr);
		di[j].ow_diff = htonl(pDI->ow_diff);
		j++;
	}
	pb->dp  += j * sizeof(srmv2_delayhdr);
	pb->len += j * sizeof(srmv2_delayhdr);
}


/* 
 * Update our delay with respect to 'ipaddr'. 
 * tsend is in ntp format, IPaddr is a 32-bit IPv4 address. 
 *
 * t1
 *   \
 *   t2
 *    |
 *   t3
 *   /
 * t4
 *
 */

void
SRMv2_Session::update_delay(u_int32_t addr, u_int32_t tsend, 
			    u_int32_t n, srmv2_delayhdr* dvec)
{
	u_int32_t local_ipaddr = LookupLocalAddr();
	if (local_ipaddr == addr) return; /* Discard our own packet. */
	u_int32_t now = ntptime();
	/* Update the one-way delay estimate (owd_est) */

	int created; 
	HashEntry* pEntry = CreateHashEntry(delayinfo_,
					    (char*) addr,
					    &created);
	srmv2_delays* pDI = 0;
	if (created) {
		pDI = new srmv2_delays;
		pDI->rtt_est = 0;
		pDI->ow_diff = 0;
		SetHashValue(pEntry, pDI);
	} else {
		pDI = (srmv2_delays *) GetHashValue(pEntry);
	}
	
	/* 
	 * For the incoming announcement, t2 = now, t1 = tsend, so 
	 * ow_diff = now - tsend. 
	 */
	pDI->ow_diff = now - tsend;
	/*
	 * t4 = now
	 * td = t1 + delta = t1 + t3 - t2 = t3 - (t2 - t1)
	 * rtt_est = (t4 - t1 - delta) = (now - td)
	 */
	for (u_int32_t i=0; i < n; i++) {
		if (ntohl(dvec[i].ip_addr) == local_ipaddr) {
			u_int32_t new_sample = (now - tsend +
						ntohl(dvec[i].ow_diff));
			if (new_sample > 0) {
				if (pDI->rtt_est == 0.0) { 
					pDI->rtt_est = new_sample;
				} else {
					pDI->rtt_est =
						(u_int32_t)(0.3 * pDI->rtt_est 
							    + 0.7 *new_sample);
				}
				srm_trace(srmSRM, ("RTT %s %f\n", intoa(addr), 
						   ntp2msec(pDI->rtt_est)));
			}
			break;
		}
	}
}


int
SRMv2_Session::srm_should_recover(SRMv2_Source *source, unsigned int cid,
				  unsigned int sseq, unsigned int eseq)
{
	srm_should_recover_proc proc = callbacks()->should_recover_proc;
	if (proc)
		return (*proc)((srm_source_t)source, cid, sseq, eseq);
	else
		return 0;
}
