/*
 * video-test.cc --
 *
 *      Creates a test device for transmitting video. It is useful in the 
 *      case you don't have a video capture card. 
 *      Currently it support two options:
 *        1. automatically-generated blue streams (aka "blue-passion"), and 
 *        2. streams created by repeating a .ppm file (both ascii and raw)
 *
 * Copyright (c) 1993-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef lint
static const char rcsid[] =
    "@(#) $Header: /usr/mash/src/repository/mash/mash-1/video/video-test.cc,v 1.11 2002/02/03 04:20:17 lim Exp $";
#endif

#include <stdio.h>
#include <stdlib.h>
#include <iostream.h>
#include <fstream.h>
#include <sys/stat.h>

#include "video-device.h"
#include "Tcl.h"
#include "module.h"
#include "rgb-converter.h"


// FIXME: This is definitely *not* the place to define video sizes
#define PAL	    0
#define NTSC	    1
// CIF
#define CIF_WIDTH   352
#define CIF_HEIGHT  288
// QCIF
#define QCIF_WIDTH   176
#define QCIF_HEIGHT  144
// Square-pixel CCIR-601 format, digitally-encoded NTSC resolution
#define SP601_NTSC_WIDTH  640
#define SP601_NTSC_HEIGHT 480
// Square-pixel CCIR-601 format, digitally-encoded PAL resolution
#define SP601_PAL_WIDTH   768
#define SP601_PAL_HEIGHT  576
// Square-pixel SIF format, digitally-encoded NTSC resolution
#define SPSIF_NTSC_WIDTH  320
#define SPSIF_NTSC_HEIGHT 240
// Square-pixel SIF format, digitally-encoded PAL resolution
#define SPSIF_PAL_WIDTH   384
#define SPSIF_PAL_HEIGHT  288


// this variable stores the name of the last file used as test. It cannot 
//	be part of the VideoCaptureTest object because the object is destroyed 
//	and recreated every time the encoder is changed
static char old_filename[1024] = "";

// Color subsampling options.
#define CF_422 0
#define CF_411 1
#define CF_CIF 2

// Test options. 
//	1. "Blue passion" creates a blue image and keeps changing its color 
//		nuance so that you send something dynamic. 
//	2. "File" creates a fixed image from a PPM file you select
#define TEST_BLUEPASSION 0
#define TEST_FILE 1


//
//	VideoCaptureTest
//
// this is the C++ object that implements the virtual, video-capture device 
//	that generates test streams. As any other VideoCaptureXXX object, it 
//	has to inherit from VideoCapture. 
//
class VideoCaptureTest : public VideoCapture {
public:
	VideoCaptureTest(const char * cformat, const char *dev);
	virtual ~VideoCaptureTest();

	virtual int command(int argc, const char*const* argv);
	virtual void start();
	virtual void stop();
	virtual void grab();
protected:
	int capture();
	void format();
	void setsize();
	int set_file(const char *filename);
	int set_bluepassion();
  
	int fd_;
	int format_;
	int cformat_;
	int port_;
  
	unsigned char *tm_;

	// definition of the frame the encoder expects
	char *yuvdata_; 
	int width_;
	int height_;

	// definition of the frame obtained from the file we are using as source
	char *rgbdata_; 
	int rgbwidth_;
	int rgbheight_;

	int decimate_;
};


//
//	TestDevice
//
// A TestDevice object creates an instance of the C++ version of the 
//	capture device (VideoCaptureTest) and links it to a tcl object 
//	(VideoCapture/Test), so that calls can be made from one world to 
//	the other. As any other XXXDevice object, it has to inherit from 
//	VideoDevice. 
//
// Creating a XXXDevice object means that the device XXX has been 
//	reserved by the creator. In order to create a new VideoCaptureXXX 
//	object, a call must be done to XXXDevice::create() with the 
//	specific parameters selected (color subsampling, size, port, ...)
//
class TestDevice : public VideoDevice {
public:
	TestDevice(const char* clsname, const char* nickname,
			const char* devname, const char* attr, int free);
  
	TclObject* create(int argc, const char*const* argv) {
		if (argc != 5) {
			printf("invalid arguments passed to TestDevice\n");
			return (0);
		}
		if (strcmp(argv[4], "422") == 0) {
			return (new VideoCaptureTest("422",name_));
		} else if (strcmp(argv[4], "411") == 0) {
			return (new VideoCaptureTest("411",name_));
		} else if (strcmp(argv[4], "cif") == 0) {
			return (new VideoCaptureTest("cif",name_));
		}
		return (0);
	}
  
protected:
	const char* name_;
};

TestDevice::TestDevice(const char* clsname, const char* nickname,
		const char *devname, const char* attr, int free) 
	: VideoDevice(clsname, nickname), name_(devname)
{
	if (free)
			attributes_ = attr; 
	else
			attributes_ = "disabled";
}



//
//	TestScanner
//
// this object is used to allow the scanning of your box in order to 
//	check if a video-capture device is present. The idea is to create 
//	an object of the type XXXScanner for every video-capture device "XXX". 
//	With this, you ensure that the XXXScanner constructor is going to be 
//	called at least once during mash initialization. This constructor 
//	checks for the existence of the video-capture device it refers to 
//	and, should it find such device, it creates a new XXXDevice object. 
//
//	In our case, we want a VideoCapture/Test tcl object to exists always, 
//	so that the TestScanner constructor always creates a new TestDevice
//
class TestScanner {
public:
	TestScanner();
};

static TestScanner find_test_devices;

TestScanner::TestScanner()
{
	const char *testclassnames[] = {"VideoCapture/Test"};
	char * nick = new char[512];
	strcpy(nick, "Test");
	char * attr = new char[512];
	strcpy (attr, "format { 411 422 cif } size { small large cif } port { blue_passion file } ");
	new TestDevice (testclassnames[0], nick, "Test", attr, 1);
}


//
//	VideoCaptureTest::VideoCaptureTest
//
VideoCaptureTest::VideoCaptureTest(const char *cformat,
                                   const char * /* dev */)
{
	if (!strcmp(cformat,"422"))
		cformat_ = CF_422;
	else if (!strcmp(cformat, "411"))
		cformat_ = CF_411;
	else if (!strcmp(cformat, "cif"))
		cformat_ = CF_CIF;

	decimate_  = 2;
	port_ = TEST_BLUEPASSION; 
	yuvdata_ = NULL; 
	rgbdata_ = NULL; 
}


//
//	VideoCaptureTest::VideoCaptureTest
//
VideoCaptureTest::~VideoCaptureTest()
{
	if (rgbdata_ != NULL) {
		delete rgbdata_;
	}
	if (yuvdata_ != NULL) {
		delete yuvdata_;
	}
}


//
//	VideoCaptureTest::command
//
// This method is called from tcl to change the video-capture device 
//	settings
//
int VideoCaptureTest::command(int argc, const char*const* argv)
{
	if (argc == 3) {
		if (strcmp(argv[1], "decimate") == 0) {
			decimate_ = atoi(argv[2]);
			if (running_)
					format();
			return (TCL_OK);
		}

		if (strcmp(argv[1], "set-file") == 0) {
			if (set_file (argv[2]) == 0) {
				return (TCL_OK);
			} else {
				return (TCL_ERROR);
			}
		}

		if (strcmp(argv[1], "port") == 0) {
			if (strcmp(argv[2], "blue_passion") == 0) {
				if (set_bluepassion () == 0) {
					return (TCL_OK);
				} else {
					return (TCL_ERROR);
				}
			} else if (strcmp(argv[2], "file") == 0) {
				if (set_file ("") == 0) {
					return (TCL_OK);
				} else {
					return (TCL_ERROR);
				}
			}

			if (running_)
					format();
			return (TCL_OK);
		}

		if (strcmp(argv[1], "fps") == 0) {
			// what goes here?
		}

	} else if (argc == 2) {

		if (strcmp(argv[1], "set-bluepassion") == 0) {
			if (set_bluepassion () == 0) {
				return (TCL_OK);
			} else {
				return (TCL_ERROR);
			}
		}

	}
  
	return (VideoCapture::command(argc, argv));
}

//
//	VideoCaptureTest::start
//
void VideoCaptureTest::start()
{
	format();
	VideoCapture::start();
}

//
//	VideoCaptureTest::stop
//
void VideoCaptureTest::stop()
{   
	VideoCapture::stop();
}


//
//	VideoCaptureTest::capture
//
// This method is called every time a frame is needed. The timing of the 
//	frame capturing is decided by the user by selecting the fps he wants. 
//	Therefore every time capture is called the video-capture device has 
//	to provide a frame.
//
int VideoCaptureTest::capture()
{
	// frame_ is an array of Y, followed by an array of U and an array of V
	// 	a. In 4:2:2 video, Y is of size width*height, U is (width*height/2), and
	//	V is (width*height/2)
	// 	b. In 4:1:1 video, Y is of size width*height, U is (width*height/4), and
	//	V is (width*height/4)
	// 	c. 4:2:0 video is the same length than 4:1:1, so 4:1:1 sizes are used
  
	switch (port_) {
		case TEST_FILE:
			if (cformat_ == CF_422) {
				memcpy(frame_, (u_char *)yuvdata_, 2*width_*height_);
			} else if (cformat_ == CF_411) {
				memcpy(frame_, (u_char *)yuvdata_, (int)(1.5*width_*height_));
			} else { // CF_CIF
				memcpy(frame_, (u_char *)yuvdata_, (int)(1.5*width_*height_));
			}
			break;

		case TEST_BLUEPASSION:
            {
			    char *y;
			    char *u;
			    char *v;
			    y = (char*)frame_;
			    u = y + width_ * height_;
			    if (cformat_ == CF_422) {
				    v = u + width_ * height_ / 2;
			    } else if (cformat_ == CF_411) {
				    v = u + width_ * height_ / 4;
			    } else { // CF_CIF
				    v = u + width_ * height_ / 4;
			    }
			    // pick out some color values
			    static int a=0x10;
			    ++a;
			    if (a > 0xf0)
					    a=0x10;
			    int b = 0xf0 - (a - 0xf0);

			    if (cformat_ == CF_422) {
				    memset(y, a, width_*height_ / 2); // top half of y
				    memset(y+(width_*height_/2), b, width_*height_/2); // bottom half of y
				    memset(u, 0xff, width_*height_ / 2); // u array
				    memset(v, 0x0, width_*height_ / 2); // v array
			    } else { // cif or 411
				    memset(y, a, width_*height_ / 2); // top half of y
				    memset(y+(width_*height_ / 2), b, width_*height_/2); // bottom half of y
				    memset(u, 0xff, width_ * height_ / 4); // u array
				    memset(v, 0x0, width_ * height_ / 4); // v array
			    }
            }
			break;

        default:
            printf("invalid TestDevice port number\n");
            break;
	}
	return (1);
}

void VideoCaptureTest::grab()
{
	if (capture() == 0)
			return;
	suppress(frame_);
	saveblks(frame_);

	YuvFrame f(media_ts(), frame_, crvec_, outw_, outh_);
	target_->recv(&f);
}

void VideoCaptureTest::format()
{
	switch (decimate_) {
	case 1: // full-sized
		width_ = SP601_NTSC_WIDTH;
		height_ = SP601_NTSC_HEIGHT;
		break;
    
	case 2: // CIF-sized
		width_ = CIF_WIDTH;
		height_ = CIF_HEIGHT;
		break;

	case 4: // QCIF - may not work
		width_ = QCIF_WIDTH;
		height_ = QCIF_HEIGHT;
		break;
	}
  
	switch (cformat_) {
	case CF_CIF:
		set_size_cif(width_, height_);
		break;
	case CF_411:
		set_size_411(width_, height_);
		break;
	case CF_422:
		set_size_422(width_, height_);
		break;
	}

	// after calling set_size_xxx, you must always allocate the 
	//	reference buffer ref_ 
	allocref();

	// if you're using a file as the test source, reconvert the RGB 
	//	source to the new dimensions
	if (port_ == TEST_FILE) {
		// create a place to store the YUV data.
		u_int8_t *yuvdata = new u_int8_t[2*width_*height_];

		// convert the RGB input into YUV
		RGB_Converter* converter;
		if (cformat_ == CF_422) {
			converter = new RGB_Converter_422 (24, NULL, 0);
		} else if (cformat_ == CF_411) {
			converter = new RGB_Converter_411 (24, NULL, 0);
		} else { // 4:2:0
			converter = new RGB_Converter_411 (24, NULL, 0);
		}
		converter->convert ((u_int8_t *)rgbdata_, rgbwidth_, rgbheight_, 
				yuvdata, width_, height_, 0);
		delete converter;

		// free the old YUV buffer
		if (yuvdata_ != NULL) {
				delete[] yuvdata_;
		}
		yuvdata_ = (char *)yuvdata;
	}

	return;
}


int VideoCaptureTest::set_file (const char* filename)
{
	char type[2+5];
	int colorscale;
	u_int8_t *rgbdata;
	int rgbwidth, rgbheight;
	// The dummy variables are for the purpose of wasting the header.
	char dummy, dummyl[128];
	int i, j, k, l;

	// open the file
	ifstream fid(filename, ios::in);
	if (fid.fail()) {
		fid.close();

		// if no filename was passed, try the last supplied filename
		if (strcmp (filename, "") == 0) {
			// no name supplied -> try the last name used
			fid.open(old_filename, ios::in);
			if (fid.fail()) {
				fid.close();
				return 0;
			}
		} else {
			// incorrect name supplied
			return 0;
		}
	} else {
		strcpy (old_filename, filename);
	}

	// read the file
	fid >> type;
	while (fid.peek() <= 13) { fid.get(dummy); }
	while (fid.peek() == '#') { fid.getline(dummyl,128,'\n'); }
	fid >> rgbwidth >> rgbheight;
	fid >> colorscale;
	// get rid of the whitespace just after the maximum color-component value
	fid.get(dummy);

	// create a place to store the RGB data.
	rgbdata = new u_int8_t[3 * rgbwidth * rgbheight];

	// read the file
	if (type[1]=='6') {
		// P6: read in binary chars and convert to ASCII.
		fid.read((char *) rgbdata, 3 * rgbwidth * rgbheight);

	} else if (type[1]=='3') {
		// P3: read in ASCII values.
		char temp;
		for (j = 0; j < rgbheight; j++) {
			for (k = 0; k < rgbwidth; k++) {
				for (l = 0; l < 3; l++) {
					// calculate the pixel component index
					i = 3 * (j * rgbwidth + k) + l;

					// get one non-white character
					fid.get(temp);
					while (temp==' ' || temp=='\t' || temp=='\n' || temp=='\r' ) {
						fid.get(temp);
					}
					rgbdata[i] = (temp - '0');
					if (temp < '0' || temp > '9') {
						printf ("Corrupted ASCII PPM file (%s)\n", old_filename);
						fid.close();
						delete[] rgbdata;
						return 0;
					}

					// read all ASCII numbers (0-9) you can
					fid.get(temp);
					while (temp >= '0' && temp <= '9') {
						rgbdata[i] = rgbdata[i] * 10 + (temp - '0');
						fid.get(temp);
					}
				}
			}
		}

	} else {
		// FIXME: [Px] unsupported (P4=raw pbm, P5=raw pgm, P2=ascii pgm, 
		//	P1=ascii pgm
		delete[] rgbdata;
		printf ("Sorry, PGM and PBM files still unsupported (P%c)\n", type[1]);
		fid.close();
		return 1;
	}
	fid.close();


	// free the old RGB buffer
	if (rgbdata_ != NULL) {
			delete[] rgbdata_;
	}

	// check in the new RGB buffer
	rgbdata_ = (char *)rgbdata;
	rgbwidth_ = rgbwidth;
	rgbheight_ = rgbheight;

	// mark the source as a file
	port_ = TEST_FILE;

	// format the frame dimensions
	format();

	return 0;
}

int VideoCaptureTest::set_bluepassion ()
{
	port_ = TEST_BLUEPASSION;
	if (running_)
			format();
	return 0;
}


