# ui-main.tcl --
#
#       FIXME: This file needs a description here.
#
# Copyright (c) 2001-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

Import enable
import Observer SessionReceiver TopLevelWindow
import VEListbox VicReceiver

######
#veUI
# This class fires up the main user interface for the Video Explorer
# application.  It creates the menubar and the basic layout for the
# multiple views.
#

Class veUI -superclass Observer 

veUI instproc init {w} {
    # session_list: Tcl list of IP addresses of the active sessions
    # listview_listbox_: A listbox widget that is used during the list view
    $self instvar listview_listbox_
    global view old_view default_prefs session_list
 
    wm title . "Video Explorer"
    wm minsize . 250 200
    
    $self build_menubar $w
    $self init_keybindings    
    $self init_preferences

    # List of all sessions that have been added
    set session_list {}
    # Denotes the current view (default: Vic)
    set view $default_prefs(view)
    # Denotes the previous view (default: Vic)
    set old_view $view

    # Two frames which are packed and unpacked when the user switches views
    frame .list
    frame .vic
    set listview_listbox_ [new VEListbox .list]
    
    if {$view == "vic"} {
	pack .vic -anchor nw
    } elseif {$view == "list"} {
	pack .list -fill both -expand true
    }
}

#
# Initializes some simple keybindings on the main Video Explorer widget.
#
veUI instproc init_keybindings {} {
    bind . <a> "new AddSessionPopUp $self"
    bind . <r> "new RemoveSessionPopUp $self"
    bind . <q> "exit"    
}

# 
# Builds the File/Edit/View menubar at the top of the Video Explorer widget.
#
veUI instproc build_menubar {w} {
    menu .menubar
    . config -menu .menubar
    foreach m {File Edit View} {
	set $m [menu .menubar.m$m -tearoff 0]
	.menubar add cascade -label $m -menu .menubar.m$m
    }
    $File add command -label "Add Session" \
	    -command "new AddSessionPopUp $self"
    $File add command -label "Remove Session" \
	    -command "new RemoveSessionPopUp $self"
    $File add command -label Quit -command exit

    $Edit add command -label Preferences -command "Pref_Dialog"

    $View add radio -label Vic -variable view -value vic\
	    -command "$self toggle_view"
    $View add radio -label List -variable view -value list\
	    -command "$self toggle_view"

}

#
# Checks that the entered IP to add is in valid addr/port format and that
# the user is not already in the session when adding a session.
#
veUI instproc check_spec_to_add {} {
    global session_list
    
    # gets the spec that was typed by the user into the entry widget
    set spec_ [.addSessionPopUp.entry get]
    set spec_split [split $spec_ /]
    set n [llength $spec_split]
    set port [lindex $spec_split 1]
    if { $n == 2 } {
	if { [string match \[0-9\]* $port] && $port <= 65536} {
	    if { [lsearch -exact $session_list $spec_] == -1 } {
		lappend session_list $spec_
		$self add_session $spec_
	    } else {
		new ErrorPopUp .spec_exists_popup "Error in IP Address" \
			"You are already connected to this session: $spec_" \
			.addSessionPopUp
	    }
	} else {
	    new ErrorPopUp .badport "Error in IP Address" \
		    "Port number must be smaller than 65537" \
		    .addSessionPopUp
	}
    } else {
	new ErrorPopUp .bad_spec_popup "Error in IP Address" \
		"Please specify both address and port\
		in the form: address/port" .addSessionPopUp
    }
}

#
# Checks that the entered IP to remove is in valid addr/port format and that
# the user is connected to that session when removing a session.
#
veUI instproc check_spec_to_remove {} {
    global session_list

    # gets the spec that was typed by the user into the entry widget
    set spec_ [.removeSessionPopUp.entry get]
    set spec_split [split $spec_ /]
    set n [llength $spec_split]
    set port [lindex $spec_split 1]
    if { $n == 2 } {
	if { [string match \[0-9\]* $port] && $port <= 65536} {
	    set spec_index [lsearch -exact $session_list $spec_]
	    if { $spec_index != -1 } {
		$self remove_session $spec_		
	    } else {
		new ErrorPopUp .spec_nonexists_popup "Error in IP Address"\
			"You are not connected to this session: $spec_"\
			.removeSessionPopUp
	    }
	} else {
	    new ErrorPopUp .badport "Error in IP Address" \
		    "Port number must be smaller than 65537" \
		    .removeSessionPopUp
	}
    } else {
	new ErrorPopUp .bad_spec_popup "Error in IP Address" \
		"Please specify both address and port\
		in the form: address/port" .removeSessionPopUp
    }
}

#
# Returns the index of the spec in the session_list
#
veUI instproc get_spec_index {spec} {
    global session_list
    return [lsearch -exact $session_list $spec]
}

#
# Adds the session by making a new VicReceiver and adding the spec
# to the list-view listbox.
#
veUI instproc add_session {spec} {
    set spec_window [join [split $spec .] /]
    global sr_vic_$spec_window
    $self instvar listview_listbox_
    destroy .addSessionPopUp

    set vic_session [new VicReceiver $spec $self]
    $listview_listbox_ insert end $spec "$self session_popup $spec"
}

#
# Removes the session by destroying the vic-view session and removing the
# spec from the list-view listbox.  It also destroys any related toplevel
# windows relating to that session.
#
veUI instproc remove_session {spec} { 
    $self instvar listview_listbox_
    global session_list
    set spec_window [join [split $spec .] /]
    global sr_vic_$spec_window sr_list_$spec_window
    set spec_index [$self get_spec_index $spec]
    set session_list [lreplace $session_list $spec_index $spec_index]

    destroy .removeSessionPopUp
    $listview_listbox_ delete $spec_index
    destroy .vic.$spec_window
    [set sr_vic_$spec_window] detachAll
    [set sr_vic_$spec_window] unpack_VEuw

    if {[winfo exists .session$spec_window]} {
	[set sr_list_$spec_window] detachAll
	[set sr_list_$spec_window] unpack_VEuw
	destroy .session$spec_window
    }
}

#
# Toggles the view as the user requests.  Depending on the old_view and
# current view, it packs and unpacks windows as necessary.
#
veUI instproc toggle_view {} {
    global session_list

    #remove the old_view
    global view old_view    
    if {$old_view == "vic"} {
	$self unpack_vic
    } elseif {$old_view == "list"} {
	$self unpack_list
    }
    
    #pack the requested view
    if {$view == "vic"} {
	pack .vic -anchor nw
	set old_view vic
    } elseif {$view == "list"} {
	pack .list -fill both -expand true
	set old_view list
    }
}

#
# Unpacks the vic-view only by unpacking the parent frame .vic and it's
# associated toplevel windows, if they exist.
#
veUI instproc unpack_vic {} {
    global session_list
    foreach session $session_list {
	set spec_window [join [split $session .] /]
	global sr_vic_$spec_window
	[set sr_vic_$spec_window] unpack_VEuw
    }
    pack forget .vic
    
}

#
# Unpacks the list-view by unpacking the parent frame .list and it's
# associated toplevel windows, if they exist.
#
veUI instproc unpack_list {} {
    global session_list
    foreach session $session_list {
	set spec_window [join [split $session .] /]
	global sr_list_$spec_window
	if {[winfo exists .session$spec_window]} {
	    [set sr_list_$spec_window] detachAll
	    [set sr_list_$spec_window] unpack_VEuw
	    destroy .session$spec_window
	}
    }
    pack forget .list
}

#
# In list-view, pop-up a window that has a SessionReciever widget, which 
# displays video for the selected spec.
#
veUI instproc session_popup {spec} {
    set spec_window [join [split $spec .] /]
    set w .list$spec_window
    global sr_list_$spec_window
        
    if {[winfo exists $w]} {	
	[set sr_list_$spec_window] detachAll
	destroy $w
    } else {
	toplevel $w
	wm title $w $spec
	wm resizable $w 0 0
	set sr_list_$spec_window [new SessionReceiver $spec $w]

	# frame from SessionReceiver init
#	pack $w -side left -expand true 
    }
}

veUI instproc init_preferences {} {
    global default_prefs
    Pref_Init ve-user-defaults ve-app-defaults
    Pref_Add {{default_prefs(view) view {CHOICE vic list} "View" \
	    "This setting toggles the view of the VideoExplorer at startup."}}
   
}
    

#######
#AddSessionPopUp
# Creates an entry pop-up window for [File->Add Session] 
#

Class AddSessionPopUp -superclass TopLevelWindow

AddSessionPopUp instproc init {ui_} {
    $self next .addSessionPopUp
    set path_ .addSessionPopUp
    toplevel $path_
    wm title $path_ "Add Session..."
    wm resizable $path_ 0 0

    label $path_.label -text "Enter the IP address of the session\
	    you would like to join:" -anchor w
    entry $path_.entry -relief sunken
    grid $path_.label -pady 5 -padx 10
    grid $path_.entry -sticky we -padx 20
    bind $path_.entry <Return> "$ui_ check_spec_to_add"

    set buttons_ $path_.buttons
    frame $buttons_
    grid $buttons_ -sticky s
    button $buttons_.enter -text Join -pady 5\
	    -command "$ui_ check_spec_to_add"
    button $buttons_.cancel -text Cancel -pady 5\
	    -command "destroy .addSessionPopUp"
    pack $buttons_.cancel -side right
    pack $buttons_.enter -side right

    focus $path_
    grab $path_

}

######
#RemoveSessionPopUp
# Creates an entry pop-up window for [File->Remove Session]
#

Class RemoveSessionPopUp -superclass TopLevelWindow

RemoveSessionPopUp instproc init {ui_} {
    $self next .removeSessionPopUp
    set path_ .removeSessionPopUp
    toplevel $path_
    wm title $path_ "Remove Session..."
    wm resizable $path_ 0 0

    label $path_.label -text "Enter the IP address of the session\
	    you would like to remove:" -anchor w
    entry $path_.entry -relief sunken
    grid $path_.label -pady 5 -padx 10
    grid $path_.entry -sticky we -padx 20
    bind $path_.entry <Return> "$ui_ check_spec_to_remove"

    set buttons_ $path_.buttons
    frame $buttons_
    grid $buttons_ -sticky s
    button $buttons_.enter -text Remove -pady 5\
	    -command "$ui_ check_spec_to_remove"
    button $buttons_.cancel -text Cancel -pady 5\
	    -command "destroy .removeSessionPopUp"
    pack $buttons_.cancel -side right
    pack $buttons_.enter -side right

    focus $path_
    grab $path_
}


######
#ErrorPopUp
# Creates a simple error window with a text message and 'OK' button
#

Class ErrorPopUp -superclass TopLevelWindow

ErrorPopUp instproc init {w title text regrab} {
    set popup $w
    toplevel $popup
    wm title $popup $title
    wm resizable $popup 0 0
    message $popup.msg -justify center -aspect 1000 -text $text
    button $popup.ok -text OK \
	    -command "destroy $popup"
    grid $popup.msg
    grid $popup.ok
    bind $popup <Escape> "destroy $popup"
    bind $popup <Return> "destroy $popup"
    focus $popup
    grab $popup
    tkwait window $popup
    grab release $popup
    focus $regrab
    grab $regrab
}
    
