# rendezvous.tcl --
#
#       FIXME: This file needs a description here.
#
# Copyright (c) 1998-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
#  @(#) $Header: /usr/mash/src/repository/mash/mash-1/tcl/net/rendezvous.tcl,v 1.14 2002/02/03 04:28:05 lim Exp $


# --------------------------------------------------
# TODO:
# - periodically flush to disk
# - rather than double msg storage (rv and rvm), keep one copy.
# - propogate timeouts to manager then to mngr observers via
#      notify_observers rendez_timeout_TYPE upcalls
# - perform vs intersection alg
# - ...
# --------------------------------------------------


import Trace AnnounceListenManager Observer Observable
#Trace on ; Trace add RendezvousManager

# --------------------------------------------------
# Class RendezvousManager
# --------------------------------------------------


# class for managing a set of rendezvous channels
# and the data sent and received on it.  Assumes msgs of the form:
#<br> "type: attr=val attr2=val2 ..." <br>
# and stores them in memory arrays.
#<p>
#Data types:
#<p> rv_() == key: spec of channel; val: Rendezvous objects handling that spec
#<p> rvMsgs_ == list of RVMsg's
#<p>
#
# Observes the individual Rendezvous channels, is observable
# by classes wishing to "register interest" in particular
# types of msgs. They do so by defining a ``rendez_recv_TYPENAME'' method
# for each type of interest.
# To receive all (new, non-refresh) msgs, they should define a
# method named just ``rendez_recv'')
#
#<p> FIXME -- add rendez_timeout notifications, use them in SrvAlloc
Class RendezvousManager -superclass {Observer Observable}


#
RendezvousManager public init {{speclist ""}} {
    Trc $class "--> ${class}::$proc"
    $self next
    $self instvar scopes_ rvMsgs_
    set rvMsgs_ ""

    if {$speclist == ""} {set speclist [$self get_option rendez]}
    if {$speclist == ""} {
	# none specified, use global spec (and refine from there)
	set s 224.2.127.253/1202/32
	$self add_spec $s
	set scopes_($s) "global"
	#puts "Rendevous messages sent/received on default global rv: $s"
    } else {
	foreach i [split $speclist ,] {
	    $self add_spec $i
	}
	#puts "Rendevous messages sent/received on $speclist"
    }

}


#
RendezvousManager public add_spec {s} {
    Trc $class "--> ${class}::$proc"
    $self instvar rv_ local_rv_

    if [info exists rv_($s)] {return}

    set r [new Rendezvous $s]
    set rv_($s) $r
    $r attach_observer $self
    if ![info exists local_rv_] {
	set local_rv_ $s
    }
}

#
RendezvousManager public rm_spec {s} {
    Trc $class "--> ${class}::$proc"
    $self instvar rv_
    if {[array names rv_ $s] != ""} {
	$rv_($s) detach_observer $self
	delete $rv_($s)
	unset rv_($s)
    } else {
	puts "Error: attempted to remove bad spec `$s'"
    }
}


#
RendezvousManager public get_specs {} {
    $self instvar rv_
    return [array names rv_]
}

# return the spec of the "most-refined/smallest" virtual scope
# discovered so far. (This is the "local" or "current" vs.)
#
RendezvousManager public get_local_rv {} {
    Trc $class "--> ${class}::$proc"
    $self instvar local_rv_
    return $local_rv_
}

# accept a query for a substring of a rendezvous msg
#<p>
# `queryString' is of the form
# <br> dog & !cat
# <br> or
# <br> name=janet | !type=mother | grr=hmm
# <br> etc....
# <br> i.e., special characters are &, |, and !
# <br> currently parentheses are NOT supported, so only all & or
# all | clauses make sense
#
#<br>FIXME add parens
#<br>FIXME this is really unoptimized for now .. linear in both
# number of search terms and number of entries
#
RendezvousManager public query {queryString} {
    Trc $class "--> ${class}::$proc"

    set msgs [$self query_msgs $queryString]

    if {$msgs == ""} {
	return ""
    } else {
	#FIXME only replies with *first* matching response
	#foreach i $indicies {lappend reply $rvData_($i)}
	return [lindex $msgs 0]
    }
}


# query as in RendezvousManager::query, but return metadata
# instead
#
RendezvousManager public query_metadata {queryString} {
    Trc $class "--> ${class}::$proc"

    set msgs [$self query_msgs $queryString]

    if {$msgs == ""} {
	return ""
    } else {
	#FIXME only reply with *one* response (first one)
	return [lindex $msgs 0]
    }
}

#
RendezvousManager public query_msgs {queryString} {
    Trc $class "--> ${class}::$proc $queryString"

    set and [string match "* & *" $queryString]
    set or [string match "* | *" $queryString]
    if {$and && $or } {
	puts "queries with both `and' (&) and `or' (|) \
		is not currently supported... returning {}."
	return ""
    }
    if {$and} {
	# compute and (&) clause
	set msgs ""
	set q [split $queryString "&"]
	foreach field $q {
	    set field [string trim $field]
	    set msgs [$self field_query $field $msgs]
	    if {$msgs==""} {return ""}
	}
    } else {
	# compute or (|) clause or single-field clause
	set msgs ""
	set q [split $queryString "|"]
	foreach field $q {
	    set field [string trim $field]
	    foreach msg [$self field_query $field] {
		if {$msg != ""} {lappend msgs $msg}
	    }
	}
	# pull out non-unique indicies due to multiple matches
	set msgs [$self uniq $msgs]
    }
    Trc $class "matching msgs = $msgs"
    return $msgs
}


# returns a list of RVMsgs matching the query for
# `qField'.  If `msgList' is specified, only those msgs
# are searched (allowing this to be iteratively refined for AND
# clauses)
#
RendezvousManager private field_query {qField {msgList ""}} {
    Trc $class "--> ${class}::$proc $qField $msgList"
    $self instvar rvMsgs_

    if {$msgList == ""} {
	set msgList $rvMsgs_
    }
    set results ""

    # do `not' query by performing normal query,
    # then inverting the results
    set is_not_query 0
    if {[string match !* $qField]} {
	set qField [string range $qField 1 end]
	set is_not_query 1
    }

    foreach m $msgList {
	if {[string first "$qField" [$m get_msg]] != -1} {
	    lappend results $m
	}
    }

    if {$is_not_query} {
	set newResults ""
	foreach m $msgList {
	    if {[lsearch -exact $results $m] == -1} {
		lappend newResults $m
	    }
	}
	Trace $class "-- field_query pre-NOT'd results: $results"
	set results $newResults
    }
    Trc $class "-- field_query results: $results"
    return $results
}


# removes duplicates from input list `l' and return the new list
#
RendezvousManager private uniq {l} {
    Trc $class "--> ${class}::$proc"
    set uniqL ""
    foreach i $l {
	if {[lsearch -exact $i $uniqL] == -1} {
	    lappend uniqL $i
	}
    }
    return $uniqL
}


# called when a msg `data' is received on channel `rspec';
# stores it into the rvData_ array and updates the rvMetadata_
# array.  If the msg was received before, only update the meta-data.
# Splits all received msgs into individual lines and treats each
# line as a separate msg.
#
RendezvousManager public recv_msg {rspec addr port data size} {
    Trc $class "--> ${class}::$proc $rspec $data"
    $self instvar rv_ rvMsgs_
    #set rv_obj $rv_($rspec)

    foreach d [split $data \n] {
	set d [string trim $d]
	if {$d == ""} {continue}

	set newrvmsg [new RVMsg $data $rspec $addr/$port]
	$newrvmsg update_meta_field "time=[clock seconds]"

	set type [$newrvmsg get_type]

	# a repeat msg?
	set dupmsg -1
	foreach rv $rvMsgs_ {
	    if {[$newrvmsg get_msg] == [$rv get_msg]} {
		set dupmsg $rv
	    }
	}

	# if new, store it
	if {$dupmsg == -1} {
	    # check for things that shouldn't be cached
	    switch $type {
		"query" {set cache_it 0}
		default {set cache_it 1}
	    }
	    if $cache_it {
		lappend rvMsgs_ $newrvmsg
	    }

	    # notify ourself of "scope" msgs
	    if {$type == "scope"} {
		$self recv_scope $newrvmsg
	    }
	} else {
	    $dupmsg update_meta_field "time=[clock seconds]"
	    delete $newrvmsg
	    set newrvmsg $dupmsg
	}

	# notify interested observers
	$self notify_observers rendez_recv $newrvmsg
	$self notify_observers rendez_recv_$type $newrvmsg
    }
}

# Handle a msg with type "scope"
#  -- start listening on new scope
#  -- update local_rv_ if necessary/possible
#  -- perform intersection operations [FIXME intersection not yet implemented]
#
RendezvousManager private recv_scope {rv_msg} {
    Trc $class "--> ${class}::$proc"
    $self instvar local_rv_ scopes_
    set sname [$rv_msg get_field name]
    set sspec [$rv_msg get_field spec]
    if {$sname == "" || $sspec == ""} {
	puts "Improperly formatted scope msg: [$rv_msg get_msg]"
    }
    set scopes_($sspec) $sname
    $self add_spec $sspec
    if {$local_rv_ == [$rv_msg rspec]} {
	set local_rv_ $sspec
    }
}

# returns the name associated with `spec'. If it is not
# known, returns the `spec' unchanged.
#
RendezvousManager public get_spec_name {spec} {
    Trc $class "--> ${class}::$proc"
    set r [$self query "scope: & spec=$spec"]
    set n [$r get_field name]
    if {$n == ""} {return $spec}
    return $n
}



# start announcing `msg' on rv with spec `spec'. If `spec'=={},
# announce on the local/current rv.
RendezvousManager public start {spec msg} {
    Trc $class "--> ${class}::$proc `$spec' announcing `$msg'"
    $self instvar rv_
    if {$spec == ""} {set spec [$self get_local_rv]}
    if [info exists rv_($spec)] {
	$rv_($spec) start $msg
    } else {
	puts "Error: not connected to `$spec': won't send msg to that addr."
    }
}

# stop announcing `msg' on rv with spec `spec'. If `spec'=={},
# stop the msgs on the current/local rv.
RendezvousManager public stop {spec msg} {
    Trc $class "--> ${class}::$proc `$spec' stop announcing `$msg'"
    $self instvar rv_
    if {$spec == ""} {set spec [$self get_local_rv]}
    if [info exists rv_($spec)] {
	$rv_($spec) stop $msg
    } else {
	puts "Error: not connected to `$spec': can't stop msgs there."
    }
}

# --------------------------------------------------
# Class Rendezvous
# --------------------------------------------------

import Timer/Adaptive/ConstBW

# a rendezvous channel monitor that listens on *one* addr/port
# and maintains info about that channel only. Propogates msgs upward
# to a interested observers such as a RendezvousManager
#
Class Rendezvous -superclass {Observable AnnounceListenManager}


#
Rendezvous public init {spec} {
    Trc $class "--> ${class}::$proc"
    eval [list $self] next $spec
    $self instvar msgs_ spec_ snet_ rnet_
    set spec_ $spec
    #FIXME
    if {$snet_ != ""} {$self ttl 16}
    $self set_timeout 600
    set msgs_ ""

    set t [new Timer/Adaptive/ConstBW 10000]
    $t randomize
    $self timer $t

    $self process_timeouts
}

# receive rendezvous ads, store
Rendezvous private recv_announcement {addr port data size} {
    Trc $class "--> ${class}::$proc $data"
    $self instvar spec_

    set t [$self get_timer]
    $t sample_size $size

    foreach msg [split $data \n] {
	$self update_msg $msg
	$self notify_observers recv_msg $spec_ $addr $port $msg $size
    }
}

# insert/update new msg
Rendezvous private update_msg {newMsg} {
    Trc $class "--> ${class}::$proc"
    $self instvar msgs_ msgtimestamps_

    set msgtimestamps_($newMsg) [clock seconds]
    if {[lsearch  $msgs_ $newMsg] != -1} {
	lappend $msgs_ $newMsg
	[$self get_timer] incr_nsrcs
    }
}

# clean out old msgs through a timout process
Rendezvous private process_timeouts {} {
    Trc $class "--> ${class}::$proc"
    $self instvar msgs_ msgtimestamps_ timeout_
    if {$timeout_ <= 0} {
	return
    }
    set currTime [clock seconds]
    foreach i $msgs_ {
	set t $msgtimestamps_($i)
	if {[expr $currTime - $t] > $timeout_} {
	    puts "Rendezvous: timing out msg $i"
	    set ind [lindex $i $msgs_]
	    set msgs_ [lreplace $msgs_ $ind $ind]
	    unset msgtimestamps_($i)
	    [$self get_timer] incr_nsrcs -1
	}
    }
    # catch in case we're deleted
    after 5000 "catch {$self process_timeouts}"
}

# set timeout ; <= 0 means no timeout
Rendezvous public set_timeout {seconds} {
    Trc $class "--> ${class}::$proc"
    $self instvar timeout_
    set timeout_ $seconds
}





# --------------------------------------------------
# Class RVMsg
# --------------------------------------------------

# a rendezvous msg that was pulled off a rendezvous channel
#
Class RVMsg

# create RVMsg from text string <i>msg</i>, received on rv channel
# <i>rspec</i>, sent from sender <i>sender_spec</i>
RVMsg public init {msg rspec sender_spec} {
    $self instvar msg_ rspec_ sender_spec_ metadata_
    set msg_ $msg
    set rspec_ $rspec
    set sender_spec_ $sender_spec
    set metadata_ "time=[clock seconds]"
}

# grabs inital colon-terminated portion of this msg's text
# string, by convention the the "type" of the msg.
# <p>
# i.e. for msg
# "will-provide: mash-object=RemoteVicApplication spec=Z ctrlspec=Y"
# <p>
# <br> `get_type' <br> returns "will-provide"
#
RVMsg public get_type {} {
    Trc $class "--> ${class}::$proc"
    $self instvar msg_
    set t [string trim [lindex $msg_ 0]]
    set lst [split $t :]
    if {[lindex $lst end] == ""} {
	return [lindex [split $t :] 0]
    }
    return ""
}

#
RVMsg public fields {} {
    Trc $class "--> ${class}::$proc"
    $self instvar msg_

    set flist ""
    set m [lrange $msg_ 1 end]
    foreach i $m {
	lappend flist [lindex [split $i =] 0]
    }
    return $flist
}

# gets the value of field 'field' in this msg
# <p>
# i.e. for msg text:
# "will-provide: mash-object=RemoteVicApplication spec=Z ctrlspec=Y"
# <p>
# <br> `get_field ctrlspec' <br> returns Y
# <br> `get_field spec' <br> returns Z
# <br> `get_field yomamaspec' <br> returns {}
#
RVMsg public get_field {field} {
    Trc $class "--> ${class}::$proc"
    $self instvar msg_
    set i [lsearch $msg_ "*$field=*"]
    if {$i == -1} {
	return ""
    } else {
	set attVal [lindex $msg_ $i]
	set idx [string first = $attVal]
	return [string range $attVal [expr $idx+1] end]
    }
}

#
RVMsg public has_field {f} {
    Trc $class "--> ${class}::$proc"
    $self instvar msg_
    set i [lsearch $msg_ "*$field=*"]
    if {$i == -1} {
	return 0
    } else {
	return 1
    }
}

#
RVMsg public get_msg {} {
    $self instvar msg_
    return $msg_
}

#
RVMsg public rspec {} {
    $self instvar rspec_
    return $rspec_
}

#
RVMsg public sender_spec {} {
    $self instvar sender_spec_
    return $sender_spec_
}

#
RVMsg public sender_addr {} {
    $self instvar sender_spec_
    return [lindex [split $sender_spec_ /] 0]
}

#
RVMsg public sender_port {} {
    $self instvar sender_spec_
    return [lindex [split $sender_spec_ /] 1]
}

#
RVMsg public get_metadata {} {
    $self instvar metadata_
    return $metadata_
}

#
RVMsg public set_metadata {m} {
    $self instvar metadata_
    set metsdata_ $m
}

#
RVMsg public update_meta_fields {fields} {
    foreach attval $fields {
	$self update_meta_field $attval
    }
}

#
RVMsg public update_meta_field {m} {
    $self instvar metadata_
    set f [lindex [split $m =] 0]
    set i [lsearch $metadata_ "*$f=*"]
    if {$i == -1} {
	set metsdata_ "$metadata_ $m"
    } else {
	set metadata_ [lreplace $metadata_ $i $i $m]
    }
}

#
RVMsg public rm_meta_field {f} {
    $self instvar metadata_
    set i [lsearch $metadata_ "*$f=*"]
     if {$i == -1} {
	 return 0
    } else {
	set metadata_ [lreplace $metadata_ $i $i]
	return 1
    }
}

#
RVMsg public get_meta_field {f} {
    $self instvar metadata_
    set i [lsearch $metadata_ "*$f=*"]
    if {$i == -1} {
	return ""
    } else {
	return [lindex [split [lindex $metadata_ $i] =] 1]
    }

}

#
RVMsg public has_meta_field {f} {
    Trc $class "--> ${class}::$proc"
    $self instvar metadata_
    set i [lsearch $metadata_ "*$field=*"]
    if {$i == -1} {
	return 0
    } else {
	return 1
    }
}

# dump string of all data about this msg
RVMsg private data {} {
    $self instvar msg_ rspec_ sender_spec_
    return "$rspec_ $sender_spec_ $msg_"
}

