# mb-pagemgr.tcl --
#
#       FIXME: This file needs a description here.
#
# Copyright (c) 1998-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
#   @(#) $Header: /usr/mash/src/repository/mash/mash-1/tcl/mb/mb-pagemgr.tcl,v 1.11 2002/02/03 04:27:17 lim Exp $

import Observable

# MediaBoard Page Manager <br>
# Manages mb pages, used to switch pages and manages canvas for each page
# <p>
# notifies the following events:
# <ul>
# <li> add_page $page_id: add a new page with id: $page_id
# <li> switch_page $page_id: switches to page with id: $page_id
# </ul>
Class MBPageMgr -superclass Observable

# $mbMgr is the mb manager object
#
MBPageMgr public init {mbMgr} {
	# when new items were drawn onto a new page, we sched a event to
        # happen when idle for switching to this page, $idleHandler_
        # stores the scheduled event id, while $nextPage_ is the next page
        # $idleHandler_ is scheduled to switch to.
        $self set idleHandler_ {}
        $self set nextPage_ {}

	$self set currPage_ {}
	$self set mgr_ $mbMgr
}

# $mbMgr is the mb manager object
#
MBPageMgr public destroy {} {
	$self instvar arPage2Canv_
	foreach page_id [array names arPage2Canv_] {
		mtrace trcMB "deleting canvas for page: $page_id"
		delete $arPage2Canv_($page_id)
	}
}

# switches to the page corr. to <u>page_id</u> when idle <br>
# since this procedure might be called when mb
# is in a process of doing something (e.g. half way drawing a line),
# we arrange for switching to occur after all events have been handled.
#
MBPageMgr public switch_page_later {page_id} {
        $self instvar idleHandler_ nextPage_

        # Reschedule if a different page is to be switched.
        #  - note that nextPage_ is initialized to "" which guarantees
        #    execution on the 1st round
        #  - also, nextPage_ will be equal to the current page when no
        #    event is scheduled, so this proc does nothing unless
        #    a different page is to be switched.

#        DbgOut "in: switch to $page_id $nextPage_ $idleHandler_"
        if {[string compare $nextPage_ $page_id]} {
                after cancel $idleHandler_
#                DbgOut "switch to $page_id"
                set idleHandler_ [after idle $self switch_page $page_id]
                set nextPage_ $page_id
        }
}

MBPageMgr public create_new_page {{page_name {}}} {
	$self instvar mgr_

        if {$page_name=={}} {
                set page_name "Page "
                append page_name [$self nextPageNumber]
        } else {
                set page_name [lindex $args 0]
        }
        set page_id [[$mgr_ sender] create_page $page_name]
        return $page_id
}

# inform mgr of a new page and canvas pair <br>
# throws error if it is not a new page (for now) <br>
#
MBPageMgr public add_page {page_id canvas} {
	$self instvar arCanv2Page_ arPage2Canv_
	if [info exists arCanv2Page_($canvas)] {
		error "MBPageMgr add_page called with already created page"
	}
	set arCanv2Page_($canvas) $page_id
	set arPage2Canv_($page_id) $canvas
	$self notify_observers add_page $page_id
}

# returns the current page
MBPageMgr public current_page {} {
	return [$self set currPage_]
}

# returns the current page
MBPageMgr public current_canvas {} {
	$self instvar currPage_
	if {$currPage_!={}} {
		return [$self page2canv $currPage_]
	} else {
		return {}
	}
}

# returns a list of pages
# right now the order is quite random
# <! FIXME: should be sorted according to MBPageList>
MBPageMgr public pagelist {} {
	$self instvar arPage2Canv_
	return [array names arPage2Canv_]
	return
}

# returns 0 if $page_id is not found, 1 otherwise
MBPageMgr public has_page {page_id} {
	$self instvar arPage2Canv_
	return [info exists arPage2Canv_($page_id)]
}

# returns the canvas associated with $page_id
MBPageMgr public page2canv {page_id} {
	$self instvar arPage2Canv_
	if ![info exists arPage2Canv_($page_id)] {
		error "cannot find $page_id"
	}
	return $arPage2Canv_($page_id)
}

# returns the page associated with canvas
MBPageMgr public canv2page {canvas} {
	$self instvar arCanv2Page_
	if ![info exists arCanv2Page_($canvas)] {
		error "cannot find $canvas"
	}
	return $arCanv2Page_($canvas)
}

# switches current page to $page_id
MBPageMgr private switch_page {page_id} {
	$self instvar currPage_ nextPage_ sender_
        $self instvar menubutton_ mgr_

        if {$page_id==$currPage_} {
                mtrace trcVerbose "same page, do nothing"
                return
        }
        mtrace trcVerbose "switching to $page_id"
	puts "before update idletasks 1"
        update idletasks
	puts "after update idletasks 1"
	if {$currPage_ != {}} {
		set old_canv [$self page2canv $currPage_]
		$old_canv unpack
	}

	set new_canv [$self page2canv $page_id]
	$new_canv pack -side left -fill both -expand true -anchor nw
	if {$currPage_ != {}} {
		$new_canv transfer_state_from $old_canv
	}

        set currPage_ $page_id
        set nextPage_ $page_id

	#FIXME: would be nice if the observer API works for this
	[$mgr_ sender] switch_page $currPage_
	$self notify_observers switch_page $currPage_
}

#
# searches thru all currently defined pages to get new number for
# the next page id
#       the page formats are <addr>_<uid>:<number>, all in hex
#
MBPageMgr public nextPageNumber {} {
        $self instvar arCanv2Page_ mgr_

        set part [$mgr_ local_srcid]
	set pages [array names arCanv2Page_]
        mtrace trcVerbose "pages:$pages, part=$part"
        set max 0
        foreach id $pages {
                if {[string first $part $id]>=0} {
                        mtrace trcExcessive \
					[concat "match $id, split returns <" \
                                        [split $id :] ">"]
                        set n "0x"
                        append n [lindex [split $id :] 1]
                        if {$n > $max} {
                                set max $n
                        }
                }
        }
        return [expr $max + 1]
}

MBPageMgr public page_label {page_id} {
        set wrk [split $page_id :]
        set srcid [lindex $wrk 0]
        # change to hex
        set h "0x"
        append h [lindex $wrk 1]
	if {$h == "0x"} {
		return "(null)"
	}
        set pagenum [format "%d" $h]
        set wrk [split $srcid _]

        set addr [lindex $wrk 0]
        set uid [lindex $wrk 1]

        set agent [[[$self set mgr_] session] get_agent]
        set src [$agent get_source $addr $uid]
        # use conventional userid@addr:page number
	$self instvar mgr_
        if {$src=={}} {
                # these values are in hex, change them to base 10 first
                set addr [format %u "0x$addr"]
                set uid [format %u "0x$uid"]
                set addr [$mgr_ intoa $addr]
                return [append nothing $uid "@" $addr ":Page " $pagenum]
        }
        return [append nothing [$src cname] ": " $pagenum]
}

# sorts page ids by host first, then page number
MBPageMgr proc sort_pages {pagelist} {
	# compares 2 mb page ids, src id first, then page number
	# note that this procedure is not visible in global space
	proc mbPageIdCompare {p1 p2} {
		puts "compare $p1 $p2"
		set t [split $p1 :]
		set h1 [lindex $t 0]
		set n1 [lindex $t 1]
		set t [split $p2 :]
		set h2 [lindex $t 0]
		set n2 [lindex $t 1]
		set r [string compare $h1 $h2]
		if {$r == 0} {
			return [expr {$n1 - $n2}]
		}
		puts "return $r"
		return $r
	}
	return [lsort -command mbPageIdCompare $pagelist]
}
