# dc-link.tcl --
#
#       Contains CLink and CLinkManager, basic building blocks of DC
#       Service Discovery Service.
#
# Copyright (c) 2000-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

Class CLinkManager
#-----------------------------------------------------------------------------
#
# Class CLinkManager
#
# Purpose:
#   Maintains a list of links between self and remote processes.
# Members:
#   m_socketServer - server socket of self
#   m_lLink - a list of CLink objects
#   m_lAttach - a list of class attached to this object so that 
#               they can receive notification whenever a new connection
#               arrive.  (Can't we use the Observable/Observer pattern?)
#
#-----------------------------------------------------------------------------


#-----------------------------------------------------------------------------
#
# CLinkManager constructor
#
# Input:
#   iPort - port to listen to for connection request
#
# Description:
#   Creates a socket and listen to client's request.  Initialize members
#   to empty.
#
#-----------------------------------------------------------------------------
CLinkManager public init { iPort }  {
    $self instvar m_socketServer
    $self instvar m_lLink

    set m_lLink ""

    if {[catch {socket -server "$self NewConnection" $iPort} m_socketServer]} {
	puts "LinkManager: Couldn't create server socket"
	exit
    }
}


#-----------------------------------------------------------------------------
#
# CLinkManager destructor
#
# Input:
#   none
#
# Description:
#   close the server, and delete each links
#
#-----------------------------------------------------------------------------
CLinkManager public destroy { } {
    $self instvar m_socketServer
    $self instvar m_lLink

    if {[catch {close $m_socketServer} err]} {
	puts "CLinkManager: Unable to close socket."
    }

    if {$m_lLink != ""} {
	foreach link $m_lLink {
	    delete $link
	}
    }
}

CLinkManager public Attach { objAttach } {
    $self instvar m_lAttach

    lappend m_lAttach $objAttach
}


#-----------------------------------------------------------------------------
#
# CLinkManager NewConnection
#
# Input:
#   newSocket, inetAddr, iPort - socket, address and port number of 
#   the new connection.
#
# Description:
#   This is the callback that is executed when a client connects to 
#   this server.
#
#-----------------------------------------------------------------------------
CLinkManager private NewConnection { newSocket inetAddr iPort } {
    $self instvar m_lLink m_lAttach

    # create the new link with the socket
    set link [new CLink $newSocket $inetAddr $iPort]
    foreach a $m_lAttach {
	set result [$a NewConnection $link]
	if {$result != 0} {
	    break
	}
    }

    # stick it into a list so that we can cleanup afterwards
    lappend m_lLink $link
}

#-----------------------------------------------------------------------------
#
#    I don't think NewLink is called anywhere.. - weitsang
#
#-----------------------------------------------------------------------------
#CLinkManager instproc NewLink { inetRemoteAddr iRemotePort } {
#    set sock [socket $inetRemoteAddr $iRemotePort]
#
#    if { $sock <= 0 } {
#	puts "CLinkManager: NewLink couldn't connect socket to host"
#	return 0
#    }
#
#    set link [new CLink $sock $inetRemoteAddr $iRemotePort]
#
#    return $link
#}

#-----------------------------------------------------------------------------
#
# CLinkManager GetSpec
#
# Description:
#   Returns a textual information about the current socket.  Return
#   string starts with the local address and is followed by socket 
#   information.
#
#-----------------------------------------------------------------------------
CLinkManager instproc GetSpec {} {
    $self instvar m_socketServer

    if {[catch {fconfigure $m_socketServer -sockname} socketInfo]} {
	return "[localaddr]: no socket info"
    }
    return "[localaddr] [lindex $socketInfo 2]"
}


#-----------------------------------------------------------------------------
#
# Class CLink 
#
#   CLink provides an abstraction for a connection between the client 
#   and a service.
# 
#   Members-
#     m_socket - socket for this connection
#     m_aMessageMap - an array of object/methods pair indexed by 
#                     messages. When a message is received, the 
#                     corresponding method is executed.
#     m_bufferRead  - buffer of messages for reading
#
#-----------------------------------------------------------------------------
Class CLink


#-----------------------------------------------------------------------------
#
# CLink constructor
#
# Input:
#   socket - socket for this connection
#
#-----------------------------------------------------------------------------
CLink private init { sock } {
    $self instvar m_socket
    $self instvar m_aMessageMap
    $self instvar m_bufferRead

    set m_bufferRead ""

    set m_socket $sock

    # set the socket to be non-blocking
    if {[catch {fconfigure $m_socket -blocking 0}]} {
	puts "CLink: Unable to configure socket"
	return
    }

    # handle the event when the socket has something to read
    if {[catch {fileevent $m_socket readable "$self Receive"}]} {
	puts "CLink: fileevent readable failed"
    }
}


#-----------------------------------------------------------------------------
#
# CLink destructor
#
# Purpose:
#   close the socket association with this link
#
#-----------------------------------------------------------------------------
CLink public destroy { } {
    $self CloseLink
}


#-----------------------------------------------------------------------------
#
# CLink MapMessage
#
# Input:
#   strMessage, obj, fnMethod
# 
# Purpose:
#   Add the method $fnMethod of object $obj to the list of callbacks to
#   run when the message $strMessage is received.
#
#-----------------------------------------------------------------------------
CLink public MapMessage { strMessage obj fnMethod } {
    $self instvar m_aMessageMap

    # the the current list of
    if { ![info exist m_aMessageMap($strMessage)] } {
	set m_aMessageMap($strMessage) ""
    } 

    # now add the new map in
    append m_aMessageMap($strMessage) " " [concat $obj $fnMethod]
}


#-----------------------------------------------------------------------------
#
# CLink UnmapMessage
#
# Input:
#   strMessage, obj, fnMethod
# 
# Purpose:
#   Remove the method $fnMethod of object $obj from the list of callbacks to
#   run when the message $strMessage is received.
#
#-----------------------------------------------------------------------------
CLink public UnmapMessage { strMessage obj fnMethod } {
    $self instvar m_aMessageMap

    if { [info exist m_aMessageMap($strMessage)] } {
	set map $m_aMessageMap($strMessage)
    } else {
	return
    }

    # now the delete map in the map
    set newMap ""
    foreach { objMap fnMethodMap } $map {
	if { $objMap != $obj || $fnMethodMap != $fnMethod } {
	    lappend newMap $objMap $fnMethod
	}
    }

    set m_aMessageMap($strMessage) $newMap
}


#-----------------------------------------------------------------------------
#
# CLink SocketRead
#
# Purpose:
#   Fileevent readable callback for the connection to the service.   
#
#-----------------------------------------------------------------------------
CLink private Receive { } {
    $self instvar m_socket
    $self instvar m_bufferRead

    # check that the socket is ok
    if { [catch {eof $m_socket} status] } {
	puts "CLink: Unable to check socket's status, closing all links."
	$self CloseLink
	return
    }
    if {$status == 1} {
	$self CloseLink
	return
    }

    if {[catch {gets $m_socket line} n]} { 
	# If we get an exception reading from the socket,
	# reset and give up
	puts "CLink: Unable to read a line from socket"
	return
    }
    while { $n > 0 } {
	# check if the line is the end of message signal
	if { $line == "__END_OF_MESSAGE" } {
	    # if so then process the buffer
	    $self ProcessMessage $m_bufferRead
	    set m_bufferRead ""
	} else {
	    # else it's part of a message to add it to the read buffer
	    set m_bufferRead "$m_bufferRead \n$line"
	}
	# read the next line
	if {[catch {gets $m_socket line} n]} { 
	    # If we get an exception reading from the socket,
	    # reset and give up
	    puts "CLink: Unable to read a line from socket. Giving up."
	    set m_bufferRead ""
	    return
	}
    }
}

CLink private ProcessMessage { cmd } {
    $self instvar m_aMessageMap

    # get the first word of the cmd and use the message map to see who
    # to send it to

    set message [lindex $cmd 0]
    set arguments [lrange $cmd 1 end]

    # if the message isn't mapped then just return
    if { ![info exists m_aMessageMap($message)] } {
	return 0
    }

    set lInvoke $m_aMessageMap($message)
    foreach { obj fnMethod } $lInvoke {
	eval "$obj $fnMethod $self $arguments"
    }
}

CLink public Send { message args } {
    $self instvar m_socket

    # for now just block on write
    if {[catch {puts $m_socket "$message $args \n__END_OF_MESSAGE"} err]} {
	puts "CLink: Unable to write to socket"
    }
    if {[catch {flush $m_socket} err]} {
	puts "CLink: Unable to flush socket"
    }	
}


CLink public CloseLink { } {
    $self instvar m_socket

    if {[catch {close $m_socket} err]} {
	puts "CLink: Unable to close socket"
    }

    # notify the user that we've closed the socket
    $self ProcessMessage {CLOSE_LINK {}}
}
