/*
 * cpuinfo.cc --
 *
 *      Provides information about the processor
 *
 * Copyright (c) 2001 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 *  $Header: /usr/mash/src/repository/mash/mash-1/misc/cpuinfo.cc,v 1.2 2002/01/03 17:57:51 lim Exp $
 */

#include"cpuinfo.h"
#include<iostream.h>
#ifdef WIN32
#include<excpt.h>
#endif

// #define ALLOW_UNIX_EXCEPTIONS

bool supportsMMX()// returns true if MMX supported, false otherwise
{
    unsigned int a=CpuInfo::get();
    return ((a&FLAG_MMX)?true:false);
}

bool supportsSIMD() // returns true if SIMD can be used (hardware&OS), false otherwise
{
    unsigned int a=CpuInfo::get();
    return (((a&FLAG_SIMD)&&(a&FLAG_SIMD_OS))?true:false);
}

#ifdef TEST_EXECUTE
int textexec();
#endif

CpuInfo cpuinfo();
bool CpuInfo::configured=false;
unsigned int CpuInfo::flags=0;

void getCpuInfo(unsigned int&);

CpuInfo::CpuInfo() { //: TclClass( (const char *)"CpuInfo") {
    name("CpuInfo");
    if(!configured) {
	getCpuInfo(flags);	
	configured=true;
#ifdef TEST_EXECUTE 
	textexec();
#endif
    }
}

unsigned int CpuInfo::get() { 
    if(configured) {
        return flags;
    }
    else {
  		getCpuInfo(flags);	
		configured=true;
#ifdef TEST_EXECUTE 
		textexec();
#endif
		return flags;
	}
}

int CpuInfo::command(int argc, const char*const* argv)
{
    if (argc == 2) {
	Tcl& tcl = Tcl::instance();
	char* cp = tcl.buffer();
	if (strcmp(argv[1], "mmx") == 0 ) {
	    if( (flags&FLAG_MMX) ) {
		strcpy(cp, "true");
	    }
	    else {
		strcpy(cp, "false");
	    }
	}
	else if( strcmp(argv[1], "simd") == 0 ) {
	    if( ((flags&FLAG_SIMD)&&(flags&FLAG_SIMD_OS)) ) {
		strcpy(cp, "true");
	    }
	    else {
		strcpy(cp, "false");
	    }		    
	}
	else if( strcmp(argv[1], "processor") == 0 ) {
	    if( flags&FLAG_INTEL ) {
		strcpy(cp, "intel");
	    }
	    else {
		strcpy(cp, "non-intel");
	    }
	}
	else return (TclObject::command(argc, argv));
	tcl.result(cp);
	return (TCL_OK);
    }
    else return (TclObject::command(argc, argv));
}

#ifdef X86_ASSEMBLER
// this function can only really detect 486 and later processors
void getCpuInfo(unsigned int&flags) 
{
    flags=0;
    int a=-1;
    int line1=-1;
    int line2=-1;
    int line3=-1;
#ifdef WIN32
    _try {
	__asm {
	    mov eax, 0
	    cpuid
	    mov [a], eax
	    mov [line1], ebx
	    mov [line2], ecx
	    mov [line3], edx
	}
    }
    _except( EXCEPTION_EXECUTE_HANDLER ) {
	cout << "exception cpuid code is " << _exception_code() << endl;
	return;
    }
#else

#ifdef ALLOW_UNIX_EXCEPTIONS
    try {
#endif
	a=0;
	__asm__ __volatile__(
    	    "cpuid"
	    : "=a" (a), "=b" (line1), "=c" (line2), "=d" (line3)
	    : "a" (a) );
#ifdef ALLOW_UNIX_EXCEPTIONS
    }
    catch( ... ) { // exception e // cout << e.what() << endl;
	cout << "caught cpuid exception" << endl;
	return;
    }
#endif

#endif

    // "GenuineIntel" ==> 0x47656e75696e65496e74656c
    if(line1!=0x756e6547 || line2!=0x6c65746e || line3!=0x49656e69) {
	return;
    }
    flags|=FLAG_INTEL; // received intel chip
    
    if(a<1) {
	return; 	// can't check family or MMX bit
    }
    
#ifdef WIN32
    __asm {
	mov eax, 1
	cpuid
	mov [a], eax	// processor type, family, model, stepping [res,type,fam,mod,stepping]
	mov [line1], edx // feature flags (intel table 5)
    }
#else
    a=1;
	__asm__ __volatile__(
    	    "cpuid"
	    : "=a" (a), "=d" (line1)
	    : "a" (a) );
#endif
    // highest 18 bits undefined and reserved
    // processor type is bits 12-13
    // family is 8-11
    // model 4-7
    // stepping 0-3
    if( /*family*/ ((a>>8)&0xf)==4 /*0100*/ ) {
	int model=((a>>4)&0xf);
	switch(model) {
	case 0:	case 1: // 486DX
	case 2: // 486SX 
	case 3: // 487, 486DX2
	case 4: // 486SL
	case 5: // 486SX2
	case 7: // 486DX2 write-back enhanced
	case 8: // 486DX4
	default: // unknown
	    flags|=FLAG_I486; 
	    break;
	}
    }
    else if( /*family*/ ((a>>8)&0xf)==5 /*0101*/ ) {
	int model=((a>>4)&0xf);
	switch(model) {
	case 1: // Pentium 60,66Mhz (may be overdrive 60,66)
	case 2: // Pentium 75-200Mhz (may be overdrive 75-133)
	case 3: // Pentium Overdrive for 486-based systems
	case 4: // Pentium with MMX (166,200Mhz) (may be overdrive 75-133)
	default: // unknown
	    flags|=FLAG_I586; 
	    break;
	}
    }
    else if( /*family*/ ((a>>8)&0xf)==6 /*0110*/) {
	int model=((a>>4)&0xf);
	switch(model) {
	case 1: // Pentium Pro
	case 3: // Pentium II model 3 (or Pentium Pro overdrive)
	case 5: // Pentium II model 5 or Celeron
	default:
	    flags|=FLAG_I686; 
	    break;
	}
    }
    else {
	// unknown intel family
	if( ((a>>8)&0xf)>6 ) { // assuming newer families have 686 capabilities
	    flags|=FLAG_I686; 
	}
    }
    if( (line1>>23)&0x1 ) { // bit 23 is mmx
	flags|=FLAG_MMX;
    }
    if( (line1>>25)&0x1 ) { // bit 25 is simd
	flags|=FLAG_SIMD;
#ifndef OLD_ASM
	// if assembler supports simd instructions, then try executing an operation
#ifdef WIN32
        _try {
		__asm xorps xmm0, xmm0; Streaming SIMD Extension
	}
	_except( EXCEPTION_EXECUTE_HANDLER ) {
	    cout << "simd exception code is " << _exception_code() << endl;
	    return;
        }
#else
	// using 'a' just to keep the compiler happy
#ifdef ALLOW_UNIX_EXCEPTIONS
	try {
#endif
	    __asm__ __volatile__(
    		"xorps %%xmm0, %%xmm0"
		: "=a" (a)
		: "a" (a) );
#ifdef ALLOW_UNIX_EXCEPTIONS
	}
	catch(...) {
	    cout << "caught xorps exception" << endl;
	    return;
	}
#endif
#endif
	flags|=FLAG_SIMD_OS;
#endif
    }
}
#else
void getCpuInfo(unsigned int&flags) 
{
    flags=0;
}
#endif

#if defined(TEST_MAIN) || defined(TEST_EXECUTE)

#ifdef TEST_MAIN
int main(int argc, char*argv[])
#else
int textexec()
#endif
{
    unsigned int a=CpuInfo::get();
    cout << ((a&FLAG_INTEL)?"intel chip":"non-intel chip") << endl;
    cout << (((a&FLAG_FAMILY_MASK)==FLAG_I086)?"086 chip":"non-086 chip") << endl;
    cout << (((a&FLAG_FAMILY_MASK)==FLAG_I286)?"286 chip":"non-286 chip") << endl;
    cout << (((a&FLAG_FAMILY_MASK)==FLAG_I386)?"386 chip":"non-386 chip") << endl;
    cout << (((a&FLAG_FAMILY_MASK)==FLAG_I486)?"486 chip":"non-486 chip") << endl;
    cout << (((a&FLAG_FAMILY_MASK)==FLAG_I586)?"586 chip":"non-586 chip") << endl;
    cout << (((a&FLAG_FAMILY_MASK)==FLAG_I686)?"686 chip":"non-686 chip") << endl;
    cout << ((a&FLAG_MMX)?"MMX chip":"non-MMX chip") << endl;
    cout << ((a&FLAG_SIMD)?"SIMD chip":"non-SIMD chip") << endl;
    cout << ((a&FLAG_SIMD_OS)?"SIMD OS":"non-SIMD OS") << endl;

    cout << "MMX supported=" << supportsMMX() << endl;
    cout << "SIMD supported=" << supportsSIMD() << endl;
    return 0;
}
#endif

CpuInfoClass cic;
