/*
 * mb-items.cc --
 *
 *      MediaBoard items
 *
 * Copyright (c) 1996-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * $Header: /usr/mash/src/repository/mash/mash-1/mb/mb-items.cc,v 1.15 2002/02/03 03:16:30 lim Exp $
 */

#include "tclcl.h"
#include "mb/mb-items.h"
#include "mb/mb-canv.h"

extern "C"
{
#include <tk.h>
};


class LineItem;
class PageItem;

/* static */
PageItem* PageItem::createItem(PageItemType type)
{
	switch (type) {

	case PgItemLine :
	case PgItemMLine:
		return new LineItem(type);
		break;

	case PgItemRect:
	case PgItemOval:
		return new PolyItem(type);
		break;

	case PgItemText:
		return new TextItem();
		break;

	case PgItemImage:
		return new ImageItem();
		break;

	case PgItemPS:
		return new PSItem();
		break;

	default:
		Trace(ALL,("Attempting to create invalid item!"));
		assert(FALSE);
		return NULL;
	} // end switch
}

/* static member function */
PageItem* PageItem::create(PageItemType type, Bool inclCoords, Byte* pb)
{
	assert( net2host(host2net((float)-24.33)) == (float)-24.33 );
	PageItem* pNewItem = createItem(type);
	int numPoints=0;
	Point *aPoints = NULL;
	if (inclCoords) {
		Pkt_Coord *pPkt = (Pkt_Coord*) pb;
		numPoints=net2host(pPkt->nPoints);
		aPoints = new Point[numPoints];
		for (int i=0; i<numPoints; i++) {
			aPoints[i].x = (double)net2host(pPkt->aPoints[i].x);
			aPoints[i].y = (double)net2host(pPkt->aPoints[i].y);
		}
		pNewItem->setPoints(aPoints, numPoints);
		pNewItem->extract(pb+sizeof(Pkt_Coord)+
				  (numPoints*sizeof(Pkt_Point)));
	} else {
		pNewItem->extract(pb);
	}
	return pNewItem;
}

/*virtual*/
Byte* PageItem::packetize(Bool inclCoords, Byte *pb)
{
	if (!inclCoords) return pb; // nothing in here

	Pkt_Coord *pPkt = (Pkt_Coord*) pb;
	pPkt->nPoints=host2net(numPoints_);
	for (int i=0; i<numPoints_; i++) {
		pPkt->aPoints[i].x = host2net((float)aPoints_[i].x);
		pPkt->aPoints[i].y = host2net((float)aPoints_[i].y);
	}
	return pb+sizeof(Pkt_Coord)+numPoints_*sizeof(Pkt_Point);
}

/*virtual*/
u_int PageItem::getPacketLen(Bool inclCoords) {
	if (inclCoords) return sizeof(Pkt_Coord)+sizeof(Pkt_Point)*numPoints_;
	else return 0;
}

/*
 *----------------------------------------------------------------------
 *
 * PageItem::formatConfigValue -- (stolen and modified from Tcl)
 *
 *	This procedure formats the current value of a configuration
 *	option, the result is suitable for calling into the tkCanvas.
 *
 * note: caller must free the string if free is set to true
 *       as far as possible, it will point the result string to
 *       the value in the item ==> value return will persistent until
 *       the content of 'this' is changed.
 *
 *       we passed in the canvas so that it can be used to transform
 *       fonts into named fonts
 *
 *----------------------------------------------------------------------
 */
//
// Note: use Tcl_Free to free the result!
//

/*
 * Returns a string representation of the item, note that fonts
 *    are returned as named fonts rather than the generic font names
 *    so this won't work when we have multiple canvases for a page.
 */


const char* caPageItemTypeNames[] = {
	"none",	"line",	"rectangle", "oval", "mline", "text", "image", "pscript",
	"invalid"
};

const char* caPgItemTyp2CanvItemTyp[] = {
	"none",	"line",	"rectangle", "oval", "line", "text", "image", "pscript",
	"invalid"
};

// gives the number of points to be stored in each item
const int caPIT2nPts[] = {
	0, 2, 2, 2, 2, 1, 1, 2,
	0
};

void LineItem::getCopy(PageItem** ppResultItem) const
{
	LineItem* pItem = new LineItem(getType());
	PageItem::getCopy((PageItem**)&pItem);
	pItem->width_ = width_;
	pItem->arrow_ = arrow_;
	if (fg_) {
		pItem->fg_ = MB_GetColorByValue(fg_);
        } else
		pItem->fg_ = NULL;
	*ppResultItem = pItem;
}

// virtual functions from PageItem
void LineItem::extract(Byte* pb) {
	Pkt_LineItem* pPkt = (Pkt_LineItem*)pb;
	width_ = net2host(pPkt->width_);
	arrow_ = (ArrowType) net2host(pPkt->arrow_);
	net2host(pPkt->fg_, fg_);
}

Byte* LineItem::packetize(Bool inclCoords, Byte *pb) {
	Byte* pbCurr = PageItem::packetize(inclCoords, pb);
	Pkt_LineItem* pPkt=(Pkt_LineItem*)pbCurr;
	pPkt->width_ = host2net((u_int16_t)width_);
	pPkt->arrow_ = host2net((u_int16_t)arrow_);
	host2net(fg_, pPkt->fg_);
	return (Byte*)(pPkt+1);
}

void PolyItem::getCopy(PageItem **ppItem) const
{
	*ppItem = new PolyItem(getType());
	PageItem::getCopy(ppItem);
	PolyItem *pDerived=(PolyItem*)(*ppItem);
	pDerived->width_ = width_;

	// if non-null,
	// use Tk to get a copy of the XColor, so that
	// the Tk_FreeColor can be called safely in the destructor
	pDerived->fillC_ = (fillC_) ?
		MB_GetColorByValue(fillC_) : (XColor *) NULL;
	pDerived->outlineC_ = (outlineC_) ?
		MB_GetColorByValue(outlineC_) : (XColor *) NULL;
}

// virtual functions from PageItem
void PolyItem::extract(Byte* pb)
{
	Pkt_PolyItem* pPkt=(Pkt_PolyItem*)pb;
	width_ = net2host(pPkt->width_);
	assert(fillC_==NULL);
	net2host(pPkt->fill_, fillC_);
	net2host(pPkt->outline_, outlineC_);
}

Byte* PolyItem::packetize(Bool inclCoords, Byte *pb)
{
	Byte* pbCurr = PageItem::packetize(inclCoords, pb);
	Pkt_PolyItem* pPkt=(Pkt_PolyItem*)pbCurr;
	pPkt->width_ = host2net((u_int16_t)width_);
	host2net(fillC_, pPkt->fill_);
	host2net(outlineC_, pPkt->outline_);
	return (Byte*)(pPkt+1);
}

void TextItem::getCopy(PageItem **ppItem) const
{
	*ppItem = new TextItem();
	PageItem::getCopy(ppItem);
	TextItem *pDerived=(TextItem*)(*ppItem);

	// use Tk to get a copy of the XColor, so that
	// the Tk_FreeColor can be called safely in the destructor
	pDerived->fillC_ = (fillC_==NULL) ?
		(XColor *) NULL : MB_GetColorByValue(fillC_);
	// same thing with the font
	pDerived->font_ = (font_== (Tk_Font)NULL) ?
		((Tk_Font) NULL) :
		MB_GetFont(MB_NameOfFont(font_));
	if (szText_)
		::AllocNCopy(&pDerived->szText_, szText_);
	else pDerived->szText_ = NULL;
}

// virtual functions from PageItem
void TextItem::extract(Byte* pb)
{
	Pkt_TextItem* pPkt=(Pkt_TextItem*)pb;
	assert(fillC_==NULL);
	net2host(pPkt->fillC_, fillC_);
	char *szFont=NULL;
	u_short lenFont = net2host(pPkt->lenFont);
	if (lenFont) {
		::AllocNNCopy(&szFont, pPkt->achFontNText, lenFont);
		font_ = MB_GetFont(szFont);
		if (!font_) Trace(ALL, ("font %s not found!",szFont));
		delete[] szFont;
	}
	u_short lenText = net2host(pPkt->lenText);
	if (lenText) {
		assert(szText_==0);
		::AllocNNCopy(&szText_, pPkt->achFontNText+lenFont, lenText);
	}
}



Byte* TextItem::packetize(Bool inclCoords, Byte *pb)
{
	Byte* pbCurr = PageItem::packetize(inclCoords, pb);
	Pkt_TextItem* pPkt=(Pkt_TextItem*)pbCurr;
	host2net(fillC_, pPkt->fillC_);
	u_short lenFont=0,lenText=0;
	if (font_) {
		char *szFont = MB_NameOfFont(font_);
		lenFont = strlen(szFont);
		strncpy(pPkt->achFontNText,szFont,lenFont);
	}
	pPkt->lenFont = host2net(lenFont);
	if (szText_ && (lenText=strlen(szText_))) {
		strncpy(pPkt->achFontNText+lenFont,szText_,lenText);
	}
	pPkt->lenText = host2net(lenText);
	return pbCurr+ PKT_ROUNDUP(sizeof(Pkt_TextItem)+
				   sizeof(char)*(lenText+lenFont));
}

u_int TextItem::getPacketLen(Bool inclCoords)
{
	u_short len=0;
	if (font_) {
		char *szFont = MB_NameOfFont(font_);
		len += strlen(szFont);
	}
	if (szText_) len+=strlen(szText_);
	return PKT_ROUNDUP( sizeof(Pkt_TextItem)+len*sizeof(char)
			    + PageItem::getPacketLen(inclCoords) );
}

