/*
 * Copyright (c) 2017, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

/*
 * @test
 * @bug 6364894
 * @requires (os.family == "windows")
 * @library /test/lib
 * @build jdk.test.lib.Asserts
 * @run main FileOpenTest
 * @summary Test to ensure that opening of hidden Vs non-hidden,
 *          read/write Vs read-only files for writing works as expected.
 */

import java.io.File;
import java.io.IOException;
import java.io.FileOutputStream;
import java.nio.file.Files;

import static jdk.test.lib.Asserts.assertTrue;

public class FileOpenTest {

    private static File tmpFile;

    public static void main(String args[]) throws Exception {
        try {
            tmpFile = File.createTempFile("FileOpenTest", "suffix");

            // Opening Writable Normal File..
            test(true);

            // Opening Writable Hidden File..
            Files.setAttribute(tmpFile.toPath(), "dos:hidden", true);
            test(false);

            // Opening Read-Only Hidden File..
            Files.setAttribute(tmpFile.toPath(), "dos:hidden", false);
            tmpFile.setReadOnly();
            test(false);

            // Opening Read-Only Normal File..
            Files.setAttribute(tmpFile.toPath(), "dos:hidden", true);
            test(false);
        } finally {
            tmpFile.delete();
        }
    }

    private static void test(boolean writable) throws Exception {

        try (FileOutputStream fs = new FileOutputStream(tmpFile)) {
            fs.write(1);
            assertTrue(writable, "Able to open READ-ONLY file for WRITING!");
            assertTrue(tmpFile.canWrite(), "Able to open READ-ONLY file for WRITING!");
        } catch(IOException e) {
            assertTrue(!writable, "Unable to open non-READ-ONLY file for WRITING!");
            System.out.println("Caught the Exception as expected");
            e.printStackTrace(System.out);
        }
    }
}
