/*
 * Copyright (c) 2001, 2018, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

package nsk.jdi.ExceptionRequest.addClassFilter_s;

import nsk.share.*;
import nsk.share.jpda.*;
import nsk.share.jdi.*;

import com.sun.jdi.*;
import com.sun.jdi.event.*;
import com.sun.jdi.request.*;

import java.util.*;
import java.io.*;

/**
 * The test for the implementation of an object of the type
 * ExceptionRequest.
 *
 * The test checks that results of the method
 * <code>com.sun.jdi.ExceptionRequest.addClassFilter(String)</code>
 * complies with its spec.
 *
 * The test checks up on the following assertion:
 *    Restricts the events generated by this request to those
 *    whose location is in a class
 *    whose name matches a restricted regular expression.
 * The cases to check include both a pattern that begin with '*'
 * one that end with '*'.
 *
 * The test works as follows.
 * - The debugger
 *   - creates two class filter objects,
 *     first one  using a pattern that begin with '*', and
 *     second one using a pattern that end with '*'
 *     to filter ExceptionEvents in two debuggee's  tested threads,
 *   - sets up two ExceptionRequests,
 *   - resumes the debuggee, and
 *   - waits for expected ExceptionEvents.
 * - The debuggee creates and starts two threads
 *   whose 'run' methods create objects of ClassTypes needed
 *   to generate Events and to test the filters.
 * - Upon getting the events, the debugger performs checks required.
 */

public class filter_s001 extends TestDebuggerType1 {

    public static void main (String argv[]) {
        System.exit(run(argv, System.out) + Consts.JCK_STATUS_BASE);
    }

    public static int run (String argv[], PrintStream out) {
        debuggeeName = "nsk.jdi.ExceptionRequest.addClassFilter_s.filter_s001a";
        return new filter_s001().runThis(argv, out);
    }

    private String testedClassName1 = "*TestClass11";
    private String testedClassName2 =
        "nsk.jdi.ExceptionRequest.addClassFilter_s.filter_s001aThread2$filter_s001aTestClass20*";

    private String className1 = "TestClass11";
    private String className2 =
        "nsk.jdi.ExceptionRequest.addClassFilter_s.filter_s001aThread2$filter_s001aTestClass20";

    protected void testRun() {

        EventRequest  eventRequest1      = null;
        EventRequest  eventRequest2      = null;

        String        property1          = "ExceptionRequest1";
        String        property2          = "ExceptionRequest2";

        Event newEvent = null;

        for (int i = 0; ; i++) {

            if (!shouldRunAfterBreakpoint()) {
                vm.resume();
                break;
            }

            display(":::::: case: # " + i);

            switch (i) {

                case 0:
                eventRequest1 = setting22ExceptionRequest(null, testedClassName1,
                                             EventRequest.SUSPEND_ALL, property1);

                display("......waiting for ExceptionEvent in expected thread");
                newEvent = eventHandler.waitForRequestedEvent(new EventRequest[]{eventRequest1}, waitTime, true);

                if ( !(newEvent instanceof ExceptionEvent)) {
                    setFailedStatus("ERROR: new event is not ExceptionEvent");
                } else {
                    String str = ((ExceptionEvent)newEvent).location().declaringType().name();
                    if (!str.endsWith(className1)) {
                        setFailedStatus("Received ExceptionEvent for unexpected class: \n\t" + str);
                    } else {
                        display("Received ExceptionEvent for expected class: \n\t" + str);
                    }

                    String property = (String) newEvent.request().getProperty("number");
                    if ( !property.equals(property1) ) {
                        setFailedStatus("event has unexpected property: " + property1);
                    }
                }

                vm.resume();
                break;

                case 1:
                eventRequest2 = setting22ExceptionRequest(null, testedClassName2,
                                             EventRequest.SUSPEND_ALL, property2);

                display("......waiting for ExceptionEvent in expected thread");
                newEvent = eventHandler.waitForRequestedEvent(new EventRequest[]{eventRequest2}, waitTime, true);

                if ( !(newEvent instanceof ExceptionEvent)) {
                    setFailedStatus("ERROR: new event is not ExceptionEvent");
                } else {
                    String str = ((ExceptionEvent)newEvent).location().declaringType().name();
                    if (!str.endsWith(className2)) {
                        setFailedStatus("Received ExceptionEvent for unexpected class: \n\t" + str);
                    } else {
                        display("Received ExceptionEvent for expected class: \n\t" + str);
                    }

                    String property = (String) newEvent.request().getProperty("number");
                    if ( !property.equals(property2) ) {
                        setFailedStatus("event has unexpected property: " + property2);
                    }
                }
                vm.resume();
                break;

                default:
                throw new Failure("** default case 2 **");
            }
        }
        return;
    }

    private ExceptionRequest setting22ExceptionRequest ( ThreadReference thread,
                                                         String          testedClass,
                                                         int             suspendPolicy,
                                                         String          property       ) {
        try {
            display("......setting up ExceptionRequest:");
            display("       thread: " + thread + "; class: " + testedClass + "; property: " + property);

            ExceptionRequest
            excr = eventRManager.createExceptionRequest(null, true, true);
            excr.putProperty("number", property);
            if (thread != null)
                excr.addThreadFilter(thread);
            excr.addClassFilter(testedClass);
            excr.setSuspendPolicy(suspendPolicy);

            display("      ExceptionRequest has been set up");
            return excr;
        } catch ( Exception e ) {
            throw new Failure("** FAILURE to set up ExceptionRequest **");
        }
    }
}
