/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "Compiler_p.h"

#include <llvm/Module.h>
#include <llvm/PassManager.h>

#include "GTLCore/AST/Expression.h"
#include "GTLCore/AST/FunctionDeclaration.h"
#include "GTLCore/AST/Tree.h"
#include "GTLCore/CodeGenerator_p.h"
#include "GTLCore/ModuleData_p.h"
#include "GTLCore/ErrorMessage.h"
#include "GTLCore/Function.h"
#include "GTLCore/Optimiser_p.h"
#include "GTLCore/Type.h"
#include "GTLCore/Macros_p.h"
#include "GTLCore/Function_p.h"
#include "GTLCore/Value.h"

#include "Debug.h"
#include "Lexer_p.h"
#include "Library.h"
#include "LibrariesManager.h"
#include "Parser_p.h"

using namespace OpenShiva;

struct Compiler::Private {
  llvm::Module* module;
  GTLCore::ModuleData* moduleData;
  GTLCore::CodeGenerator* codeGenerator;
  Lexer* lexer;
  Parser* parser;
  int channelsNb;
  bool isKernel;
  bool isStdLibrary;
  std::map< GTLCore::String, GTLCore::Value > parameters;
};

Compiler::Compiler( bool _isKernel, int _channelsNb ) : d(new Private)
{
  d->module = 0;
  d->moduleData = 0;
  d->codeGenerator = 0;
  d->channelsNb = _channelsNb;
  d->isKernel = _isKernel;
}

Compiler::~Compiler()
{
  delete d;
}

bool Compiler::isKernel() const
{
  return d->isKernel;
}

bool Compiler::isStdLib() const
{
  return d->isStdLibrary;
}

bool Compiler::compile(bool _isStdLibrary, const GTLCore::String& _sourceCode, const GTLCore::String& _kernelName, GTLCore::ModuleData* _moduleData, GTLCore::String& _nameSpace, const std::map< GTLCore::String, GTLCore::Value >& parameters)
{
  d->isStdLibrary = _isStdLibrary;
  SHIVA_DEBUG("Compile: " << _kernelName << " : " << _sourceCode);
  // Initialise the module structure
  SHIVA_ASSERT( d->module == 0 );
  d->module = _moduleData->llvmModule();
  d->moduleData = _moduleData;
  d->codeGenerator = new GTLCore::CodeGenerator( d->moduleData );
  setModuleData(d->moduleData);
  d->parameters = parameters;

  // Create Standard Library functions
  // CtlStdLib functions (except print which get a special treatement)
  createStdLibFunction( "assert", "assert", GTLCore::Type::Void, 1, GTLCore::Type::Boolean);
  createStdLibFunction( "isnan_f", "isnan_f", GTLCore::Type::Boolean, 1, GTLCore::Type::Float);
  createStdLibFunction( "isnan_h", "isnan_h", GTLCore::Type::Boolean, 1, GTLCore::Type::Float);
  // C Math functions
  createStdLibFunction( "rand", "rand", GTLCore::Type::Integer32, 0);
  createStdLibFunction( "acos", "acosf", GTLCore::Type::Float, 1, GTLCore::Type::Float);
  createStdLibFunction( "asin", "asinf", GTLCore::Type::Float, 1, GTLCore::Type::Float);
  createStdLibFunction( "atan", "atanf", GTLCore::Type::Float, 1, GTLCore::Type::Float);
  createStdLibFunction( "atan2", "atan2f", GTLCore::Type::Float, 2, GTLCore::Type::Float, GTLCore::Type::Float);
  createStdLibFunction( "cos", "cosf", GTLCore::Type::Float, 1, GTLCore::Type::Float);
  createStdLibFunction( "sin", "sinf", GTLCore::Type::Float, 1, GTLCore::Type::Float);
  createStdLibFunction( "tan", "tanf", GTLCore::Type::Float, 1, GTLCore::Type::Float);
  createStdLibFunction( "cosh", "coshf", GTLCore::Type::Float, 1, GTLCore::Type::Float);
  createStdLibFunction( "sinh", "sinhf", GTLCore::Type::Float, 1, GTLCore::Type::Float);
  createStdLibFunction( "tanh", "tanhf", GTLCore::Type::Float, 1, GTLCore::Type::Float);
  createStdLibFunction( "exp", "expf", GTLCore::Type::Float, 1, GTLCore::Type::Float);
  createStdLibFunction( "exp_h", "expf", GTLCore::Type::Float, 1, GTLCore::Type::Float);
  createStdLibFunction( "log", "logf", GTLCore::Type::Float, 1, GTLCore::Type::Float);
  createStdLibFunction( "log_h", "logf", GTLCore::Type::Float, 1, GTLCore::Type::Float);
  createStdLibFunction( "log10", "log10f", GTLCore::Type::Float, 1, GTLCore::Type::Float);
  createStdLibFunction( "log10_h", "log10f", GTLCore::Type::Float, 1, GTLCore::Type::Float);
  createStdLibFunction( "pow", "powf", GTLCore::Type::Float, 2, GTLCore::Type::Float, GTLCore::Type::Float);
  createStdLibFunction( "pow_h", "powf", GTLCore::Type::Float, 2, GTLCore::Type::Float, GTLCore::Type::Float);
  createStdLibFunction( "sqrt", "sqrtf", GTLCore::Type::Float, 1, GTLCore::Type::Float);
  createStdLibFunction( "floor", "floorf", GTLCore::Type::Float, 1, GTLCore::Type::Float);
  createStdLibFunction( "fmod", "fmodf", GTLCore::Type::Float, 2, GTLCore::Type::Float, GTLCore::Type::Float);
  
  if( not d->isStdLibrary)
  { // of course you don't want to import the standard library when building the standard library
    importModule("shivastdlib");
  }

  // Init the lexer
  std::istringstream iss(_sourceCode);
  d->lexer = new Lexer( &iss );
  d->parser = new Parser( this, d->lexer );
  GTLCore::AST::Tree* tree = d->parser->parse();
  if( tree and errorMessages().size() == 0)
  {
    tree->generate( d->moduleData, d->codeGenerator );
    SHIVA_DEBUG( *d->module );
    for( std::list<GTLCore::AST::FunctionDeclaration*>::const_iterator it = tree->functionsDeclarations().begin();
         it != tree->functionsDeclarations().end(); ++it)
    {
      _moduleData->appendFunction( (*it)->function()->name(), (*it)->function());
    }
    // Success
    GTLCore::Optimiser::instance()->d->passManager()->run( *d->module );
  } else {
    // Failure
    SHIVA_DEBUG("failure " << (*errorMessages().begin()).line() << ": " << (*errorMessages().begin()).errorMessage());
//     delete d->module;
    d->module = 0;
  }
  _nameSpace = d->parser->nameSpace();
  // Clean up
  SHIVA_DEBUG("Clean up");
  
  delete tree;

/*  foreach(GTLCore::Function* function, functionsToDelete())
  {
    delete function;
  }*/
  SHIVA_DEBUG("Namespace = " << d->parser->nameSpace() );
  delete d->lexer;
  d->lexer = 0;
  delete d->parser;
  d->parser = 0;
  llvm::Module* mod = d->module;
  delete d->codeGenerator;
  d->codeGenerator = 0;
  d->module = 0;
  d->moduleData = 0;
  SHIVA_DEBUG("Compilation finished");
  return mod;
}

bool Compiler::importModule(const GTLCore::String& name)
{
  Library* library = LibrariesManager::instance()->loadLibrary( name, d->channelsNb );
  if( library )
  {
    if(not library->isCompiled())
    {
      library->compile();
      if(not library->isCompiled())
      {
        SHIVA_DEBUG("Compilation error: " << std::endl << library->compilationErrorsMessage() ); // TODO: report the error
        appendErrors( library->compilationErrors() );
        return false;
      }
    }
    d->moduleData->linkWith( library->data()->llvmModule() );
    // Append the function coming from the imported module
    std::list<GTLCore::Function*> functions = library->functions();
    foreach( GTLCore::Function* function, functions )
    {
      GTLCore::Function* newFunction = GTLCore::Function::Private::createExternalFunction( d->moduleData, function );
      declareFunction( newFunction->name(), newFunction );
      functionsToDelete().push_back( newFunction );
    }
  }
  return library;
}

GTLCore::TypesManager* Compiler::typesManager()
{
  return d->moduleData->typesManager();
}

GTLCore::AST::Expression* Compiler::standardConstant( const GTLCore::String& _name )
{
  // TODO standardConstant
  
/*  // Check if it is a parameter
  std::map< GTLCore::String, GTLCore::Value >::iterator it = d->parameters.find(_name);
  if( it != d->parameters.end() )
  {
    return GTLCore::AST::Expression::fromValue( it->second );
  }*/
  
  return 0;
}

const std::map< GTLCore::String, GTLCore::Value >& Compiler::parameters() const
{
  return d->parameters;
}
