/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "VirtualMachine_p.h"

#include "Function_p.h"
#include "Value.h"

#include "Debug.h"

// JIT
#include <llvm/ExecutionEngine/ExecutionEngine.h>
#include <llvm/Module.h>
#include <llvm/ModuleProvider.h>
#include <llvm/DerivedTypes.h>
#include <llvm/Target/TargetOptions.h>

#include "llvm/Support/CommandLine.h"

using namespace GTLCore;

struct VirtualMachine::Private {
  llvm::ExecutionEngine* executionEngine;
  static VirtualMachine* virtualMachine;
};

VirtualMachine* VirtualMachine::Private::virtualMachine = 0;

VirtualMachine::VirtualMachine() : d(new Private)
{
  d->executionEngine = 0;
#ifndef ENABLE_SIMD
  char** argv = new char*[2];
  argv[0] = const_cast<char*>( "GTLVM");
  argv[1] = const_cast<char*>( "-mattr=-3dnow,-3dnowa,-mmx,-sse,-sse2,-sse3,-ssse3" ); // WE ARE NOT AT A HACK MORE
  llvm::cl::ParseCommandLineOptions(2, argv, ""); // DISABLE MMX/SSE TODO check if more recent version of llvm support vectors in structure
#endif

  llvm::ExceptionHandling = true;

  GTLCore::String errorMessage;
  d->executionEngine = llvm::ExecutionEngine::create( new llvm::ExistingModuleProvider( new llvm::Module("dummy") ), false, &errorMessage);
  GTL_DEBUG("Error while creating execution engine : " << errorMessage);
}

VirtualMachine::~VirtualMachine()
{
  delete d->executionEngine;
  delete d;
}

void VirtualMachine::registerModule( llvm::ModuleProvider* mp)
{
  GTL_ASSERT(d->executionEngine);
  d->executionEngine->addModuleProvider(mp);
}

void VirtualMachine::unregisterModule( llvm::ModuleProvider* mp)
{
  if(d->executionEngine)
  {
    llvm::Module* module = mp->getModule();
    d->executionEngine->removeModuleProvider(mp);
    for( llvm::Module::FunctionListType::iterator it = module->getFunctionList().begin();
         it != module->getFunctionList().end(); ++it)
    {
      d->executionEngine->updateGlobalMapping( it, 0);
    }
    for( llvm::Module::GlobalListType::iterator it = module->getGlobalList().begin();
         it != module->getGlobalList().end(); ++it)
    {
      d->executionEngine->updateGlobalMapping( it, 0);
    }
  }
}

void *VirtualMachine::getPointerToFunction(llvm::Function *F)
{
  return d->executionEngine->getPointerToFunction(F);
}

void *VirtualMachine::getPointerToFunction(Function *F, int _nbParameters)
{
  return getPointerToFunction( F->d->data->function(_nbParameters ) );
}

void* VirtualMachine::getGlobalVariablePointer( llvm::GlobalVariable* GV)
{
    return d->executionEngine->getPointerToGlobal( GV );
}

#if 0
void VirtualMachine::setGlobalVariable( llvm::GlobalVariable *GV, const GTLCore::Value& value)
{
  void* ptr = d->executionEngine->getPointerToGlobal( GV );
  if( GV->getType()->getElementType() == llvm::Type::Int32Ty )
  {
    *(int*)ptr = value.asInt32();
  } else if( GV->getType()->getElementType() == llvm::Type::Int1Ty )
  {
    *(bool*)ptr = value.asBoolean();
  } else if( GV->getType()->getElementType() == llvm::Type::FloatTy )
  {
    GTL_DEBUG("Set " << value.asFloat() << " on ptr " << ptr << " from value = " << *(float*)ptr);
    *(float*)ptr = value.asFloat();
    GTL_DEBUG( *(float*)ptr );
  }
}

GTLCore::Value VirtualMachine::getGlobalVariable( llvm::GlobalVariable *GV)
{
  void* ptr = d->executionEngine->getPointerToGlobal( GV );
  if( GV->getType()->getElementType() == llvm::Type::Int32Ty )
  {
    return GTLCore::Value( *(int*)ptr);
  }
  if( GV->getType()->getElementType() == llvm::Type::Int1Ty )
  {
    return GTLCore::Value( *(bool*)ptr);
  }
  if( GV->getType()->getElementType() == llvm::Type::FloatTy )
  {
    return GTLCore::Value( *(float*)ptr);
  }
  return GTLCore::Value();
}
#endif

VirtualMachine* VirtualMachine::instance()
{
  if( not Private::virtualMachine)
  {
    Private::virtualMachine = new VirtualMachine;
  }
  return Private::virtualMachine;
}

llvm::ExecutionEngine* VirtualMachine::executionEngine()
{
  return d->executionEngine;
}
