/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "ModuleData_p.h"

#include <sstream>

#include "Debug.h"
#include "GTLCore/Function.h"
#include "TypesManager.h"
#include "Macros_p.h"
#include "Optimiser_p.h"

#include <llvm/Module.h>
#include <llvm/PassManager.h>
#include <llvm/Target/TargetData.h>
#include <llvm/Target/TargetMachine.h>
#include <llvm/Target/TargetMachineRegistry.h>
#include <llvm/Analysis/Verifier.h>
#include <llvm/Transforms/Utils/Cloning.h>
#include <llvm/Linker.h>
#include <llvm/Support/raw_ostream.h>

using namespace GTLCore;

ModuleData::ModuleData( llvm::Module* llvmModule) : m_llvmModule(llvmModule), m_llvmLinkedModule(0), m_typesManager(new TypesManager)
{
  
}

ModuleData::~ModuleData()
{
  for( std::map<ScopedName, Function*>::iterator it = m_functions.begin();
       it != m_functions.end(); ++it)
  {
    delete it->second;
  }
  delete m_llvmModule;
  delete m_typesManager;
}

bool ModuleData::appendFunction(const ScopedName& name, Function* function)
{
  GTL_DEBUG(name);
  if( m_functions.find(name) != m_functions.end())
  {
    return false;
  }
  m_functions[name] = function;
  return true;
}

Function* ModuleData::function(const String& _currentNameSpace, const String& _name)
{
  GTL_DEBUG( _currentNameSpace << "::" << _name );
  for( std::map<ScopedName, Function*>::iterator it = m_functions.begin();
       it != m_functions.end(); ++it)
  {
    GTL_DEBUG( it->first );
    if( (it->first.nameSpace() == "" or it->first.nameSpace() == _currentNameSpace )
        and it->first.name() == _name )
    {
      GTL_DEBUG( it->second );
      return it->second;
    }
  }
  return 0;
}

Function* ModuleData::function(const ScopedName& name)
{
  std::map<ScopedName, Function*>::iterator it = m_functions.find(name);
  if( it == m_functions.end())
  {
    return 0;
  } else {
    return it->second;
  }
}

std::list<Function*> ModuleData::functions()
{
  std::list<Function*> functions;
  GTL_DEBUG( m_functions.size());
  for( std::map<ScopedName, Function*>::iterator it = m_functions.begin();
       it != m_functions.end(); ++it)
  {
    functions.push_back(it->second);
  }
  return functions;
}

void ModuleData::linkWith( const GTLCore::String& _string)
{
  foreach( const GTLCore::String& mod, m_linkModuleWithArchives )
  {
    if( mod == _string ) return;
  }
  m_linkModuleWithArchives.push_back( _string );
  
}

void ModuleData::linkWith( const llvm::Module* _module )
{
  foreach( const llvm::Module* mod, m_linkModuleWith )
  {
    if( mod == _module ) return;
  }
  m_linkModuleWith.push_back( _module );
}

void ModuleData::doLink()
{
  GTL_ASSERT( not m_llvmLinkedModule );
  m_llvmLinkedModule = m_llvmModule;
  m_llvmModule = llvm::CloneModule( m_llvmModule );
  llvm::Linker linker("", m_llvmLinkedModule);
  GTLCore::String errorMessage;
  foreach( const llvm::Module* mod, m_linkModuleWith )
  {
    llvm::Module* clone = llvm::CloneModule( mod );
    linker.LinkInModule( clone, &errorMessage );
    GTL_DEBUG("Linking error: " << errorMessage );
    delete clone;
  }
  foreach( const GTLCore::String& mod, m_linkModuleWithArchives )
  {
    bool v = false;
    linker.LinkInArchive( llvm::sys::Path( mod), v);
  }
  
  linker.releaseModule();
  GTLCore::Optimiser::instance()->d->passManager()->run( *m_llvmModule );
}

