/*
 *  Copyright (c) 2009 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "MemoryManager_p.h"

#include <stdint.h>
#include <list>

#include "Debug.h"
#include "Macros_p.h"

using namespace GTLCore;

#define DEFAULT_SEGMENT_SIZE 100000

// #define DEBUG_MEMORY_MANAGER

#ifdef DEBUG_MEMORY_MANAGER

#define MM_DEBUG(msg) GTL_DEBUG(msg)

#else

#define MM_DEBUG(msg)

#endif


struct MemoryManager::Header {
  enum Status {
    USED, ///< Indicate the memory is still used
    FREED ///< Indicate the memory has been freed
  };
  Status status;
  int size;
  Header* previous;
  Header* next;
  uint8_t* data;
};

struct MemoryManager::Segment {
  inline Segment( int size )
  {
    size += sizeof(Header);
    start = new uint8_t[size];
    end = start + size * sizeof(uint8_t);
    nextFree = start;
    previous = 0;
  }
  ~Segment() {
    delete[] start;
  }
  inline bool contains( void* ptr ) const
  {
    return (ptr >= start and ptr < end)
        or ( ptr == end and (
          reinterpret_cast<Header*>(reinterpret_cast<uint8_t*>(ptr)-sizeof(Header))->size == 0 )  );
  }
  inline bool canContains(int size) const {
    return uint(end - nextFree) >= uint(size + sizeof(Header));
  }
  inline void* allocate(int size)
  {
    MM_DEBUG( "Begin allocate" );
    printStatus();
    GTL_ASSERT( canContains(size) );
    int sizeTotal = size + sizeof(Header);
    Header* nextFreeH = reinterpret_cast<Header*>(nextFree);
    nextFreeH->status = Header::USED;
    nextFreeH->size = size;
    nextFreeH->previous = previous;
    nextFreeH->next = 0;
    nextFreeH->data = nextFree + sizeof(Header);
    MM_DEBUG( nextFreeH << " with a previous of " << previous << " data " << (void*)nextFreeH->data << " and size of " << size );
    if(previous)
    {
      previous->next = nextFreeH;
    }
    previous = nextFreeH;
    nextFree += sizeTotal;
    printStatus();
    MM_DEBUG( "End allocate" );
    return nextFreeH->data;
  }
  inline void desallocate(void* ptr)
  {
    MM_DEBUG( "Begin desallocate" );
    printStatus();
    GTL_ASSERT(contains(ptr));
    uint8_t* ptr8 = reinterpret_cast<uint8_t*>(ptr);
    Header* currentHeader = reinterpret_cast<Header*>(ptr8-sizeof(Header));
    GTL_ASSERT(currentHeader->status == Header::USED);
    currentHeader->status = Header::FREED;
    if(currentHeader->next == 0)
    {
      while(true)
      {
        if(currentHeader->previous == 0)
        {
          MM_DEBUG("No previous, free = " << uint(end - nextFree) );
          nextFree = reinterpret_cast<uint8_t*>(currentHeader);
          previous = 0;
	  printStatus();
	  MM_DEBUG( "End desallocate" );
          return;
        }
        Header* previousCurrentHeader = currentHeader->previous;
        if( previousCurrentHeader->status == Header::FREED )
        {
          MM_DEBUG("Has free previous = " << previousCurrentHeader << " " << (void*)previousCurrentHeader->data );
          currentHeader = previousCurrentHeader;
          MM_DEBUG( currentHeader->next );
        } else {
          MM_DEBUG("Previous (" << previousCurrentHeader << ") isn't free = " << uint(end - nextFree) );
          previous = previousCurrentHeader;
          previous->next = 0;
          nextFree = reinterpret_cast<uint8_t*>(currentHeader);
	  MM_DEBUG("Now free" << uint(end - nextFree));
	  printStatus();
	  MM_DEBUG( "End desallocate" );
          return;
        }
      }
    }
    printStatus();
  }
  inline void printStatus()
  {
    MM_DEBUG( "### nextFree = " << (void*)nextFree << " previous = " << (void*)previous << " start = " << (void*)start << " end = " << (void*)end << " Free space = " << uint(end - nextFree) );
    if(previous)
    {
      MM_DEBUG( "### previous->status = " << previous->status << " previous->size = " << previous->size
	     << " previous->previous = " << previous->previous << " previous->next = " << previous->next
	     << " previous->data = " << (void*)previous->data );
    }
  }
  uint8_t* nextFree;
  Header* previous;
  uint8_t* start;
  uint8_t* end;
};

struct MemoryManager::Private {
  Private() : uniqueSegment(new Segment(DEFAULT_SEGMENT_SIZE)) {}
  Segment* uniqueSegment;
  std::list<Segment*> segments;
  static Private* s_instance;
};

MemoryManager::Private* MemoryManager::Private::s_instance = new MemoryManager::Private;

void* MemoryManager::allocate(int size)
{
  MM_DEBUG("Allocate: " << size);
  if( Private::s_instance->uniqueSegment )
  {
    if( Private::s_instance->uniqueSegment->canContains(size) )
    {
      void* ptr = Private::s_instance->uniqueSegment->allocate(size);
      MM_DEBUG("ptr = " << ptr);
      return ptr;
    } else {
      Private::s_instance->segments.push_back(Private::s_instance->uniqueSegment);
      Private::s_instance->uniqueSegment = 0;
    }
  } else {
    foreach(Segment* segment, Private::s_instance->segments)
    {
      if( segment->canContains(size) )
      {
        void* ptr = segment->allocate(size);
        GTL_ASSERT(segment->contains(ptr));
        MM_DEBUG("ptr = " << ptr);
        return ptr;
      }
    }
  }
  MM_DEBUG("Create new segment among " << Private::s_instance->segments.size() << " other segments" );
  int newSegSize = size > DEFAULT_SEGMENT_SIZE ? size : DEFAULT_SEGMENT_SIZE;
  Segment* segment = new Segment(newSegSize);
  Private::s_instance->segments.push_back(segment);
  void* ptr = segment->allocate(size);
  GTL_ASSERT(segment->contains(ptr));
  MM_DEBUG("ptr = " << ptr);
  return ptr;
}

void MemoryManager::desallocate(void* ptr)
{
  MM_DEBUG("Desallocate ptr = " << ptr);
  if( Private::s_instance->uniqueSegment )
  {
    GTL_ASSERT(Private::s_instance->uniqueSegment->contains(ptr));
    Private::s_instance->uniqueSegment->desallocate(ptr);
  } else {
    foreach(Segment* segment, Private::s_instance->segments)
    {
      if( segment->contains(ptr) )
      {
        segment->desallocate(ptr);
        return;
      }
    }
    GTL_ABORT("Not allocated pointer.");
  }
}
