/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * version 2 of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "Compiler_p.h"

#include <llvm/Module.h>
#include <llvm/PassManager.h>

#include "GTLCore/AST/Expression.h"
#include "GTLCore/AST/FunctionDeclaration.h"
#include "GTLCore/AST/Tree.h"
#include "GTLCore/CodeGenerator_p.h"
#include "GTLCore/ModuleData_p.h"
#include "GTLCore/ErrorMessage.h"
#include "GTLCore/Optimiser_p.h"

#include "Debug.h"
#include "Lexer_p.h"
#include "Parser_p.h"

using namespace OpenShiva;

struct Compiler::Private {
  llvm::Module* module;
  GTLCore::ModuleData* moduleData;
  GTLCore::CodeGenerator* codeGenerator;
  Lexer* lexer;
  Parser* parser;
};

Compiler::Compiler() : d(new Private)
{
  d->module = 0;
  d->moduleData = 0;
  d->codeGenerator = 0;
}

Compiler::~Compiler()
{
  delete d;
}

bool Compiler::compile(const GTLCore::String& _sourceCode, const GTLCore::String& _kernelName, GTLCore::ModuleData* _moduleData, GTLCore::String& _nameSpace)
{
  SHIVA_DEBUG("Compile: " << _kernelName << " : " << _sourceCode);
  // Initialise the module structure
  SHIVA_ASSERT( d->module == 0 );
  d->module = _moduleData->llvmModule();
  d->moduleData = _moduleData;
  d->codeGenerator = new GTLCore::CodeGenerator( d->moduleData );
  
  // Init the lexer
  std::istringstream iss(_sourceCode);
  d->lexer = new Lexer( &iss );
  d->parser = new Parser( this, d->lexer );
  GTLCore::AST::Tree* tree = d->parser->parse();
  if( tree and errorMessages().size() == 0)
  {
    tree->generate( d->moduleData, d->codeGenerator );
    for( std::list<GTLCore::AST::FunctionDeclaration*>::const_iterator it = tree->functionsDeclarations().begin();
         it != tree->functionsDeclarations().end(); ++it)
    {
      _moduleData->appendFunction( (*it)->function()->name(), (*it)->function());
    }
    // Success
    GTLCore::Optimiser::instance()->d->passManager()->run( *d->module );
  } else {
    // Failure
    SHIVA_DEBUG("failure " << (*errorMessages().begin()).line() << ": " << (*errorMessages().begin()).errorMessage());
//     delete d->module;
    d->module = 0;
  }
  _nameSpace = d->parser->nameSpace();
  // Clean up
  SHIVA_DEBUG("Clean up");
  
  delete tree;

/*  for(std::vector<GTLCore::Function*>::iterator it = d->functionsToDelete.begin();
      it != d->functionsToDelete.end(); ++it)
  {
    delete *it;
  }*/
  delete d->lexer;
  d->lexer = 0;
  delete d->parser;
  d->parser = 0;
  llvm::Module* mod = d->module;
  delete d->codeGenerator;
  d->codeGenerator = 0;
  d->module = 0;
  d->moduleData = 0;
  SHIVA_DEBUG("Compilation finished");
  return mod;
}

GTLCore::TypeManager* Compiler::typeManager()
{
  return d->moduleData->typeManager();
}

GTLCore::AST::Expression* Compiler::standardConstant( const GTLCore::String& _name )
{
  return 0;
}
