 /*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * version 2 of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#ifndef _GTLCore_TYPE_P_H_
#define _GTLCore_TYPE_P_H_

#include "Type.h"

namespace llvm {
  class Type;
}

namespace GTLCore {
  class Function;
  class Visitor;
  class Type::StructFunctionMember {
    public:
      /**
       * 
       * @param _name 
       * @param _symbolName the name of the symbol to call for executing this member,
       *                    the first parameter of the function will contains a pointer
       *                    to this structure
       * @param _type 
       * @param _parameters 
       */
      StructFunctionMember(Function* _function);
      StructFunctionMember(const StructFunctionMember& _rhs);
      StructFunctionMember& operator=(const StructFunctionMember& _rhs);
      ~StructFunctionMember();
      /**
        * @return the name of the member.
        */
      const GTLCore::String& name() const;
      /**
        * @return the type of the function.
        */
      const Type* returnType() const;
      /**
        * @return the list of arguments
        */
      const std::vector<Parameter>& parameters() const;
      
      /**
       * @return the function associated with this member
       */
      const Function* function() const;
    private:
      struct Private;
      Private* const d;
  };
  
  /**
   * @internal
   * Private information of the type object
   */
  struct Type::Private {
    friend class Type;
    public:
      Private() : m_type(0), structDataMembers(0), structFunctionMembers(0), structPrivateFunctionMembers(0), m_visitor(0) {}
      Private( Type::DataType _dataType );
      ~Private();
      const llvm::Type * type() const { return m_type; }
      /**
       * @return a pointer version of the type a.k.a. llvm::PointerType::get( type(), 0)
       */
      const llvm::Type * pointerType() const;
      /**
       * @return how the type will be used as function argument (usually return type(),
       *         except for ARRAY and STRUCTURE where it return pointerType() )
       */
      const llvm::Type * asArgumentType() const;
      void addFunctionMember( const StructFunctionMember& );
      void addPrivateFunctionMember( const StructFunctionMember& );
      int memberToIndex(const GTLCore::String&);
      /**
       * @return the function member associated with that name, or null if none
       */
      const Type::StructFunctionMember* functionMember( const GTLCore::String&) const;
      /**
       * @return the private function member associated with that name, or null if none
       */
      const Type::StructFunctionMember* privateFunctionMember( const GTLCore::String&) const;
      const Visitor* visitor() const;
      void setVisitor( const Visitor* );
      /**
       * This will create an undefined \ref Type associated with the llvm type given in
       * parameter.
       * This is used in some special case, for instance for function pointer in
       * OpenShiva's wrappers.
       */
      static Type* createArbitraryType( const llvm::Type* _type);
      /**
        * Select the type that has the higher priority
        */
      static const Type* selectType(const Type* type1, const Type* type2);
      /**
       * @return true if this a complex structure (a structure where one of its element
       *         is an array or an other structure)
       * 
       * @code
       * Type* type1 = new Type( struct { int a; int b; } );
       * Type* type2 = new Type( struct { int a; int b[2]; } );
       * Type* type3 = new Type( struct { int a; struct { int b; } } );
       * type1->d->isComplexStructure() == false;
       * type2->d->isComplexStructure() == true;
       * type3->d->isComplexStructure() == true;
       * @endcode
       */
      bool isComplexStructure();
      /**
       * @return true if this a "nested" array (ie int a[2][3]; is nested, while int a[2]; isn't)
       */
      bool isNestedArray();
      
    private:
      void setType( const llvm::Type* _type);
      const llvm::Type* m_type;
      DataType dataType;
      int vectorSize;
      const Type* arrayType;
      GTLCore::String structName;
      std::vector<StructDataMember>* structDataMembers;
      std::vector<Type::StructFunctionMember>* structFunctionMembers;
      std::vector<Type::StructFunctionMember>* structPrivateFunctionMembers;
      const Visitor* m_visitor;
  };
  
}

#endif
