/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * version 2 of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "ParserBase_p.h"

#include <map>

#include <llvm/Constants.h>
#include <llvm/DerivedTypes.h>
#include <llvm/Type.h>

#include "CodeGenerator_p.h"
#include "ErrorMessage.h"
#include "ExpressionResult_p.h"
#include "Type.h"
#include "Type_p.h"
#include "Value.h"

#include "AST/AccessorExpression.h"
#include "AST/BinaryExpression.h"
#include "AST/CoumpoundExpression.h"
#include "AST/Expression.h"
#include "AST/FunctionDeclaration.h"
#include "AST/Tree.h"
#include "AST/UnaryExpression.h"
#include "CompilerBase_p.h"
#include "ConvertCenter_p.h"
#include "LexerBase_p.h"
#include "Debug.h"
#include "Token_p.h"
#include "Function_p.h"
#include "Utils_p.h"

#include "TypeManager.h"
#include "TypeManager_p.h"

using namespace GTLCore;

struct ParserBase::Context
{
  std::map< ScopedName, VariableNG* > variables;
};

struct ParserBase::Private {
  Token currentToken;
  CompilerBase* compiler;
  LexerBase* lexer;
  String nameSpace;
  std::list< Context > contextes;
  std::list<VariableNG*> functionVariables;
};

ParserBase::ParserBase(CompilerBase* _compiler, LexerBase* _lexer) : d(new Private)
{
  d->compiler = _compiler;
  d->lexer = _lexer;
}

ParserBase::~ParserBase()
{
  delete d;
}

void ParserBase::getNextToken()
{
  d->currentToken = d->lexer->nextToken();
}

const Token& ParserBase::currentToken()
{
  return d->currentToken;
}

void ParserBase::parseStructDefinition()
{
  GTL_ASSERT( d->currentToken.type == Token::STRUCT );
  getNextToken();
  if( isOfType( d->currentToken, Token::IDENTIFIER ) )
  {
    String name = d->currentToken.string;
    getNextToken();
    if( d->compiler->typeManager()->d->isKnownType( name ) )
    { // It has allready been declared
      reportError( name + " has allready been declared", d->currentToken);
    } else {
      std::vector<Type::StructDataMember> members;
      if( isOfType( d->currentToken, Token::STARTBRACE ) )
      {
        getNextToken(); // Positionate on the first struct member
        while( true )
        { // Parse struct member
          if( isType(d->currentToken) )
          {
            const Type* smtype = parseType();
            GTL_ASSERT( smtype );
            if( isOfType( d->currentToken, Token::IDENTIFIER ) )
            {
              String smname = d->currentToken.string;
              getNextToken();
              // Check if it's an array
              std::list<int> memberArraySize = expressionsListToIntegersList( parseArraySize(true) );
              smtype = d->compiler->typeManager()->getArray( smtype, memberArraySize.size() );
              // Check if expression end by a SEMI and add the struct member
              if( isOfType( d->currentToken, Token::SEMI ) )
              {
                getNextToken();
                members.push_back( Type::StructDataMember( smname, smtype, memberArraySize ) );
              } else {
                return;
              }
            } else {
              return;
            }
          } else if( d->currentToken.type == Token::ENDBRACE )
          {
            getNextToken();
            break;
          } else {
            GTL_DEBUG("Unexpected");
            reportUnexpected( d->currentToken );
            return;
          }
        }
        if( isOfType( d->currentToken, Token::SEMI ) )
        {
          getNextToken();
          d->compiler->typeManager()->d->createStructure( name, members);
        }
      }
    }
  }
}

AST::AccessorExpression* ParserBase::parseMemberArrayExpression(AST::AccessorExpression* _expression, bool _constantExpression)
{
  if( d->currentToken.type == Token::DOT )
  {
    getNextToken();
    if( d->currentToken.type == Token::SIZE )
    {
      getNextToken(); // eat size
      return new AST::ArraySizeAccessorExpression( _expression );
    } else if( isOfType( d->currentToken, Token::IDENTIFIER ) ) {
      String name = d->currentToken.string;
      getNextToken();
      if( d->currentToken.type == Token::STARTBRACKET )
      {
        getNextToken(); // Eat the start bracket
        const Type::StructFunctionMember* sfm = _expression->type()->d->functionMember( name );
        if( not sfm )
        {
          reportError("Unknown member: '" + name + "' for structure " + _expression->type()->structName(), d->currentToken );
        } else {
          std::list<AST::Expression*> arguments = parseArguments( sfm->name(), sfm->parameters() );
          return new AST::FunctionMemberAccessorExpression(_expression, sfm, arguments);
        }
      } else {
        int index = _expression->type()->d->memberToIndex( name );
        if( index == -1 )
        {
          reportError("Unknown member: '" + name + "' for structure " + _expression->type()->structName(), d->currentToken );
          delete _expression;
          return 0;
        } else {
          return parseMemberArrayExpression( new AST::StructAccessorExpression( _expression ,  index ), _constantExpression);
        }
      }
    }
  } else if( d->currentToken.type == Token::STARTBOXBRACKET )
  {
    getNextToken();
    AST::Expression* expr = parseExpression( _constantExpression );
    if( isOfType( d->currentToken, Token::ENDBOXBRACKET ) )
    {
      getNextToken();
      return parseMemberArrayExpression( new AST::ArrayAccessorExpression( _expression , expr  ), _constantExpression);
    }
  } else {
    return _expression;
  }
  return 0;
}

std::list< AST::Expression* > ParserBase::parseArraySize(bool _constantExpression)
{
  std::list< AST::Expression* > list_;
  while( d->currentToken.type == Token::STARTBOXBRACKET )
  {
    getNextToken();
    if( d->currentToken.type == Token::ENDBOXBRACKET )
    {
      getNextToken();
      list_.push_back( 0 );
    } else {
      AST::Expression* expr = parseExpression( _constantExpression );
      if( expr and d->currentToken.type == Token::ENDBOXBRACKET )
      {
        getNextToken();
        list_.push_back( expr );
      } else {
        reportUnexpected( d->currentToken );
        list_.push_back( 0 );
      }
    }
  }
  return list_;
}

AST::CoumpoundExpression* ParserBase::parseCoumpoundExpression( const Type* _type, bool _constantExpression )
{
  GTL_ASSERT( _type );
  GTL_ASSERT( d->currentToken.type == Token::STARTBRACE );
  getNextToken(); // Eat '{'
  std::vector< AST::Expression* > expressions_;
  int index = 0;
  while(true)
  {
    AST::Expression* expression = 0;
    if( d->currentToken.type == Token::STARTBRACE )
    {
      const Type* type = 0;
      if ( _type->dataType() == Type::ARRAY )
      {
        type = _type->embeddedType();
      } else if( _type->dataType() == Type::STRUCTURE )
      {
        type = (*_type->structDataMembers())[index].type();
      }
      GTL_ASSERT( type );
      if( type->dataType() != Type::STRUCTURE and type->dataType() != Type::ARRAY )
      {
        reportUnexpected( d->currentToken );
        return 0;
      }
      expression = parseCoumpoundExpression( type, _constantExpression );
    } else {
      expression = parseExpression( _constantExpression );
    }
    expressions_.push_back( expression );
    if( d->currentToken.type == Token::COMA )
    {
      getNextToken();
    } else if( d->currentToken.type == Token::ENDBRACE )
    {
      getNextToken();
      return new AST::CoumpoundExpression(_type, expressions_);
    }
    ++index;
  }
}

AST::Expression* ParserBase::parseExpression(bool _constantExpression, const GTLCore::Type* _type)
{
  // The first token of an expression is either a primary, or an unary expression or a parenthesis
  if( d->currentToken.type == Token::STARTBRACKET )
  {
    getNextToken();
    AST::Expression* expr = parseExpression(_constantExpression);
    if( expr and isOfType( d->currentToken, Token::ENDBRACKET ) )
    {
      getNextToken();
      if( expr and d->currentToken.isBinaryOperator() )
      {
        expr = parseFlatBinaryOperator( _constantExpression, expr );
      }
    }
    return expr;
  } else if( d->currentToken.isPrimary() )
  {
    GTL_DEBUG( d->currentToken.isPrimary() << " " << d->currentToken);
    AST::Expression* exprConst = parsePrimaryExpression(_constantExpression);
    if( exprConst )
    { // What's next ? A binary operator or a ;
      if( d->currentToken.isExpressionTerminal() )
      {
        return exprConst;
      } else if( d->currentToken.isBinaryOperator() ) {
        return parseFlatBinaryOperator( _constantExpression, exprConst );
      } else {
        GTL_DEBUG("unexpected");
        reportUnexpected( d->currentToken );
        getNextToken();
        delete exprConst;
        return 0;
      }
    } else {
      // An error has occured
      reportError("Parse error while parsing constant", d->currentToken );
      return 0;
    }
  } else if( d->currentToken.isUnaryOperator() )
  {
    AST::Expression* expr = parseUnaryOperator(_constantExpression);
    // What's next ? An unary operator is either followed by a binary operator or a semi
    if( d->currentToken.isExpressionTerminal())
    {
      return expr;
    } else if( expr and d->currentToken.isBinaryOperator() ) {
      return parseFlatBinaryOperator( _constantExpression, expr );
    }
  } else if( currentToken().type == Token::STARTBRACE )
  {
    AST::CoumpoundExpression* coumpoundExpression = parseCoumpoundExpression( _type, true );
    if( coumpoundExpression )
    {
      return new AST::GlobalDataExpression( coumpoundExpression );
    }
  } else {
    GTL_DEBUG("unexpected");
    reportUnexpected( d->currentToken );
    return 0;
  }
  return 0;
}

#if 1
AST::Expression* ParserBase::parseFlatBinaryOperator( bool _constantExpression, AST::Expression* _lhs)
{
  GTL_ASSERT(_lhs);
  AST::Expression* current = _lhs;
  while( d->currentToken.isBinaryOperator() )
  {
    GTL_DEBUG( "Looping in parseFlatBinaryOperator");
    current = parseBinaryOperator( _constantExpression, current );
    if( not current ) // An error has occured
    {
      return 0;
    }
  }
  return current;
}

AST::Expression* ParserBase::parseBinaryOperator( bool _constantExpression, AST::Expression* _lhs)
{
  GTL_ASSERT(_lhs);
  Token binOp = d->currentToken;
  GTL_DEBUG( "parseBinaryOperator: " << binOp )
  getNextToken();
  AST::Expression* expr = 0;
  while(true)
  {
    GTL_DEBUG( binOp << " " << expr );
    // What's next ? A primary expression or an unary operator or a parenthesis
    if( expr == 0 )
    {
      if( d->currentToken.type == Token::STARTBRACKET )
      {
        getNextToken();
        expr = parseExpression(_constantExpression);
        if( not isOfType( d->currentToken, Token::ENDBRACKET ) )
        {
          delete expr;
          return 0;
        }
        getNextToken();
      } else if( d->currentToken.isPrimary() )
      {
        expr = parsePrimaryExpression(_constantExpression);
      } else if( d->currentToken.isUnaryOperator() ) {
        expr = parseUnaryOperator(_constantExpression);
      } else if( currentToken().type == Token::STARTBRACE ) {
        AST::CoumpoundExpression* coumpoundExpression = parseCoumpoundExpression( _lhs->type() , true );
        if( coumpoundExpression )
        {
          return new AST::GlobalDataExpression( coumpoundExpression );
        }
      } else {
        GTL_DEBUG("unexpected");
        reportUnexpected( d->currentToken );
        return 0;
      }
    }
    // What's next, either a semi or an other binary operator
    if( d->currentToken.isExpressionTerminal() ) {
      GTL_DEBUG( binOp << " terminal" );
      return createBinaryOperator( binOp, _lhs, expr);
    } else if( d->currentToken.isBinaryOperator() ) {
      GTL_DEBUG( binOp << " vs " << d->currentToken.isBinaryOperator());
      GTL_ASSERT( binOp.binaryOperationPriority() != -1 );
      GTL_ASSERT( d->currentToken.binaryOperationPriority() != -1 );
      
      if( d->currentToken.binaryOperationPriority() == binOp.binaryOperationPriority() )
      {
        GTL_DEBUG( binOp << " collapse _lhs and expr and keep going" );
        _lhs = createBinaryOperator( binOp, _lhs, expr );
        binOp = d->currentToken;
        getNextToken(); // eat the current operator
        expr = 0;
      } else if( d->currentToken.binaryOperationPriority() > binOp.binaryOperationPriority() )
      {
        GTL_DEBUG( binOp << " keep parsing to collapse expr with what is next" );
        expr = parseBinaryOperator( _constantExpression, expr );
        if( expr == 0 ) return 0;
        if( d->currentToken.isExpressionTerminal() )
        {
          return createBinaryOperator( binOp, _lhs, expr);
        }
      } else {
        GTL_DEBUG( binOp << " collapse _lhs and expr and return" );
        return createBinaryOperator( binOp, _lhs, expr );
      }
    } else {
      GTL_DEBUG("unexpected");
      reportUnexpected( d->currentToken );
      return 0;
    }
  }
}

#endif

#if 0

AST::Expression* ParserBase::parseFlatBinaryOperator( bool _constantExpression, AST::Expression* _lhs )
{
  AST::Expression* currentExpr = _lhs;
  while( true )
  {
    Token binOp = d->currentToken;
    AST::Expression* nextExpr = currentExpr;
    bool lhsUsed = false;
    do {
      nextExpr = parseBinaryOperator( _constantExpression, nextExpr, lhsUsed );
      if( not d->currentToken.isBinaryOperator() )
      {
        if( lhsUsed )
        {
          return nextExpr;
        } else {
          return createBinaryOperator( binOp, currentExpr, nextExpr );
        }
      }
      GTL_ASSERT( binOp.binaryOperationPriority() != -1 );
      GTL_ASSERT( d->currentToken.binaryOperationPriority() != -1 );
    } while( binOp.binaryOperationPriority() < d->currentToken.binaryOperationPriority() );
    currentExpr = createBinaryOperator( binOp, currentExpr, nextExpr );
  }
}

AST::Expression* ParserBase::parseBinaryOperator( bool _constantExpression, AST::Expression* lhs, bool& lhsUsed )
{
  GTL_ASSERT(lhs);
  Token binOp = d->currentToken;
  getNextToken();
  AST::Expression* expr = 0;
  // What's next ? A primary expression or an unary operator or a parenthesis
  if( d->currentToken.type == Token::STARTBRACKET )
  {
    getNextToken();
    expr = parseExpression(_constantExpression);
    if( not isOfType( d->currentToken, Token::ENDBRACKET ) )
    {
      delete expr;
      return 0;
    }
    getNextToken();
  } else if( d->currentToken.isPrimary() )
  {
    expr = parsePrimaryExpression(_constantExpression);
  } else if( d->currentToken.isUnaryOperator() ) {
    expr = parseUnaryOperator(_constantExpression);
  } else {
    GTL_DEBUG("unexpected");
    reportUnexpected( d->currentToken );
    return 0;
  }
  // What's next, either a semi or an other binary operator
  if( d->currentToken.isExpressionTerminal() ) {
    lhsUsed = true;
    return createBinaryOperator( binOp, lhs, expr);
  } else if( d->currentToken.isBinaryOperator() ) {
    GTL_ASSERT( binOp.binaryOperationPriority() != -1 );
    GTL_ASSERT( d->currentToken.binaryOperationPriority() != -1 );
    // Check which operator has the priority over the other one
    if( binOp.binaryOperationPriority() < d->currentToken.binaryOperationPriority())
    { // This operator has lowest priority, which means the result of the second operator will be the right hand side for this operator
//       AST::Expression* rhs = parseBinaryOperator( _constantExpression, expr );
//       return createBinaryOperator( binOp, lhs, rhs );
      lhsUsed = false;
      return expr;
    } else {
      // This operator has more priority, which means the result of this operator will be the left hand side for the second operator
      lhsUsed = true;
      bool v = false;
      AST::Expression* nlhs = createBinaryOperator( binOp, lhs, expr );
      return parseBinaryOperator( _constantExpression, nlhs, v );
    }
  } else {
    GTL_DEBUG("unexpected");
    reportUnexpected( d->currentToken );
    return 0;
  }
}

#endif

AST::Expression* ParserBase::parseUnaryOperator( bool _constantExpression )
{
  Token unaryTok = d->currentToken;
  GTL_ASSERT(unaryTok.isUnaryOperator() );
  getNextToken();
  // What's next ? An unary operator is either followed by a parenthesis or an other unary operator or a primary
  AST::Expression* expr = 0;
  if( d->currentToken.type == Token::STARTBRACKET ) {
    expr = parseExpression(_constantExpression);
    if( not isOfType( d->currentToken, Token::ENDBRACKET ) )
    {
      delete expr;
      return 0;
    }
    getNextToken();
  } else if( d->currentToken.isPrimary() )
  {
    expr = parsePrimaryExpression(_constantExpression);
  } else if( d->currentToken.isUnaryOperator() ) {
    expr = parseUnaryOperator(_constantExpression);
  } else {
    GTL_DEBUG("unexpected");
    reportUnexpected( d->currentToken );
    return 0;
  }
  switch( unaryTok.type )
  {
    case Token::MINUS:
      return new AST::MinusUnaryExpression( expr );
    case Token::MINUSMINUS:
    {
      AST::AccessorExpression* varexpr = dynamic_cast<AST::AccessorExpression*>( expr );
      if( varexpr  )
      {
        if( expr->type() == Type::Integer32 )
        {
          return new AST::MinusMinusUnaryExpression( varexpr );
        } else {
          reportError( "'--' operator works only on integer variable", unaryTok);
        }
      } else {
        reportError( "'--' operator requires a variable", unaryTok );
      }
    }
    case Token::PLUSPLUS:
    {
      AST::AccessorExpression* varexpr = dynamic_cast<AST::AccessorExpression*>( expr );
      if( varexpr  )
      {
        if( expr->type() == Type::Integer32 )
        {
          return new AST::PlusPlusUnaryExpression( varexpr );
        } else {
          reportError( "'++' operator works only on integer variable", unaryTok);
        }
      } else {
        reportError( "'++' operator requires a variable", unaryTok );
      }
    }
    case Token::NOT:
    {
      return new AST::NotUnaryExpression( expr );
    }
    case Token::TILDE:
    {
      if( expr->type() != Type::Integer32 )
      {
        reportError( "'~' operator only work with integer" , unaryTok );
        return 0;
      }
      return new AST::TildeUnaryExpression( expr );
    }
    default:
    {
      GTL_ASSERT( true );
      return 0;
    }
  }
}

void ParserBase::parseFunction()
{
  GTL_ASSERT( d->currentToken.isFunctionType() );
  // Parse the type
  const Type* returnType = parseFunctionType();
  if( not returnType ) return;
  
  // Clear the list of variables
  d->functionVariables.clear();
  
  // Next is the name of the function
  if( isOfType( d->currentToken, Token::IDENTIFIER ) )
  {
    Token nameToken = d->currentToken;
    ScopedName name( d->nameSpace, d->currentToken.string );
    getNextToken();
    // Next is start bracket
    if( isOfType( d->currentToken, Token::STARTBRACKET ) )
    {
      std::vector< AST::FunctionParameter* > params;
      bool needToHaveInitialiser = false;
      getNextToken();
      while(true)
      {
        // Next is either a type or a end bracket
        if( isType(d->currentToken) or d->currentToken.type == Token::OUTPUT or d->currentToken.type == Token::INPUT or d->currentToken.type == Token::VARYING )
        {
          bool output = false;
          bool varying = false;
          if( d->currentToken.type == Token::OUTPUT )
          {
            output = true;
            getNextToken();
          } else if( d->currentToken.type == Token::VARYING )
          {
            varying = true;
            getNextToken();
          }else if( d->currentToken.type == Token::INPUT )
          {
            getNextToken();
          }
          if( isType( d->currentToken ) )
          {
            const Type* ptype = parseType();
            if( not ptype )
            {
              return;
            }
            if(isOfType( d->currentToken, Token::IDENTIFIER ) )
            {
              String pname = d->currentToken.string;
              getNextToken();
              AST::Expression* initialiser = 0;
              std::list<AST::Expression*> memberArraySize = parseArraySize(true);
              ptype = d->compiler->typeManager()->getArray( ptype, memberArraySize.size() );
              deleteAll( memberArraySize);
              if( d->currentToken.type == Token::EQUAL )
              {
                getNextToken();
                initialiser = parseExpression( false, ptype );
                needToHaveInitialiser = true;
              } else if( needToHaveInitialiser )
              {
                reportError( "Parameter need to have default initialiser once a parameter on the left had one.", d->currentToken );
                return;
              }
              if( d->currentToken.type == Token::COMA or d->currentToken.type == Token::ENDBRACKET )
              {
                if( d->currentToken.type == Token::COMA )
                {
                  getNextToken();
                }
                Value initialiserValue;
                if( initialiser)
                {
                  CodeGenerator cg( 0 );
                  GenerationContext gc( &cg, 0, 0, 0);
                  llvm::Constant* constant = initialiser->generateValue( gc, 0 ).constant();
                  if( constant )
                  {
                    if( constant->getType() == llvm::Type::Int32Ty )
                    {
                      initialiserValue = (int)dynamic_cast<llvm::ConstantInt* >( constant )->getValue().getLimitedValue();
                    } else if ( constant->getType() == llvm::Type::FloatTy )
                    {
                      initialiserValue = dynamic_cast<llvm::ConstantFP* >( constant )->getValueAPF().convertToFloat();
                    } else if ( constant->getType() == llvm::Type::Int1Ty ) {
                      initialiserValue = (bool)(dynamic_cast<llvm::ConstantInt* >( constant )->getValue().getLimitedValue());
                    }
                  }
                }
                params.push_back( new AST::FunctionParameter( Parameter(pname, ptype, output, varying, initialiserValue), initialiser ) );
              } else {
                GTL_DEBUG("Unexpected");
                reportUnexpected( d->currentToken );
                return;
              }
            } else {
              GTL_DEBUG("Unexpected");
              reportUnexpected( d->currentToken );
              return;
            }
          } else {
            GTL_DEBUG("Unexpected");
            reportUnexpected( d->currentToken );
          }
        }
        else if( isOfType( d->currentToken, Token::ENDBRACKET ) )
        {
          getNextToken();
          if( isOfType( d->currentToken, Token::STARTBRACE ) )
          {
            AST::FunctionDeclaration* fd = new AST::FunctionDeclaration( name, returnType, params );
            startContext();
            // TODO insert functions arguments in the context
            for( unsigned int i = 0; i < params.size(); ++i)
            {
              d->contextes.begin()->variables[ ScopedName("",params[i]->parameter().name() )] = fd->parametersVariable()[i];
            }
            AST::Statement* statement = parseStatementList();
            endContext();
            if( statement )
            {
              bool v = d->compiler->declareFunction( name, fd->function() );
              if(v)
              {
                fd->setStatement( statement );
                tree()->append( fd );
              } else {
                delete fd;
                reportError( "Function '" + name.toString() + "' has allready been declared", nameToken);
              }
            }
            return;
          }
        } else {
          GTL_DEBUG("Unexpected");
          reportUnexpected( d->currentToken );
          return;
        }
      }
    }
  }
}

AST::Statement* ParserBase::parseStatementList()
{
  GTL_ASSERT( d->currentToken.type == Token::STARTBRACE );
  getNextToken();
  std::list<AST::Statement*> list;
  while(d->currentToken.type != Token::ENDBRACE)
  {
    AST::Statement* statement = parseStatement();
    if( not statement )
    {
      return 0;
    }
    list.push_back( statement );
  }
  getNextToken();
  return new AST::StatementsList( list );
}

AST::Statement* ParserBase::parsePrintStatement()
{
  GTL_ASSERT( d->currentToken.type == Token::PRINT );
  getNextToken();
  if( isOfType( d->currentToken, Token::STARTBRACKET ) )
  {
    getNextToken(); // eat the '('
    std::list<AST::Expression*> expressions;
    while(true)
    {
      AST::Expression* expr = 0;
      if( d->currentToken.type == Token::STRING_CONSTANT )
      {
        expr = new AST::StringExpression( d->currentToken.string );
        getNextToken();
      } else {
        expr = parseExpression(false);
      }
      expressions.push_back(expr);
      if( d->currentToken.type == Token::ENDBRACKET )
      {
        getNextToken();
        break;
      } else if( not isOfType(d->currentToken, Token::COMA) ) {
        getNextToken();
        return 0;
      }
      getNextToken();
    }
    isOfType( d->currentToken, Token::SEMI);
    getNextToken();
    return new AST::PrintStatement( expressions );
  }
  return 0;
}

AST::Statement* ParserBase::parseStatementOrList()
{
  startContext();
  AST::Statement* statement;
  if( d->currentToken.type == Token::STARTBRACE )
  {
    statement = parseStatementList();
  } else {
    statement = parseStatement();
  }
  endContext();
  return statement;
}

AST::Statement* ParserBase::parseWhileStatement()
{
  GTL_ASSERT( d->currentToken.type == Token::WHILE );
  getNextToken();
  if( isOfType( d->currentToken, Token::STARTBRACKET ) )
  {
    getNextToken(); // eat the '('
    AST::Expression* expression = parseExpression(false);
    if( isOfType( d->currentToken, Token::ENDBRACKET ) )
    {
      getNextToken(); // eat the ')'
      AST::Statement* statement = parseStatementOrList();
      return new AST::WhileStatement( expression, statement);
    }
  }
  return 0;
}

AST::Statement* ParserBase::parseForStatement()
{
  startContext();
  GTL_ASSERT( d->currentToken.type == Token::FOR );
  getNextToken();
  if( isOfType( d->currentToken, Token::STARTBRACKET ) )
  {
    getNextToken(); // eat the '('
    AST::Statement* initexpression = 0;
    if( d->currentToken.type == Token::SEMI )
    {
      getNextToken(); // eat the ';'
    } else {
      initexpression = parseStatement();
    }
    AST::Expression* comparexpression = parseExpression(false);
    if( isOfType( d->currentToken, Token::SEMI ) )
    {
      getNextToken(); // eat the ';'
      AST::Expression* updateexpression = 0;
      if( d->currentToken.type != Token::ENDBRACKET )
      {
        updateexpression = parseExpression(false);
      }
      if( isOfType( d->currentToken, Token::ENDBRACKET ) )
      {
        getNextToken(); // eat the ')'
        AST::Statement* statement = parseStatementOrList();
        endContext();
        return new AST::ForStatement(initexpression, comparexpression, updateexpression, statement);
      }
      delete updateexpression;
      delete comparexpression;
    }
    delete initexpression;
  }
  endContext();
  return 0;
}

AST::Statement* ParserBase::parseIfStatement()
{
  GTL_ASSERT( d->currentToken.type == Token::IF );
  getNextToken();
  if( isOfType( d->currentToken, Token::STARTBRACKET ) )
  {
    getNextToken(); // eat the '('
    AST::Expression* expression = parseExpression(false);
    if( isOfType( d->currentToken, Token::ENDBRACKET ) )
    {
      getNextToken(); // eat the ')'
      AST::Statement* statement = parseStatementOrList();
      if( d->currentToken.type == Token::ELSE )
      {
        getNextToken(); // eat the else
        AST::Statement* elsestatement = parseStatementOrList();
        return new AST::IfElseStatement( expression, statement, elsestatement);
      } else {
        return new AST::IfStatement( expression, statement);
      }
    }
  }
  return 0;
}

AST::Statement* ParserBase::parseReturnStatement()
{
  GTL_ASSERT( d->currentToken.type == Token::RETURN );
  getNextToken();
  if( d->currentToken.type == Token::SEMI )
  { // TODO check type
    getNextToken(); // eat the coma
    return new AST::ReturnStatement( 0, functionVariables() );
  } else {
    AST::Expression* expr = parseExpression(false);
    if(expr and isOfType( d->currentToken, Token::SEMI ) )
    {
      getNextToken();
      return new AST::ReturnStatement( expr, functionVariables() );
    } else {
      return 0;
    }
  }
}

AST::Statement* ParserBase::parseExpressionStatement()
{
  AST::Statement* statement = parseExpression(false);
  if(isOfType( d->currentToken, Token::SEMI ) )
  {
    getNextToken();
    return statement;
  } else {
    getNextToken();
    return 0;
  }
}

AST::Statement* ParserBase::parseVariableDeclaration()
{
  bool constant = false;
  if( d->currentToken.type == Token::CONST )
  {
    constant = true;
    getNextToken();
  }
  if( isType(d->currentToken) )
  {
    const Type* type = parseType();
    GTL_ASSERT(type);
    if( isOfType( d->currentToken, Token::IDENTIFIER ) )
    {
      ScopedName name("", d->currentToken.string);
      if( d->contextes.begin()->variables.find( name ) == d->contextes.begin()->variables.end() )
      {
        getNextToken();
        std::list<AST::Expression*> initialsize = parseArraySize(false);
        type = d->compiler->typeManager()->getArray( type, initialsize.size() );
        bool initialised = false;
        AST::Expression* initialiser = 0;
        if( d->currentToken.type == Token::EQUAL )
        {
          getNextToken(); // eat the equal
          // Check if it's a coumpound expression
          initialiser = parseExpression( false, type );
          initialised = true;
        }
        // Create the variable declaration statement
        AST::VariableDeclaration* variable = new AST::VariableDeclaration( type, initialiser, constant, initialsize );
        // Insert the variable in the context
        d->contextes.begin()->variables[name] = variable->variable();
        GTL_ASSERT( variable->variable() );
        GTL_DEBUG(initialised << " " << Token::typeToString( d->currentToken.type ) );
        if( d->currentToken.type == Token::COMA and not initialised ) {
          // it's possible to initialise a variable by calling a function taking as output the variable
          getNextToken(); // eat the coma
          variable->setFunctionIntialiser(parseExpression( false ));
          initialised = true;
        }
        if( constant and not initialised )
        {
          reportError( "Unitialised constant.", d->currentToken );
          return 0;
        }
        if( isOfType(d->currentToken, Token::SEMI ) )
        {
          getNextToken(); // eat the semi
          return variable;
        } else {
          GTL_DEBUG("Expecting ';'");
          getNextToken();
          return 0;
        }
      } else {
        reportError("Variable " + name.name() + " has allready been defined.", d->currentToken);
        return 0;
      }
    }
    return 0;
  } else {
    GTL_DEBUG("Unexpected");
    reportUnexpected( d->currentToken );
    return 0;
  }
}

const Type* ParserBase::parseFunctionType()
{
  if( d->currentToken.type == Token::VOID )
  {
    getNextToken();
    return Type::Void;
  } else {
    const Type* type = parseType();
    if( not type ) return 0;
    std::list< AST::Expression* > sizes = parseArraySize(true);
    deleteAll( sizes );
    return d->compiler->typeManager()->getArray( type, sizes.size() );
  }
}

const Type* ParserBase::parseType()
{
  switch( d->currentToken.type )
  {
    case Token::BOOL:
      getNextToken();
      return Type::Boolean;
    case Token::INT:
      getNextToken();
      return Type::Integer32;
    case Token::HALF:
    case Token::FLOAT:
      getNextToken();
      return Type::Float;
    case Token::UNSIGNED:
    {
      getNextToken();
      if( d->currentToken.type == Token::INT )
      {
        getNextToken();
      }
      return Type::UnsignedInteger32;
    }
    case Token::IDENTIFIER:
    {
      const Type* type = d->compiler->typeManager()->getStructure( d->currentToken.string );
      if( not type )
      {
        reportError("Unknown type : " + d->currentToken.string, d->currentToken );
      }
      getNextToken();
      return type;
    }
    default:
      reportError("Expected type before " + Token::typeToString( d->currentToken.type ), d->currentToken);
      getNextToken();
      return Type::Integer32;
  }
}

AST::Expression* ParserBase::parsePrimaryExpression(bool _constantExpression)
{
  switch( d->currentToken.type )
  {
    case Token::INTEGER_CONSTANT:
    {
      int v = d->currentToken.i;
      getNextToken();
      return new AST::NumberExpression<int>( v );
    }
    case Token::FLOAT_CONSTANT:
    {
      float v = d->currentToken.f;
      getNextToken();
      return new AST::NumberExpression<float>( v );
    }
    case Token::TTRUE:
      getNextToken();
      return new AST::NumberExpression<bool>( true );
    case Token::TFALSE:
      getNextToken();
      return new AST::NumberExpression<bool>( false );
    case Token::IDENTIFIER:
    {
      if(_constantExpression)
      { // If we are computing a constant expression, we can only use the previously declared constant variable
        for( std::list< AST::GlobalConstantDeclaration*>::const_iterator it = tree()->globalConstantDeclarations().begin();
             it != tree()->globalConstantDeclarations().end(); ++it)
        {
          if( (*it)->name().name() == d->currentToken.string )
          {
            getNextToken();
            // Put the expr in a proxy to avoid double free when the AST tree is deleted
            return new AST::ProxyExpression(  (*it)->initialiser() );
          }
        }
        AST::Expression* stdconst = d->compiler->standardConstant( d->currentToken.string );
        if( stdconst )
        {
          return stdconst;
        }
        getNextToken();
        reportError( "Unknown constant: " + d->currentToken.string, d->currentToken );
        return 0;
      } else {
        // It can be either a call to a function or a variable access
        ScopedName name("", d->currentToken.string );
        getNextToken(); // eat the identifier
        if( d->currentToken.type == Token::COLONCOLON )
        {
          getNextToken(); // eat the ::
          name = ScopedName( name.name(), d->currentToken.string );
          getNextToken(); // eat the name
        }
        if( d->currentToken.type == Token::STARTBRACKET )
        { // It's a function call
          Function* function = d->compiler->function( name );
          if( not function )
          {
            reportError("Unknown function: " + name.toString(), d->currentToken);
            getNextToken();
            return 0;
          }
          getNextToken();
          // Parse arguments
          std::list<AST::Expression*> arguments = parseArguments( function->name().toString(), function->parameters() );
          if( arguments.size() >= function->d->data->minimumParameters() and arguments.size() <= function->d->data->maximumParameters() )
          {
            return new AST::FunctionCallExpression( function, arguments ) ;
          } else {
            reportError("Incorrect number of paramaters", d->currentToken);
            return 0;
          }
        } else {
          VariableNG* var = getVariable( name );
          if(not var)
          {
            AST::Expression* stdconst = d->compiler->standardConstant( name.name() );
            if( stdconst )
            {
              return stdconst;
            } else {
              reportError( "Unknown variable: " + name.toString(), d->currentToken );
              return 0;
            }
          }
          return parseMemberArrayExpression(new AST::VariableAccessorExpression( var ) , _constantExpression);
        }
      }
      break;
    }
    default:
      GTL_DEBUG("unexpected");
      reportUnexpected( d->currentToken );
  }
  return 0;
}

std::list<AST::Expression*> ParserBase::parseArguments( const String& _name, const std::vector< Parameter >& _parameters )
{
  std::list<AST::Expression*> arguments;
  while(true)
  {
    if( d->currentToken.type == Token::ENDBRACKET )
    {
      break;
    } else if( arguments.size() == _parameters.size() ) {
      // Too much arguments
      reportError( "Too many arguments for function '" + _name + "'", d->currentToken);
      return arguments;
    } else {
      int posArg = arguments.size();
      AST::Expression* expression = parseExpression( false, _parameters[ posArg ].type() );
      arguments.push_back( expression);
      if( _parameters[ posArg ].output() )
      {
        if( not dynamic_cast< AST::AccessorExpression* >( expression ) )
        {
          reportError( "Parameter of function '" + _name + "' is an output parameter and requires a variable as argument", d->currentToken );
        }
      }
      if( d->currentToken.type == Token::COMA )
      {
        getNextToken();
      } else if( d->currentToken.type != Token::ENDBRACKET )
      {
        GTL_DEBUG("Unexpected");
        reportUnexpected( d->currentToken );
        return std::list<AST::Expression*>();
      }
    }
  }
  GTL_ASSERT( d->currentToken.type == Token::ENDBRACKET );
  getNextToken(); // eat the end bracket
  return arguments;
}

void ParserBase::reachNextSemi()
{
  while( d->currentToken.type != Token::SEMI and d->currentToken.type != Token::END_OF_FILE )
  {
    getNextToken();
  }
  getNextToken();
}

// Utilities

bool ParserBase::isOfType( const Token& token, Token::Type type )
{
  if( token.type == type )
  {
    return true;
  } else {
    reportError("Expected " + Token::typeToString(type) + " before " + Token::typeToString(token.type)  + ".", token);
    return false;
  }
}

void ParserBase::checkNextTokenIsSemi()
{
  getNextToken();
  if( d->currentToken.type != Token::SEMI )
  {
    reportError("Expected ';' before " + Token::typeToString(d->currentToken.type) + ".", d->currentToken );
  }
}

void ParserBase::reportError( const String& errMsg, const Token& token )
{
  if( d->compiler)
  {
    d->compiler->appendError( ErrorMessage( errMsg, token.line, "" ) );
  } else {
    GTL_DEBUG( errMsg);
  }
}

void ParserBase::reportUnexpected( const Token& token )
{
  reportError("Unexpected: " + Token::typeToString( token.type ), token );
  getNextToken();
}

AST::Expression* ParserBase::createBinaryOperator( const Token& token, AST::Expression* lhs, AST::Expression* rhs )
{
  if( not lhs or not rhs )
  {
    delete lhs;
    delete rhs;
    return 0;
  }
  if( token.type == Token::EQUAL )
  {
    AST::AccessorExpression* ve = dynamic_cast<AST::AccessorExpression*>( lhs );
    if( ve )
    {
      return new AST::AssignementBinaryExpression( ve, rhs );
    }
    reportError("Left hand side of an assignement expression must be a variable.", token);
    return 0;
    
  } else {
    std::pair<AST::Expression*, AST::Expression*>  ce = d->compiler->convertCenter()->createConvertExpressions( lhs, rhs );
    GTL_DEBUG( "Convert from (" << *lhs->type() << ", " << *rhs->type() << " ) to ( " << *ce.first->type() << ", " << *ce.second->type() << " )" );
    if( not ce.first or not ce.second )
    {
      reportError( "Can do a binary operation only on two numbers", token );
      if(ce.first)
      {
        delete ce.first;
      } else {
        delete lhs;
      }
      if(ce.second)
      {
        delete ce.second;
      } else {
        delete rhs;
      }
      return 0;
    }
    lhs = ce.first;
    rhs = ce.second;
    switch( token.type )
    {
      case Token::OR:
        return new AST::OrBinaryExpression( lhs, rhs );
      case Token::AND:
        return new AST::AndBinaryExpression( lhs, rhs );
      case Token::BITOR:
        return new AST::BitOrBinaryExpression( lhs, rhs );
      case Token::BITXOR:
        return new AST::BitXorBinaryExpression( lhs, rhs );
      case Token::BITAND:
        return new AST::BitAndBinaryExpression( lhs, rhs );
      case Token::EQUALEQUAL:
        return new AST::EqualEqualBinaryExpression( lhs, rhs );
      case Token::DIFFERENT:
        return new AST::DifferentBinaryExpression( lhs, rhs );
      case Token::INFERIOREQUAL:
        return new AST::InferiorEqualBinaryExpression( lhs, rhs );
      case Token::INFERIOR:
        return new AST::InferiorBinaryExpression( lhs, rhs );
      case Token::SUPPERIOREQUAL:
        return new AST::SupperiorEqualBinaryExpression( lhs, rhs );
      case Token::SUPPERIOR:
        return new AST::SupperiorBinaryExpression( lhs, rhs );
      case Token::RIGHTSHIFT:
        return new AST::RightShiftBinaryExpression( lhs, rhs );
      case Token::LEFTSHIFT:
        return new AST::LeftShiftBinaryExpression( lhs, rhs );
      case Token::PLUS:
        return new AST::AdditionBinaryExpression( lhs, rhs );
      case Token::MINUS:
        return new AST::SubstractionBinaryExpression( lhs, rhs );
      case Token::MULTIPLY:
        return new AST::MultiplicationBinaryExpression( lhs, rhs );
      case Token::DIVIDE:
        return new AST::DivisionBinaryExpression( lhs, rhs );
      case Token::MODULO:
      {
        if( lhs->type() != rhs->type() and rhs->type()  != Type::Integer32 )
        {
          reportError( "'~' operator only work with integer" , token );
          return 0;
        }
        return new AST::ModuloBinaryExpression( lhs, rhs );
      }
      default:
        return 0;
    }
  }
}

bool ParserBase::isType( const Token& token )
{
  return token.isNativeType() or ( token.type == Token::IDENTIFIER and d->compiler->typeManager()->d->isKnownType( token.string ) );
}

// Context management

void ParserBase::startContext()
{
  d->contextes.push_front( Context() );
}

void ParserBase::endContext()
{
  d->contextes.pop_front();
}

VariableNG* ParserBase::getVariable( const ScopedName& n ) const
{
  GTL_DEBUG("getVariable " << n );
  for( std::list<Context>::const_iterator cit = d->contextes.begin();
       cit != d->contextes.end(); cit++)
  {
    for( std::map<ScopedName, VariableNG*>::const_iterator it = cit->variables.begin();
         it != cit->variables.end(); ++it)
    {
      GTL_DEBUG( " storage: " << it->first << " namespace: " << d->nameSpace );
      if( it->first == n
          or ( it->first.nameSpace() == d->nameSpace and it->first.name() == n.name() ) )
      {
        return it->second;
      }
    }
  }
  return 0;
}

std::list<VariableNG*> ParserBase::functionVariables() const
{
  return d->functionVariables;
}

void ParserBase::declareVariable( const ScopedName& _scopedName, VariableNG* _variable)
{
  d->contextes.begin()->variables[_scopedName] = _variable;
  d->functionVariables.push_back( _variable );
}

std::list<int> ParserBase::expressionsListToIntegersList( const std::list< AST::Expression* >& list )
{
  std::list<int> integersList;
  CodeGenerator cg( 0 );
  GenerationContext gc( &cg, 0, 0, 0);
  for( std::list< AST::Expression* >::const_iterator it = list.begin();
       it != list.end(); ++it)
  {
    if( *it )
    {
      if( (*it)->isConstant() )
      {
        llvm::ConstantInt* v = dynamic_cast<llvm::ConstantInt* >( (*it)->generateValue( gc, 0 ).constant() );
        GTL_ASSERT( v );
        integersList.push_back( v->getValue().getLimitedValue() );
      } else {
        reportError( "Expected constant expression.", d->currentToken );
      }
    } else {
      integersList.push_back( -1 );
    }
  }
  return integersList;
}

void ParserBase::setNameSpace( const String& _nameSpace )
{
  d->nameSpace = _nameSpace;
}

const String& ParserBase::nameSpace() const
{
  return d->nameSpace;
}

TypeManager* ParserBase::typeManager()
{
  return d->compiler->typeManager();
}
