/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | Copyright (C) 2004-2010, 2019 OpenCFD Ltd.
     \\/     M anipulation  |
-------------------------------------------------------------------------------
                            | Copyright (C) 2011-2016 OpenFOAM Foundation
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Class
    Foam::RectangularMatrix

Description
    A templated 2D rectangular m x n matrix of objects of \<Type\>.

    The matrix dimensions are used for subscript bounds checking etc.

SourceFiles
    RectangularMatrixI.H
    RectangularMatrix.C

\*---------------------------------------------------------------------------*/

#ifndef RectangularMatrix_H
#define RectangularMatrix_H

#include "Matrix.H"
#include "SquareMatrix.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{

/*---------------------------------------------------------------------------*\
                      Class RectangularMatrix Declaration
\*---------------------------------------------------------------------------*/

template<class Type>
class RectangularMatrix
:
    public Matrix<RectangularMatrix<Type>, Type>
{

public:

    // Constructors

        //- Construct null
        inline RectangularMatrix();

        //- Construct a square matrix (rows == columns)
        inline explicit RectangularMatrix(const label n);

        //- Construct given number of rows/columns
        inline RectangularMatrix(const label m, const label n);

        //- Construct given number of rows/columns
        //- initializing all elements to zero
        inline RectangularMatrix(const label m, const label n, const zero);

        //- Construct given number of rows/columns
        //- initializing all elements to the given value
        inline RectangularMatrix(const label m, const label n, const Type& val);

        //- Construct given number of rows/columns
        inline explicit RectangularMatrix(const labelPair& dims);

        //- Construct given number of rows/columns
        //- initializing all elements to zero
        inline RectangularMatrix(const labelPair& dims, const zero);

        //- Construct given number of rows/columns
        //- initializing all elements to the given value
        inline RectangularMatrix(const labelPair& dims, const Type& val);

        //- Construct from a block of another matrix
        template<class MatrixType>
        inline RectangularMatrix(const ConstMatrixBlock<MatrixType>& mat);

        //- Construct from a block of another matrix
        template<class MatrixType>
        inline RectangularMatrix(const MatrixBlock<MatrixType>& mat);

        //- Construct as copy of a square matrix
        inline RectangularMatrix(const SquareMatrix<Type>& mat);

        //- Construct from Istream.
        inline explicit RectangularMatrix(Istream& is);

        //- Clone
        inline autoPtr<RectangularMatrix<Type>> clone() const;


    // Member Operators

        //- Assign all elements to zero
        inline void operator=(const zero);

        //- Assign all elements to value
        inline void operator=(const Type& val);
};


// * * * * * * * * * * * * * * * Global Functions  * * * * * * * * * * * * * //

template<class Type>
class typeOfInnerProduct<Type, RectangularMatrix<Type>, RectangularMatrix<Type>>
{
public:

    typedef RectangularMatrix<Type> type;
};


template<class Type>
class typeOfInnerProduct<Type, RectangularMatrix<Type>, SquareMatrix<Type>>
{
public:

    typedef RectangularMatrix<Type> type;
};


template<class Type>
class typeOfInnerProduct<Type, SquareMatrix<Type>, RectangularMatrix<Type>>
{
public:

    typedef RectangularMatrix<Type> type;
};


// * * * * * * * * * * * * * * * Friend Functions  * * * * * * * * * * * * * //

template<class Type>
RectangularMatrix<Type> outer(const Field<Type>& f1, const Field<Type>& f2);


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#include "RectangularMatrixI.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
