/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | Copyright (C) 2004-2010, 2018 OpenCFD Ltd.
     \\/     M anipulation  |
-------------------------------------------------------------------------------
                            | Copyright (C) 2011-2016 OpenFOAM Foundation
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Application
    testHashing

Description

\*---------------------------------------------------------------------------*/

#include "IOstreams.H"
#include "IOobject.H"
#include "IFstream.H"

#include "stringList.H"
#include "labelList.H"
#include "labelPair.H"
#include "edgeList.H"
#include "faceList.H"
#include "triFaceList.H"

#include "Hash.H"

using namespace Foam;

void infoHashString
(
    unsigned modulus,
    std::initializer_list<std::string> lst
)
{
    if (modulus)
    {
        Info<< "basic string hashing (mod " << label(modulus) << ")" << nl;

        for (const auto& str : lst)
        {
            Info<<"hash(" << str.c_str() << ")="
                << (Hash<string>()(str) % modulus) << nl;
        }

    }
    else
    {
        Info<< "basic string hashing" << nl;

        for (const auto& str : lst)
        {
            Info<<"hash(" << str.c_str() << ")="
                << Hash<string>()(str) << nl;
        }
    }
}



void reportHashList(const UList<string>& list)
{
    Info<< "contiguous = " << contiguous<string>() << nl << nl;

    for (const string& val : list)
    {
        unsigned hash1 = string::hash()(val);

        Info<< hex << hash1 << ": " << val << nl;
    }
}


void reportHashList(const UList<label>& list)
{
    Info<<"contiguous = " << contiguous<label>() << nl << nl;

    for (const label val : list)
    {
        // Direct value
        unsigned hash1 = Hash<label>()(val);

        // Hashed byte-wise
        unsigned hash2 = Hash<label>()(val, 0);

        Info<< hex << hash1
            << " (seeded: " << hash2 << ")"
            << ": " << dec << val << nl;
    }
}


void reportHashList(const UList<face>& list)
{
    Info<<"contiguous = " << contiguous<label>() << nl << nl;

    for (const face& f : list)
    {
        // Direct value
        unsigned hash1 = face::Hash<>()(f);

        unsigned hash2 = Hash<face>()(f);

        Info<< hex << "face::Hash<> " << hash1
            << " Hash<face> " << hash2
            << ": " << dec << flatOutput(f) << nl;
    }
}


void reportHashList(const UList<labelList>& list)
{
    for (const labelList& val : list)
    {
        unsigned hash1 = Hasher
        (
            val.cdata(),
            val.size() * sizeof(label)
        );

        unsigned hash2 = Hash<labelList>()(val);
        unsigned hash2b = labelList::Hash<>()(val);

        Info<< hex << hash1 << " or " << hash2
            << "(" << hash2b << ") "
            << ": " << dec << val << nl;
    }

    unsigned hash2 = Hash<labelListList>()(list);
    unsigned hash2bad = HasherT(list);

    Info<< hex << hash2 << " : " << dec << flatOutput(list) << nl
        << hex << hash2bad << " as direct hash would be wrong"
        << dec << nl;
}


typedef Pair<word> wordPair;

void reportHashList(const UList<wordPair>& list)
{
    Info<<"contiguous = " << contiguous<wordPair>() << nl << nl;

    for (const wordPair& pr : list)
    {
        unsigned hash1 = Hash<wordPair>()(pr);

        // as FixedList
        unsigned hash2 = wordPair::Hash<>()(pr);

        // as FixedList
        unsigned hash2sym = wordPair::SymmHash<>()(pr);

        // as FixedList
        unsigned hash3 = Hash<FixedList<word,2>>()(pr);

        Info<< hex << hash1 << " (as FixedList: " << hash2
            << ") or " << hash3
            << " symm-hash:" << hash2sym
            << " : "<< dec << flatOutput(pr) << nl;
    }
}


void reportHashList(const UList<labelPair>& list)
{
    Info<<"contiguous = " << contiguous<labelPair>() << nl << nl;

    for (const labelPair& pr : list)
    {
        unsigned hash1 = Hash<labelPair>()(pr);

        // as FixedList
        unsigned hash2 = labelPair::Hash<>()(pr);

        // as FixedList
        unsigned hash3 = Hash<labelPair>()(pr);

        Info<< hex << hash1 << " (as FixedList: " << hash2
            << ") or " << hash3
            << " : "<< dec << pr << nl;
    }
}


void reportHashList(const UList<labelPairPair>& list)
{
    Info<<"contiguous = " << contiguous<labelPairPair>() << nl << nl;

    for (const labelPairPair& pr : list)
    {
        unsigned hash1 = Hash<labelPairPair>()(pr);

        // as FixedList
        unsigned hash2 = labelPairPair::Hash<>()(pr);

        // as FixedList
        unsigned hash3 = Hash<labelPairPair>()(pr);

        Info<< hex << hash1 << " (as FixedList: " << hash2
            << ") or " << hash3
            << " : "<< dec << pr << nl;
    }
}


void reportHashList(const UList<edge>& list)
{
    Info<<"contiguous = " << contiguous<edge>() << nl << nl;

    for (const edge& e : list)
    {
        unsigned hash1 = Hash<edge>()(e);

        // as FixedList
        unsigned hash2 = labelPair::Hash<>()(e);

        // as FixedList
        unsigned hash3 = Hash<labelPair>()(e);

        Info<< hex << hash1 << " (as FixedList: " << hash2
            << ") or " << hash3
            << " : "<< dec << e << nl;
    }
}


void reportHashList(const UList<triFace>& list)
{
    Info<<"contiguous = " << contiguous<triFace>() << nl << nl;

    for (const triFace& f : list)
    {
        // direct value
        unsigned hash1 = Hash<triFace>()(f);
        unsigned hash2 = FixedList<label, 3>::Hash<>()(f);

        Info<< hex << hash1 << " (as FixedList: " << hash2
            << "): " << dec << f << nl;
    }
}


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //
//  Main program:

int main(int argc, char *argv[])
{
    infoHashString(8, {"asdathis1", "adsxf", "hij", "klmpq"});

    IFstream is("hashingTests");

    if (!is.good())
    {
        Info<< "No \"hashingTests\" file found ...\n";
    }

    while (is.good())
    {
        const word listType(is);

        if (listType.empty()) continue;

        Info<< nl;
        IOobject::writeDivider(Info) << listType << nl;

        if (listType == "stringList")
        {
            stringList list(is);

            reportHashList(list);
        }
        else if (listType == "labelList")
        {
            labelList list(is);
            reportHashList(list);
        }
        else if (listType == "faceList")
        {
            faceList list(is);
            reportHashList(list);
        }
        else if (listType == "labelListList")
        {
            List<labelList> list(is);
            reportHashList(list);
        }
        else if (listType == "wordPairList")
        {
            List<wordPair> list(is);
            reportHashList(list);
        }
        else if (listType == "labelPairList")
        {
            labelPairList list(is);
            reportHashList(list);
        }
        else if (listType == "labelPairPairList")
        {
            List<labelPairPair> list(is);
            reportHashList(list);
        }
        else if (listType == "edgeList")
        {
            edgeList list(is);
            reportHashList(list);
        }
        else if (listType == "triFaceList")
        {
            triFaceList list(is);
            reportHashList(list);
        }
        else
        {
            Info<< "unknown type: " << listType << nl;
        }
    }

    Info<< "\nEnd\n" << nl;

    return 0;
}


// ************************************************************************* //
