#!/usr/bin/perl -w

###################################################
# package to parse the mapi-properties files and 
# generate code for libmapi in OpenChange
#
# Perl code based on pidl one from Andrew Tridgell and the Samba team
#
# Copyright (C) Julien Kerihuel 2005-2008
# released under the GNU GPL

use strict;
use Getopt::Long;

my $ret = "";
my $tabs = "";

sub indent() { $tabs.="\t"; }
sub deindent() { $tabs = substr($tabs, 1); }
sub mparse($) { $ret .= $tabs.(shift)."\n"; }

my($opt_outputdir) = '.';
my($opt_parser) = '';

my	%prop_types = (
    0x0		=> "PT_UNSPECIFIED",
    0x1		=> "PT_NULL",
    0x2		=> "PT_SHORT",
    0x3		=> "PT_LONG",
    0x4		=> "PT_FLOAT",
    0x5		=> "PT_DOUBLE",
    0x6		=> "PT_CURRENCY",
    0x7		=> "PT_APPTIME",
    0xa		=> "PT_ERROR",
    0xb		=> "PT_BOOLEAN",
    0xd		=> "PT_OBJECT",
    0x14	=> "PT_I8",
    0x1e	=> "PT_STRING8",
    0x1f	=> "PT_UNICODE",
    0x40	=> "PT_SYSTIME",
    0x48	=> "PT_CLSID",
    0x102	=> "PT_BINARY",
# Multi-valued property types
    0x1002	=> "PT_MV_SHORT",
    0x1003	=> "PT_MV_LONG",
    0x1004	=> "PT_MV_FLOAT",
    0x1005	=> "PT_MV_DOUBLE",
    0x1006	=> "PT_MV_CURRENCY",
    0x1007	=> "PT_MV_APPTIME",
    0x1014	=> "PT_MV_I8",
    0x101e	=> "PT_MV_STRING8",
    0x101f	=> "PT_MV_UNICODE",
    0x1040	=> "PT_MV_SYSTIME",
    0x1048	=> "PT_MV_CLSID",
    0x1102	=> "PT_MV_BINARY"
);

my	%prop_names = (
    "PT_UNSPECIFIED"	=>	0x0,
    "PT_NULL"		=>	0x1,
    "PT_SHORT"		=>	0x2,
    "PT_LONG"		=>	0x3,
    "PT_FLOAT"		=>	0x4,
    "PT_DOUBLE"		=>	0x5,
    "PT_CURRENCY"	=>	0x6,
    "PT_APPTIME"	=>	0x7,
    "PT_ERROR"		=>	0xa,
    "PT_BOOLEAN"	=>	0xb,
    "PT_OBJECT"		=>	0xd,
    "PT_I8"		=>	0x14,
    "PT_STRING8"	=>	0x1e,
    "PT_UNICODE"	=>	0x1f,
    "PT_SYSTIME"	=>	0x40,
    "PT_CLSID"		=>	0x48,
    "PT_BINARY"		=>	0x102,
# Multi-valued property types
    "PT_MV_SHORT"	=>	0x1002,
    "PT_MV_LONG"	=>	0x1003,
    "PT_MV_FLOAT"	=>	0x1004,
    "PT_MV_DOUBLE"	=>	0x1005,
    "PT_MV_CURRENCY"	=>	0x1006,
    "PT_MV_APPTIME"	=>	0x1007,
    "PT_MV_I8"		=>	0x1014,
    "PT_MV_STRING8"	=>	0x101e,
    "PT_MV_UNICODE"	=>	0x101f,
    "PT_MV_SYSTIME"	=>	0x1040,
    "PT_MV_CLSID"	=>	0x1048,
    "PT_MV_BINARY"	=>	0x1102
);

# main program

my $result = GetOptions (
			 'outputdir=s' => \$opt_outputdir,
			 'parser=s' => \$opt_parser
			 );

if (not $result) {
    exit(1);
}

#####################################################################
# read a file into a string
sub FileLoad($)
{
    my($filename) = shift;
    local(*INPUTFILE);
    open(INPUTFILE, $filename) || return undef;
    my($saved_delim) = $/;
    undef $/;
    my($data) = <INPUTFILE>;
    close(INPUTFILE);
    $/ = $saved_delim;
    return $data;
}

#####################################################################
# write a string into a file
sub FileSave($$)
{
    my($filename) = shift;
    my($v) = shift;
    local(*FILE);
    open(FILE, ">$filename") || die "can't open $filename";    
    print FILE $v;
    close(FILE);
}

#####################################################################
# generate mapitags.h file
sub mapitags_header($)
{
    my $contents = shift;
    my $line;
    my @lines;
    my @prop;
    my $prop_type;
    my $prop_value;

    mparse "/* parser auto-generated by mparse */";
    mparse "#ifndef __MAPITAGS_H__";
    mparse "#define __MAPITAGS_H__";
    mparse "";

    @lines = split(/\n/, $contents);
    foreach $line (@lines) {
	$line =~ s/^\#+.*$//;
	if ($line) {
	    @prop = split(/\s+/, $line);
	    $prop_type = hex $prop[0];
	    $prop_type &= 0xFFFF;
	    $prop_value = hex $prop[0];
	    $prop_value = ($prop_value >> 16) & 0xFFFF;

	    mparse sprintf "#define %-51s PROP_TAG(%-13s, 0x%.04x) /* %s */", $prop[1], $prop_types{$prop_type}, $prop_value, $prop[0] if ($prop_types{$prop_type});
	    if (($prop_type == 0x1e) || ($prop_type == 0x101e)){
		$prop_type++;
		$prop[0] = sprintf "0x%.8x", ((hex $prop[0]) & 0xFFFF0000) + $prop_type;
		mparse sprintf "#define %-51s PROP_TAG(%-13s, 0x%.04x) /* %s */", "$prop[1]_UNICODE", $prop_types{$prop_type}, $prop_value, $prop[0] if ($prop_types{$prop_type});
	    }
	    $prop_type = 0xa;
	    $prop[0] = sprintf "0x%.8x", ((hex $prop[0]) & 0xFFFF0000) + $prop_type;
	    mparse sprintf "#define %-51s PROP_TAG(%-13s, 0x%.04x) /* %s */", "$prop[1]_ERROR", $prop_types{$prop_type}, $prop_value, $prop[0] if ($prop_types{$prop_type});
	}
    }
    mparse "";
    mparse "#endif /* !__MAPITAGS_H__ */";

    return $ret;
    
}


#####################################################################
# generate mapitags.c file
sub mapitags_interface($)
{
    my $contents = shift;
    my $line;
    my @lines;
    my @prop;
    my $prop_url;
    my $prop_type;
    my $prop_value;

    mparse "/* parser auto-generated by mparse */";
    mparse "#include <libmapi/libmapi.h>";
    mparse "#include <libmapi/proto_private.h>";
    mparse "#include <gen_ndr/ndr_exchange.h>";
    mparse "#include <libmapi/mapitags.h>";
    mparse "";
    mparse "struct mapi_proptags";
    mparse "{";
    indent;
    mparse "uint32_t	proptag;";
    mparse "uint32_t	proptype;";
    mparse "const char	*propname;";
    deindent;
    mparse "};";
    mparse "";
    mparse "static struct mapi_proptags mapitags[] = {";
    indent;
    
    @lines = split(/\n/, $contents);
    foreach $line (@lines) {
	$line =~ s/^\#+.*$//;
	if ($line) {
	    @prop = split(/\s+/, $line);
	    $prop_type = hex $prop[0];
	    $prop_type &= 0xFFFF;
	    $prop_value = hex $prop[0];
	    $prop_value = ($prop_value >> 16) & 0xFFFF;
	    if ($prop_types{$prop_type}) {
		mparse sprintf "{ %-51s, %-13s, \"%s\" },", $prop[1], $prop_types{$prop_type}, $prop[1];
		if (($prop_type == 0x1e) || ($prop_type == 0x101e)) {
		    $prop_type++;
		    mparse sprintf "{ %-51s, %-13s, \"%s\" },", "$prop[1]_UNICODE", $prop_types{$prop_type}, "$prop[1]_UNICODE";
		}
		    $prop_type = 0xa;
		mparse sprintf "{ %-51s, %-13s, \"%s\" },", "$prop[1]_ERROR", $prop_types{$prop_type}, $prop[1];
	    }
	}
    }

    mparse sprintf "{ %-51s, %-13d, \"NULL\"}", 0, 0;
    deindent;
    mparse "};";
    mparse "";
    mparse "_PUBLIC_ const char *get_proptag_name(uint32_t proptag)";
    mparse "{";
    indent;
    mparse "uint32_t idx;";
    mparse "";
    mparse "for (idx = 0; mapitags[idx].proptag; idx++) {";
    indent;
    mparse "if (mapitags[idx].proptag == proptag) { ";
    indent;
    mparse "return mapitags[idx].propname;";
    deindent;
    mparse "}";
    deindent;
    mparse "}";
    mparse "";
    mparse "return NULL;";
    deindent;
    mparse "}";
    mparse "";
    mparse "_PUBLIC_ uint32_t get_proptag_value(const char *propname)";
    mparse "{";
    indent;
    mparse "uint32_t idx;";
    mparse "";
    mparse "for (idx = 0; mapitags[idx].proptag; idx++) {";
    indent;
    mparse "if (!strcmp(mapitags[idx].propname, propname)) { ";
    indent;
    mparse "return mapitags[idx].proptag;";
    deindent;
    mparse "}";
    deindent;
    mparse "}";
    mparse "";
    mparse "return 0;";
    deindent;
    mparse "}";
    mparse "";

    return $ret;
    
}

#####################################################################
# generate mapitags_enum.idl file
sub mapitags_enum($)
{
    my $contents = shift;
    my $line;
    my @lines;
    my @prop;
    my $prop_type;
    my %hash;

    mparse "/* parser auto-generated by mparse */";
    mparse "";

    mparse "typedef [v1_enum, flag(NDR_PAHEX)] enum {";
    indent;
    
    @lines = split(/\n/, $contents);
    foreach $line (@lines) {
	$line =~ s/^\#+.*$//;
	if ($line) {
	    @prop = split(/\s+/, $line);
	    $prop_type = hex $prop[0];
	    $prop_type &= 0xFFFF;

	    mparse sprintf "%-51s = %s,", $prop[1], $prop[0];

	    if (($prop_type == 0x1e) || ($prop_type == 0x101e)) {
		$prop_type = hex $prop[0];
		$prop_type++;
		mparse sprintf "%-51s = 0x%.8x,", "$prop[1]_UNICODE", $prop_type;
	    }
	    if (!exists($hash{((hex $prop[0]) & 0xFFFF0000)})) {
		$prop_type = 0xa;
		$prop_type += ((hex $prop[0]) & 0xFFFF0000);
		mparse sprintf "%-51s = 0x%.8x,", "$prop[1]_ERROR", $prop_type;
		$hash{((hex $prop[0]) & 0xFFFF0000)} = 1;
	    }
	}
    }
    mparse sprintf "%-51s = %s", "MAPI_PROP_RESERVED", "0xFFFFFFFF";
    deindent;
    mparse "} MAPITAGS;";    
    mparse "";
    
    return $ret;
}

#####################################################################
# generate swig_mapitags.h file
sub mapitags_swig($)
{
    my $contents = shift;
    my $line;
    my @lines;
    my @prop;
    my $prop_type;
    my $prop_value;
    my %hash;

    mparse "/* parser auto-generated by mparse */";
    mparse "#ifndef __SWIG_MAPITAGS_H__";
    mparse "#define __SWIG_MAPITAGS_H__";
    mparse "";

    @lines = split(/\n/, $contents);
    foreach $line (@lines) {
	$line =~ s/^\#+.*$//;
	if ($line) {
	    @prop = split(/\s+/, $line);
	    $prop_type = hex $prop[0];
	    $prop_type &= 0xFFFF;

	    mparse sprintf "#define %-51s %s", $prop[1], $prop[0];

	    if (($prop_type == 0x1e) || ($prop_type == 0x101e)) {
		$prop_type = hex $prop[0];
		$prop_type++;
		mparse sprintf "#define %-51s 0x%.8x", "$prop[1]_UNICODE", $prop_type;
	    }
	    if (!exists($hash{((hex $prop[0]) & 0xFFFF0000)})) {
		$prop_type = 0xa;
		$prop_type += ((hex $prop[0]) & 0xFFFF0000);
		mparse sprintf "#define %-51s 0x%.8x", "$prop[1]_ERROR", $prop_type;
		$hash{((hex $prop[0]) & 0xFFFF0000)} = 1;
	    }
	}
    }

    mparse "";
    mparse "#endif /* !__SWIG_MAPITAGS_H__ */";

    return $ret;  
}

#####################################################################
# generate mapi_nameid_private.h file
sub mapi_nameid_private_header($)
{
    my $contents = shift;
    my $line;
    my @lines;
    my @prop;
    my $prop_OOM;
    my $prop_ID;
    my $prop_Type;
    my $prop_name;
    my $prop_Kind;
    my $prop_OLEGUID;

    mparse "/* parser auto-generated by mparse */";
    mparse "#ifndef __MAPI_NAMEID_PRIVATE_H__";
    mparse "#define __MAPI_NAMEID_PRIVATE_H__";
    mparse "";
    mparse "static struct mapi_nameid_tags mapi_nameid_tags[] = {";
    indent;
    
    @lines = split(/\n/, $contents);
    foreach $line (@lines) {
	$line =~ s/^\#+.*$//;
	if ($line) {
	    @prop = split(/\s+/, $line);

	    if ($prop[1] ne "NULL") {
		$prop[1] = sprintf "\"%s\"", $prop[1];
	    }
	    if ($prop[3] ne "NULL") {
		$prop[3] = sprintf "\"%s\"", $prop[3]
	    }
	    if ($prop[0] eq "NULL") {
		$prop[0] = "0x00000000";
	    }
	    mparse sprintf "{ %-51s, %-40s, %-15s, %-30s, %-30s, %-15s, %-20s, 0x0 },", 
	    $prop[0], $prop[1], $prop[2], $prop[3], $prop[4], $prop[5], $prop[6];
	}
    }

    mparse sprintf "{ 0x00000000, NULL, 0x0, NULL, PT_UNSPECIFIED, 0x0, NULL, 0x0 }";
    deindent;
    mparse "};";
    mparse "";
    mparse "#endif /* !MAPI_NAMEID_PRIVATE_H__ */";

    return $ret;
}

#####################################################################
# generate mapi_nameid.h file
sub mapi_nameid_header($)
{
    my $contents = shift;
    my $line;
    my @lines;
    my @prop;
    my $propID;
    my $proptype;
    my $property;
    my $counter = 0;

    mparse "/* parser auto-generated by mparse */";
    mparse "#ifndef __MAPI_NAMEID_H__";
    mparse "#define __MAPI_NAMEID_H__";
    mparse "";

    mparse "/* NOTE TO DEVELOPERS: If this is a MNID_STRING named property,";
    mparse " * then we use the arbitrary 0xa000-0afff property ID range for";
    mparse " * internal mapping purpose only.";
    mparse " */";

    mparse "";
    mparse "struct mapi_nameid_tags {";
    indent;
    mparse "uint32_t			proptag;";
    mparse "const char			*OOM;";
    mparse "uint16_t			lid;";
    mparse "const char			*Name;";
    mparse "uint32_t			propType;";
    mparse "uint8_t				ulKind;";
    mparse "const char			*OLEGUID;";
    mparse "uint32_t			position;";
    deindent;
    mparse "};";
    mparse "";
    mparse "struct mapi_nameid {";
    indent;
    mparse "struct MAPINAMEID		*nameid;";
    mparse "uint16_t			count;";
    mparse "struct mapi_nameid_tags		*entries;";
    deindent;
    mparse "};";
    mparse "";

    @lines = split(/\n/, $contents);
    foreach $line (@lines) {
	$line =~ s/^\#+.*$//;
	if ($line) {
	    @prop = split(/\s+/, $line);
	    if ($prop[0] ne "NULL") {
		$proptype = $prop_names{$prop[4]};
		$propID = hex $prop[2];
		if ($propID == 0) {
		    # 40960 == 0xa000
		    $propID = 40960 + $counter;
		    $counter++;
		}
		$property = sprintf "0x%.8x", ($propID << 16) | $proptype;
		mparse sprintf "#define %-51s %s", $prop[0], $property; 
	    }
	}
    }

    mparse "";
    mparse "#endif /* !MAPI_NAMEID_H__ */";

    return $ret;
}

#####################################################################
# generate mapicode.h file
sub mapicodes_header($)
{
    my $contents = shift;
    my $line;
    my @lines;
    my @prop;
    my $error;

    mparse "/* parser auto-generated by mparse */";
    mparse "#ifndef __MAPICODE_H__";
    mparse "#define __MAPICODE_H__";
    mparse "";
    mparse "#define MAPI_RETVAL_IF(x,e,c)	\\";
    mparse "do {				\\";
    mparse "	if (x) {			\\";
    mparse "		errno = (e);		\\";
    mparse "		if (c) {		\\";
    mparse "			talloc_free(c);	\\";
    mparse "		}			\\";
    mparse "		return -1;		\\";
    mparse "	}				\\";
    mparse "} while (0);";
    mparse "";
    mparse "#define OPENCHANGE_RETVAL_IF(x,e,c)	\\";
    mparse "do {				\\";
    mparse "	if (x) {			\\";
    mparse "		set_errno(e);		\\";
    mparse "		if (c) {		\\";
    mparse "			talloc_free(c);	\\";
    mparse "		}			\\";
    mparse "		return (e);		\\";
    mparse "	}				\\";
    mparse "} while (0);";
    mparse "";
    mparse "#define OPENCHANGE_RETVAL_ERR(e,c)	\\";
    mparse "do {				\\";
    mparse "	set_errno(e);			\\";
    mparse "	if (c) {			\\";
    mparse "		talloc_free(c);		\\";
    mparse "	}				\\";
    mparse "	return (e);			\\";
    mparse "} while (0);";
    mparse "";
    mparse "/* Status macros for MAPI */";
    mparse "typedef unsigned long	SCODE;";
    mparse "";
    mparse "";
    mparse "#define SEVERITY_ERROR	1";
    mparse "#define SEVERITY_WARN	0";
    mparse "";
    mparse "#define FACILITY_ITF	4";
    mparse "#define	MAKE_MAPI_CODE(sev, fac, code) \\";
    mparse "(((SCODE)(sev)<<31)|((SCODE)(fac)<<16)|((SCODE)(code)))";
    mparse "";
    mparse "#define	MAKE_MAPI_E(code) (MAKE_MAPI_CODE(SEVERITY_ERROR, FACILITY_ITF, code))";
    mparse "#define	MAKE_MAPI_S(code) (MAKE_MAPI_CODE(SEVERITY_WARN, FACILITY_ITF, code))";
    mparse "";
    mparse "#define	MAPI_STATUS_V(x) ((SCODE)x)";
    mparse "";
    mparse "#define	MAPI_STATUS_IS_OK(x) (MAPI_STATUS_V(x) == 0)";
    mparse "#define	MAPI_STATUS_IS_ERR(x) ((MAPI_STATUS_V(x) & 0xc0000000) == 0xc0000000)";
    mparse "#define	MAPI_STATUS_EQUAL(x,y) (MAPI_STATUS_V(x) == MAPI_STATUS_V(y))";
    mparse "";
    mparse "#define	MAPI_STATUS_IS_OK_RETURN(x) do { \\";
    mparse "		if (MAPI_STATUS_IS_OK(x)) {\\";
    mparse "			return x;\\";
    mparse "		}\\";
    mparse "} while (0)";
    mparse "";
    mparse "#define	MAPI_STATUS_NOT_OK_RETURN(x) do { \\";
    mparse "		if (!MAPI_STATUS_IS_OK(x)) {\\";
    mparse "			return x;\\";
    mparse "		}\\";
    mparse "} while (0)";
    mparse "";
    mparse "#define	MAPI_STATUS_IS_ERR_RETURN(x) do { \\";
    mparse "		if (MAPI_STATUS_IS_ERR(x)) {\\";
    mparse "			return x;\\";
    mparse "		}\\";
    mparse "} while (0)";
    mparse "";
    mparse "#define	MAPI_STATUS_NOT_ERR_RETURN(x) do { \\";
    mparse "		if (!MAPI_STATUS_IS_ERR(x)) {\\";
    mparse "			return x;\\";
    mparse "		}\\";
    mparse "} while (0)";
    mparse "";
    mparse "";
    mparse "#endif /* !__MAPICODE_H__ */";

    return $ret;
    
}

#####################################################################
# generate mapicode.c file

sub mapicodes_interface($)
{
    my $contents = shift;
    my $line;
    my @lines;
    my @errors;

    mparse "/* parser auto-generated by mparse */";
    mparse "#include <libmapi/libmapi.h>";
    mparse "#include <libmapi/proto_private.h>";
    mparse "#include <gen_ndr/ndr_exchange.h>";
    mparse "";
    mparse "void set_errno(enum MAPISTATUS status)";
    mparse "{";
    indent;
    mparse "errno = status;";
    deindent;
    mparse "}";
    mparse "";
    mparse "struct mapi_retval {";
    indent;
    mparse "uint32_t	err;";
    mparse "const char	*name;";
    deindent;
    mparse "};";
    mparse "";
    mparse "static const struct mapi_retval mapi_retval[] = {";
    indent;

    @lines = split(/\n/, $contents);
    foreach $line (@lines) {
	$line =~ s/^\#+.*$//;
	if ($line) {
	    @errors = split(/\s+/, $line);
	    mparse sprintf "{ %8s, \"%s\" },", $errors[0], $errors[1];
	}
    }


    mparse " { 0x00000000, NULL }";
    deindent;
    mparse "};";
    mparse "";
    mparse "_PUBLIC_ void mapi_errstr(const char *function, uint32_t mapi_code)";
    mparse "{";
    indent;
    mparse "struct ndr_print	ndr_print;";
    mparse "";
    mparse "ndr_print.depth = 1;";
    mparse "ndr_print.print = ndr_print_debug_helper;";
    mparse "ndr_print_MAPISTATUS(&ndr_print, function, mapi_code);";
    deindent;
    mparse "}";
    mparse "";
    mparse "_PUBLIC_ const char *mapi_get_errstr(uint32_t mapi_code)";
    mparse "{";
    indent;
    mparse "uint32_t i;";
    mparse "";
    mparse "for (i = 0; mapi_retval[i].name; i++) {";
    indent;
    mparse "if (mapi_retval[i].err == mapi_code) {";
    indent;
    mparse "return mapi_retval[i].name;";
    deindent;
    mparse "}";
    deindent;
    mparse "}";
    mparse "";
    mparse "return NULL;";
    deindent;
    mparse "}";
}

#####################################################################
# generate mapicodes_enum.idl file
sub mapicodes_enum($)
{
    my $contents = shift;
    my $line;
    my @lines;
    my @prop;
    my $prop_type;

    mparse "/* parser auto-generated by mparse */";
    mparse "";

    mparse "typedef [public, v1_enum, flag(NDR_PAHEX)] enum {";
    indent;
    
    @lines = split(/\n/, $contents);
    foreach $line (@lines) {
	$line =~ s/^\#+.*$//;
	if ($line) {
	    @prop = split(/\s+/, $line);
	    mparse sprintf "%-51s = %s,", $prop[1], $prop[0];
	}
    }
    mparse sprintf "%-51s = %s", "MAPI_E_RESERVED", "0xFFFFFFFF";
    deindent;
    mparse "} MAPISTATUS;";    
    mparse "";
    
    return $ret;
}

#####################################################################
# generate swig_mapicodes.h file
sub mapicodes_swig($)
{
    my $contents = shift;
    my $line;
    my @lines;
    my @prop;
    my $prop_type;
    my $prop_value;

    mparse "/* parser auto-generated by mparse */";
    mparse "#ifndef __SWIG_MAPICODES_H__";
    mparse "#define __SWIG_MAPICODES_H__";
    mparse "";

    @lines = split(/\n/, $contents);
    foreach $line (@lines) {
	$line =~ s/^\#+.*$//;
	if ($line) {
	    @prop = split(/\s+/, $line);
	    mparse sprintf "#define %-51s %s", $prop[1], $prop[0];
	}
    }
    mparse sprintf "#define %-51s %s", "MAPI_E_RESERVED", "0xFFFFFFFF";
    mparse "";
    mparse "#endif /* !__SWIG_MAPITAGS_H__ */";

    return $ret;  
}

sub process_file($)
{
    my $mapi_file = shift;
    my $outputdir = $opt_outputdir;

    print "Parsing $mapi_file\n";
    my $contents = FileLoad($mapi_file);
    defined $contents || return undef;

    
    if ($opt_parser eq "mapitags") {
	print "Generating $outputdir" . "mapitags.h\n";
	my $parser = ("$outputdir/mapitags.h");
	FileSave($parser, mapitags_header($contents));
	
	print "Generating $outputdir" . "mapitags.c\n";
	$ret = '';
	my $code_parser = ("$outputdir/mapitags.c");
	FileSave($code_parser, mapitags_interface($contents));
	
	print "Generating mapitags_enum.h\n";
	$ret = '';
	my $enum_parser = ("mapitags_enum.h");
	FileSave($enum_parser, mapitags_enum($contents));
    }

    if ($opt_parser eq "mapicodes") {
	print "Generating $outputdir" . "mapicode.h\n";
	my $parser = ("$outputdir/mapicode.h");
	FileSave($parser, mapicodes_header($contents));

	print "Generating $outputdir" . "mapicode.c\n";
	$ret = '';
	my $code_parser = ("$outputdir/mapicode.c");
	FileSave($code_parser, mapicodes_interface($contents));

	print "Generating mapicodes_enum.h\n";
	$ret = '';
	my $enum_parser = ("mapicodes_enum.h");
	FileSave($enum_parser, mapicodes_enum($contents));
    }

    if ($opt_parser eq "mapi_nameid") {
	print "Generating $outputdir" . "mapi_nameid_private.h\n";
	$ret = '';
	my $parser = ("$outputdir/mapi_nameid_private.h");
	FileSave($parser, mapi_nameid_private_header($contents));

	print "Generating $outputdir" . "mapi_nameid.h\n";
	$ret = '';
	my $nameid_parser = ("$outputdir/mapi_nameid.h");
	FileSave($nameid_parser, mapi_nameid_header($contents));
    }

    if ($opt_parser eq "swig_mapitags") {
	print "Generating $outputdir" . "swig_mapitags.h\n";
	$ret = '';
	my $swig_parser_tag = ("$outputdir/swig_mapitags.h");
	FileSave($swig_parser_tag, mapitags_swig($contents));
    }
    
    if ($opt_parser eq "swig_mapicodes") {
	print "Generating $outputdir" . "swig_mapicodes.h\n";
	my $swig_parser_code = ("$outputdir/swig_mapicodes.h");
	FileSave($swig_parser_code, mapicodes_swig($contents));
    }
}

process_file($_) foreach (@ARGV);
