(*
 * Compute the digest of a value.  This works the naive way:
 *    1. Convert the value to a string
 *    2. Compute its MD5 digest
 * This can be fairly expensive if the value is big.  The
 * current implementation is designed so that we can at least
 * compress the string a bit.
 *
 * ----------------------------------------------------------------
 *
 * @begin[license]
 * Copyright (C) 2005-2007 Mojave Group, Caltech
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; version 2
 * of the License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 * 
 * Additional permission is given to link this library with the
 * with the Objective Caml runtime, and to redistribute the
 * linked executables.  See the file LICENSE.OMake for more details.
 *
 * Author: Jason Hickey @email{jyh@cs.caltech.edu}
 * Modified by: Aleksey Nogin @email{anogin@hrl.com}
 * @end[license]
 *)
open Lm_printf
open Lm_symbol
open Lm_string_set
open Lm_hash

open Omake_ir
open Omake_env
open Omake_node
open Omake_shell_type
open Omake_command_type

(*
 * Codes.
 *)
(* %%MAGICBEGIN%% *)
type code =
   CodeApplyExp
 | CodeApplyString
 | CodeArgv
 | CodeArrayOfString
 | CodeArrayString
 | CodeArrow
 | CodeBegin
 | CodeBodyString
 | CodeCase
 | CodeCaseExp
 | CodeCases
 | CodeCasesExp
 | CodeCasesString
 | CodeCaseString
 | CodeCommaExp
 | CodeCommand
 | CodeCommands
 | CodeNoneString
 | CodeConstString
 | CodeEagerApply
 | CodeEnd
 | CodeExpString
 | CodeIfExp
 | CodeOpenExp
 | CodeIncludeExp
 | CodeLazyApply
 | CodeLetFunExp
 | CodeLetObjectExp
 | CodeLetThisExp
 | CodeLetVarExp
 | CodeMethodApplyExp
 | CodeMethodApplyString
 | CodeNormalApply
 | CodeQuoteString
 | CodeQuoteStringString
 | CodeExportExp
 | CodeCancelExportExp
 | CodeReturnCatchExp
 | CodeStringExp
 | CodeReturnExp
 | CodeReturnObjectExp
 | CodeReturnSaveExp
 | CodeScopeDynamic
 | CodeScopeGlobal
 | CodeScopePrivate
 | CodeScopeProtected
 | CodeSectionExp
 | CodeSequenceExp
 | CodeSequenceString
 | CodeShellExp
 | CodeStaticExp
 | CodeSpace
 | CodeSuperApplyExp
 | CodeSuperApplyString
 | CodeThisString
 | CodeValApply
 | CodeValArray
 | CodeValBody
 | CodeValData
 | CodeValDir
 | CodeValFloat
 | CodeValFun
 | CodeValFunValue
 | CodeValInt
 | CodeValMap
 | CodeValMethodApply
 | CodeValNode
 | CodeValNone
 | CodeValObject
 | CodeValPrim
 | CodeValQuote
 | CodeValQuoteString
 | CodeValSequence
 | CodeValString
 | CodeValImplicit
 | CodeValSuperApply
 | CodeVarDefApply
 | CodeVarDefNormal
 | CodeLetKeyExp
 | CodeKeyString
 | CodeKeyExp
 | CodeValKey
 | CodeArg
 | CodeArgString
 | CodeArgData
 | CodeArgNone
 | CodePipeAnd
 | CodePipeOr
 | CodePipeSequence
 | CodePipeCommand
 | CodePipeApply
 | CodePipeCond
 | CodePipeCompose
 | CodePipeGroup
 | CodePipeBackground
 | CodeCommandEnvItem
 | CodeCommandEnv
 | CodeTrue
 | CodeFalse
 | CodeQuietFlag
 | CodeAllowFailureFlag
 | CodeAllowOutputFlag
 | CodeCommandFlags
 | CodeCmdArg
 | CodeCmdNode
 | CodePipe
 | CodeRedirectNode
 | CodeRedirectArg
 | CodeRedirectNone
(* %%MAGICEND%% *)

module type HashSig =
sig
   include Lm_hash_sig.HashDigestSig

   val add_code   : t -> code -> unit
end;;

(* %%MAGICBEGIN%% *)
module Hash : HashSig =
struct
   include Lm_hash.HashDigest

   (*
    * Add a code.
    *)
   let add_code buf (code : code) =
      add_bits buf (Obj.magic code)

end;;
(* %%MAGICEND%% *)

(*
 * Variable squashing.
 *)
let squash_var buf v =
   Hash.add_string buf (string_of_symbol v)

let rec squash_vars buf vars =
   match vars with
      [v] ->
         squash_var buf v
    | v :: vars ->
         squash_var buf v;
         Hash.add_code buf CodeSpace;
         squash_vars buf vars
    | [] ->
         ()

(*
 * File.
 *)
let squash_node buf node =
   Hash.add_string buf (Node.absname node)

(*
 * String representations.
 *)
let squash_scope buf scope =
   match scope with
      ScopeGlobal ->
         Hash.add_code buf CodeScopeGlobal
    | ScopeProtected ->
         Hash.add_code buf CodeScopeProtected
    | ScopeDynamic ->
         Hash.add_code buf CodeScopeDynamic
    | ScopePrivate ->
         Hash.add_code buf CodeScopePrivate

let squash_strategy buf strategy =
   let s =
      match strategy with
         LazyApply ->
            CodeLazyApply
       | EagerApply ->
            CodeEagerApply
       | NormalApply ->
            CodeNormalApply
   in
      Hash.add_code buf s

let squash_def_kind buf kind =
   let s =
      match kind with
         VarDefNormal ->
            CodeVarDefNormal
       | VarDefAppend ->
            CodeVarDefApply
   in
      Hash.add_code buf s

(*
 * Squash string expressions.
 *)
let rec squash_string_exp pos buf e =
   Hash.add_code buf CodeBegin;
   begin
      match e with
         NoneString _ ->
            Hash.add_code buf CodeNoneString
       | ConstString (_, s) ->
            Hash.add_code buf CodeConstString;
            Hash.add_string buf s
       | KeyString (_, strategy, s) ->
            Hash.add_code buf CodeKeyString;
            squash_strategy buf strategy;
            Hash.add_string buf s
       | ApplyString (_, strategy, scope, v, sl) ->
            Hash.add_code buf CodeApplyString;
            squash_strategy buf strategy;
            Hash.add_code buf CodeSpace;
            squash_scope buf scope;
            Hash.add_code buf CodeSpace;
            squash_var buf v;
            Hash.add_code buf CodeSpace;
            squash_string_exp_list pos buf sl
       | SuperApplyString (_, strategy, scope, v1, v2, sl) ->
            Hash.add_code buf CodeSuperApplyString;
            squash_strategy buf strategy;
            Hash.add_code buf CodeSpace;
            squash_scope buf scope;
            Hash.add_code buf CodeSpace;
            squash_var buf v1;
            Hash.add_code buf CodeSpace;
            squash_var buf v2;
            Hash.add_code buf CodeSpace;
            squash_string_exp_list pos buf sl
       | MethodApplyString (_, strategy, scope, vars, sl) ->
            Hash.add_code buf CodeMethodApplyString;
            squash_strategy buf strategy;
            Hash.add_code buf CodeSpace;
            squash_scope buf scope;
            Hash.add_code buf CodeSpace;
            squash_vars buf vars;
            Hash.add_code buf CodeSpace;
            squash_string_exp_list pos buf sl
       | SequenceString (_, sl) ->
            Hash.add_code buf CodeSequenceString;
            squash_string_exp_list pos buf sl
       | ArrayString (_, sl) ->
            Hash.add_code buf CodeArrayString;
            squash_string_exp_list pos buf sl
       | ArrayOfString (_, s) ->
            Hash.add_code buf CodeArrayOfString;
            squash_string_exp pos buf s
       | QuoteString (_, sl) ->
            Hash.add_code buf CodeQuoteString;
            squash_string_exp_list pos buf sl
       | QuoteStringString (_, c, sl) ->
            Hash.add_code buf CodeQuoteStringString;
            Hash.add_char buf c;
            squash_string_exp_list pos buf sl
       | BodyString (_, e) ->
            Hash.add_code buf CodeBodyString;
            squash_exp pos buf e
       | ExpString (_, e) ->
            Hash.add_code buf CodeExpString;
            squash_exp pos buf e
       | CasesString (_, cases) ->
            Hash.add_code buf CodeCasesString;
            squash_cases_exp pos buf cases
       | ThisString (_, scope) ->
            Hash.add_code buf CodeThisString;
            squash_scope buf scope
   end;
   Hash.add_code buf CodeEnd

and squash_string_exp_list pos buf sl =
   match sl with
      [s] ->
         squash_string_exp pos buf s
    | s :: sl ->
         squash_string_exp pos buf s;
         Hash.add_code buf CodeSpace;
         squash_string_exp_list pos buf sl
    | [] ->
         ()

and squash_case_exp pos buf (v, s, e) =
   Hash.add_code buf CodeCaseString;
   squash_var buf v;
   Hash.add_code buf CodeSpace;
   squash_string_exp pos buf s;
   Hash.add_code buf CodeSpace;
   squash_exp pos buf e;
   Hash.add_code buf CodeEnd

and squash_cases_exp pos buf cases =
   Hash.add_code buf CodeCasesString;
   List.iter (squash_case_exp pos buf) cases;
   Hash.add_code buf CodeEnd

(*
 * Squash an expression.
 *)
and squash_exp pos buf e =
   Hash.add_code buf CodeBegin;
   begin
      match e with
         LetVarExp (_, scope, v, def, s) ->
            Hash.add_code buf CodeLetVarExp;
            squash_scope buf scope;
            Hash.add_code buf CodeSpace;
            squash_var buf v;
            Hash.add_code buf CodeSpace;
            squash_def_kind buf def;
            Hash.add_code buf CodeSpace;
            squash_string_exp pos buf s
       | KeyExp (_, v) ->
            Hash.add_code buf CodeKeyExp;
            Hash.add_string buf v
       | LetKeyExp (_, v, def, s) ->
            Hash.add_code buf CodeLetKeyExp;
            Hash.add_string buf v;
            Hash.add_code buf CodeSpace;
            squash_def_kind buf def;
            Hash.add_code buf CodeSpace;
            squash_string_exp pos buf s
       | LetFunExp (_, scope, v, params, s) ->
            Hash.add_code buf CodeLetFunExp;
            squash_scope buf scope;
            Hash.add_code buf CodeSpace;
            squash_var buf v;
            Hash.add_code buf CodeSpace;
            squash_vars buf params;
            Hash.add_code buf CodeSpace;
            squash_exp pos buf s
       | LetObjectExp (_, scope, v, el) ->
            Hash.add_code buf CodeLetObjectExp;
            squash_scope buf scope;
            Hash.add_code buf CodeSpace;
            squash_var buf v;
            Hash.add_code buf CodeSpace;
            squash_exp_list pos buf el
       | LetThisExp (_, s) ->
            Hash.add_code buf CodeLetThisExp;
            squash_string_exp pos buf s
       | ShellExp (_, s) ->
            Hash.add_code buf CodeShellExp;
            squash_string_exp pos buf s
       | IfExp (_, cases) ->
            Hash.add_code buf CodeIfExp;
            squash_if_cases pos buf cases
       | SequenceExp (_, el) ->
            Hash.add_code buf CodeSequenceExp;
            squash_exp_list pos buf el
       | SectionExp (_, s, el) ->
            Hash.add_code buf CodeSectionExp;
            squash_string_exp pos buf s;
            Hash.add_code buf CodeArrow;
            squash_exp_list pos buf el
       | OpenExp (_, nodes) ->
            Hash.add_code buf CodeOpenExp;
            List.iter (fun node ->
                  Hash.add_code buf CodeCommaExp;
                  squash_node buf node) nodes
       | IncludeExp (_, s, sl) ->
            Hash.add_code buf CodeIncludeExp;
            squash_string_exp pos buf s;
            Hash.add_code buf CodeCommaExp;
            squash_string_exp_list pos buf sl
       | ApplyExp (_, scope, v, sl) ->
            Hash.add_code buf CodeApplyExp;
            squash_scope buf scope;
            Hash.add_code buf CodeSpace;
            squash_var buf v;
            Hash.add_code buf CodeSpace;
            squash_string_exp_list pos buf sl
       | SuperApplyExp (_, scope, v1, v2, sl) ->
            Hash.add_code buf CodeSuperApplyExp;
            squash_scope buf scope;
            Hash.add_code buf CodeSpace;
            squash_var buf v1;
            Hash.add_code buf CodeSpace;
            squash_var buf v2;
            Hash.add_code buf CodeSpace;
            squash_string_exp_list pos buf sl
       | MethodApplyExp (_, scope, vars, sl) ->
            Hash.add_code buf CodeMethodApplyExp;
            squash_scope buf scope;
            Hash.add_code buf CodeSpace;
            squash_vars buf vars;
            Hash.add_code buf CodeSpace;
            squash_string_exp_list pos buf sl
       | ExportExp (_, s) ->
            Hash.add_code buf CodeExportExp;
            squash_string_exp pos buf s
       | CancelExportExp _ ->
            Hash.add_code buf CodeCancelExportExp
       | ReturnCatchExp (_, e) ->
            Hash.add_code buf CodeReturnCatchExp;
            squash_exp pos buf e
       | StringExp (_, s) ->
            Hash.add_code buf CodeStringExp;
            squash_string_exp pos buf s
       | ReturnExp (_, s) ->
            Hash.add_code buf CodeReturnExp;
            squash_string_exp pos buf s
       | ReturnObjectExp (_, vars) ->
            Hash.add_code buf CodeReturnObjectExp;
            squash_vars buf vars
       | ReturnSaveExp _ ->
            Hash.add_code buf CodeReturnSaveExp
       | StaticExp (_, node, key, el) ->
            Hash.add_code buf CodeStaticExp;
            squash_node buf node;
            Hash.add_code buf CodeSpace;
            squash_var buf key;
            Hash.add_code buf CodeSpace;
            squash_exp_list pos buf el
   end;
   Hash.add_code buf CodeEnd

and squash_exp_list pos buf el =
   match el with
      [e] ->
         squash_exp pos buf e
    | e :: el ->
         squash_exp pos buf e;
         Hash.add_code buf CodeSpace;
         squash_exp_list pos buf el
    | [] ->
         ()

and squash_if_case pos buf (s, e) =
   Hash.add_code buf CodeCaseExp;
   squash_string_exp pos buf s;
   Hash.add_code buf CodeSpace;
   squash_exp pos buf e;
   Hash.add_code buf CodeEnd

and squash_if_cases pos buf cases =
   Hash.add_code buf CodeCasesExp;
   List.iter (squash_if_case pos buf) cases;
   Hash.add_code buf CodeEnd

(*
 * Compute the digest of a value.
 *)
let rec squash_value pos buf v =
   Hash.add_code buf CodeBegin;
   begin
      match v with
         ValNone ->
            Hash.add_code buf CodeValNone;
       | ValInt i ->
            Hash.add_code buf CodeValInt;
            Hash.add_int buf i
       | ValFloat x ->
            Hash.add_code buf CodeValFloat;
            Hash.add_float buf x
       | ValSequence vl ->
            Hash.add_code buf CodeValSequence;
            squash_values pos buf vl
       | ValArray vl ->
            Hash.add_code buf CodeValArray;
            squash_values pos buf vl
       | ValString s ->
            Hash.add_code buf CodeValString;
            Hash.add_string buf s
       | ValData s ->
            Hash.add_code buf CodeValData;
            Hash.add_string buf s
       | ValQuote vl ->
            Hash.add_code buf CodeValQuote;
            squash_values pos buf vl
       | ValQuoteString (c, vl) ->
            Hash.add_code buf CodeValQuoteString;
            Hash.add_char buf c;
            squash_values pos buf vl
       | ValApply (_, scope, v, vl) ->
            Hash.add_code buf CodeValApply;
            squash_scope buf scope;
            Hash.add_code buf CodeSpace;
            squash_var buf v;
            Hash.add_code buf CodeSpace;
            squash_values pos buf vl
       | ValImplicit (_, scope, v) ->
            Hash.add_code buf CodeValImplicit;
            squash_scope buf scope;
            Hash.add_code buf CodeSpace;
            squash_var buf v;
       | ValSuperApply (_, scope, v1, v2, vl) ->
            Hash.add_code buf CodeValSuperApply;
            squash_scope buf scope;
            Hash.add_code buf CodeSpace;
            squash_var buf v1;
            Hash.add_code buf CodeSpace;
            squash_var buf v2;
            Hash.add_code buf CodeSpace;
            squash_values pos buf vl
       | ValMethodApply (_, scope, vars, vl) ->
            Hash.add_code buf CodeValMethodApply;
            squash_scope buf scope;
            Hash.add_code buf CodeSpace;
            squash_vars buf vars;
            Hash.add_code buf CodeSpace;
            squash_values pos buf vl
       | ValFun (_, _, params, body) ->
            Hash.add_code buf CodeValFun;
            squash_vars buf params;
            Hash.add_code buf CodeArrow;
            squash_exp pos buf body
       | ValFunValue (_, _, params, body) ->
            Hash.add_code buf CodeValFunValue;
            squash_vars buf params;
            Hash.add_code buf CodeArrow;
            squash_value pos buf body
       | ValPrim (_, _, f) ->
            Hash.add_code buf CodeValPrim;
            squash_var buf (squash_prim_fun f)
       | ValNode node ->
            Hash.add_code buf CodeValNode;
            Hash.add_string buf (Node.fullname node)
       | ValDir dir ->
            Hash.add_code buf CodeValDir;
            Hash.add_string buf (Dir.fullname dir)
       | ValBody (_, e) ->
            Hash.add_code buf CodeValBody;
            squash_exp pos buf e
       | ValObject obj ->
            Hash.add_code buf CodeValObject;
            squash_object pos buf obj
       | ValMap obj ->
            Hash.add_code buf CodeValMap;
            squash_map pos buf obj
       | ValCases cases ->
            squash_cases pos buf cases
       | ValKey (_, v) ->
            Hash.add_code buf CodeValKey;
            Hash.add_string buf v
       | ValEnv _
       | ValRules _
       | ValChannel _
       | ValClass _
       | ValOther _ as v ->
            let print_error buf =
               fprintf buf "@[<v 3>Non digestable value:@ @[<hv 3>%a@]@ Contact the OMake team at omake@@metaprl.org if you think this should be supported@]@." pp_print_value v
            in
               raise (OmakeFatalErr (pos, LazyError print_error))
   end;
   Hash.add_code buf CodeEnd

and squash_values pos buf vl =
   match vl with
      [v] ->
         squash_value pos buf v
    | v :: vl ->
         squash_value pos buf v;
         Hash.add_code buf CodeSpace;
         squash_values pos buf vl
    | [] ->
         ()

and squash_object pos buf obj =
   SymbolTable.iter (fun x v ->
         Hash.add_code buf CodeBegin;
         squash_var buf x;
         Hash.add_code buf CodeArrow;
         squash_value pos buf v;
         Hash.add_code buf CodeEnd) (Omake_env.squash_object obj)

and squash_map pos buf map =
   venv_map_iter (fun x v ->
         Hash.add_code buf CodeBegin;
         squash_value pos buf x;
         Hash.add_code buf CodeArrow;
         squash_value pos buf v;
         Hash.add_code buf CodeEnd) map

and squash_case pos buf (x, v1, x2) =
   Hash.add_code buf CodeCase;
   squash_var buf x;
   Hash.add_code buf CodeSpace;
   squash_value pos buf v1;
   Hash.add_code buf CodeSpace;
   squash_value pos buf v1;
   Hash.add_code buf CodeEnd

and squash_cases pos buf cases =
   Hash.add_code buf CodeCases;
   List.iter (squash_case pos buf) cases;
   Hash.add_code buf CodeEnd

(*
 * Commands.
 *)
let squash_command_flag buf flag =
   let code =
      match flag with
         QuietFlag ->
            CodeQuietFlag
       | AllowFailureFlag ->
            CodeAllowFailureFlag
       | AllowOutputFlag ->
            CodeAllowOutputFlag
   in
      Hash.add_code buf code

let squash_command_flags buf flags =
   Hash.add_code buf CodeCommandFlags;
   List.iter (squash_command_flag buf) flags;
   Hash.add_code buf CodeEnd

let squash_arg_string buf arg =
   match arg with
      ArgString s ->
         Hash.add_code buf CodeArgString;
         Hash.add_string buf s
    | ArgData s ->
         Hash.add_code buf CodeArgData;
         Hash.add_string buf s

let squash_arg buf arg =
   Hash.add_code buf CodeArg;
   List.iter (squash_arg_string buf) arg;
   Hash.add_code buf CodeEnd

let squash_redirect buf chan =
   match chan with
      RedirectNode node ->
         Hash.add_code buf CodeRedirectNode;
         squash_node buf node
    | RedirectArg arg ->
         Hash.add_code buf CodeRedirectArg;
         squash_arg buf arg
    | RedirectNone ->
         Hash.add_code buf CodeRedirectNone

let squash_argv buf argv =
   Hash.add_code buf CodeArgv;
   List.iter (squash_arg buf) argv;
   Hash.add_code buf CodeEnd

let squash_command_env_item buf (v, arg) =
   Hash.add_code buf CodeCommandEnvItem;
   squash_var buf v;
   Hash.add_code buf CodeSpace;
   squash_arg buf arg;
   Hash.add_code buf CodeEnd

let squash_command_env buf env =
   Hash.add_code buf CodeCommandEnv;
   List.iter (squash_command_env_item buf) env;
   Hash.add_code buf CodeEnd

let squash_exe buf exe =
   match exe with
      CmdArg arg ->
         Hash.add_code buf CodeCmdArg;
         squash_arg buf arg
    | CmdNode node ->
         Hash.add_code buf CodeCmdNode;
         squash_node buf node

let squash_pipe_op buf op =
   let code =
      match op with
         PipeAnd -> CodePipeAnd
       | PipeOr  -> CodePipeOr
       | PipeSequence -> CodePipeSequence
   in
      Hash.add_code buf code

let squash_pipe_command pos buf (info : arg_cmd) =
   let { cmd_env   = env;
         cmd_exe   = exe;
         cmd_argv  = argv;
         cmd_stdin = stdin;
         cmd_stdout = stdout;
         cmd_stderr = stderr;
         cmd_append = append
       } = info
   in
      Hash.add_code buf CodePipeCommand;
      squash_command_env buf env;
      Hash.add_code buf CodeSpace;
      squash_exe buf exe;
      Hash.add_code buf CodeSpace;
      squash_argv buf argv;
      Hash.add_code buf CodeSpace;
      squash_redirect buf stdin;
      Hash.add_code buf CodeSpace;
      squash_redirect buf stdout;
      Hash.add_code buf CodeSpace;
      Hash.add_bool buf stderr;
      Hash.add_code buf CodeSpace;
      Hash.add_bool buf append;
      Hash.add_code buf CodeEnd

let squash_pipe_apply pos buf (info : arg_apply) =
   let { apply_name = name;
         apply_args = args;
         apply_stdin = stdin;
         apply_stdout = stdout;
         apply_stderr = stderr;
         apply_append = append
       } = info
   in
      Hash.add_code buf CodePipeApply;
      squash_var buf name;
      Hash.add_code buf CodeSpace;
      squash_values pos buf args;
      Hash.add_code buf CodeSpace;
      squash_redirect buf stdin;
      Hash.add_code buf CodeSpace;
      squash_redirect buf stdout;
      Hash.add_code buf CodeSpace;
      Hash.add_bool buf stderr;
      Hash.add_code buf CodeSpace;
      Hash.add_bool buf append;
      Hash.add_code buf CodeEnd

let rec squash_pipe pos buf (pipe : arg_pipe) =
   (match pipe with
       PipeApply (_, info) ->
          squash_pipe_apply pos buf info
     | PipeCommand (_, info) ->
          squash_pipe_command pos buf info
     | PipeCond (_, op, pipe1, pipe2) ->
          Hash.add_code buf CodePipeCond;
          squash_pipe_op buf op;
          squash_pipe pos buf pipe1;
          squash_pipe pos buf pipe2
     | PipeCompose (_, b, pipe1, pipe2) ->
          Hash.add_code buf CodePipeCompose;
          Hash.add_bool buf b;
          squash_pipe pos buf pipe1;
          squash_pipe pos buf pipe2
     | PipeGroup (_, info) ->
          squash_pipe_group pos buf info
     | PipeBackground (_, pipe) ->
          Hash.add_code buf CodePipeBackground;
          squash_pipe pos buf pipe);
   Hash.add_code buf CodeEnd

and squash_pipe_group pos buf info =
   let { group_stdin = stdin;
         group_stdout = stdout;
         group_stderr = stderr;
         group_append = append;
         group_pipe   = pipe
       } = info
   in
      Hash.add_code buf CodePipeGroup;
      squash_redirect buf stdin;
      Hash.add_code buf CodeSpace;
      squash_redirect buf stdout;
      Hash.add_code buf CodeSpace;
      Hash.add_bool buf stderr;
      Hash.add_code buf CodeSpace;
      Hash.add_bool buf append;
      Hash.add_code buf CodeSpace;
      squash_pipe pos buf pipe;
      Hash.add_code buf CodeEnd

let squash_command_line pos buf (command : arg_command_inst) =
   match command with
      CommandPipe argv ->
         Hash.add_code buf CodePipe;
         squash_pipe pos buf argv;
         Hash.add_code buf CodeEnd
    | CommandEval e ->
         squash_exp pos buf e
    | CommandValues values ->
         squash_values pos buf values

let squash_command pos buf (command : arg_command_line) =
   let { command_dir = dir;
         command_inst = inst
       } = command
   in
      Hash.add_code buf CodeCommand;
      Hash.add_string buf (Dir.fullname dir);
      squash_command_line pos buf inst;
      Hash.add_code buf CodeEnd

let squash_commands pos buf commands =
   Hash.add_code buf CodeCommands;
   List.iter (squash_command pos buf) commands;
   Hash.add_code buf CodeEnd

(*
 * Get the digest of some commands.
 *)
let digest_of_commands pos commands =
   match commands with
      [] ->
         None
    | _ ->
         let buf = Hash.create () in
         let () = squash_commands pos buf commands in
            Some (Hash.digest buf)

(*
 * -*-
 * Local Variables:
 * End:
 * -*-
 *)
