/* This file is part of Om.  Copyright (C) 2005 Dave Robillard.
 * 
 * Om is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 * 
 * Om is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "EnablePatchEvent.h"
#include "Om.h"
#include "OmApp.h"
#include "util.h"
#include "OSCSender.h"
#include "ObjectStore.h"

namespace Om {


EnablePatchEvent::EnablePatchEvent(Request* request, const string& patch_path)
: SlowEvent(request),
  m_patch_path(patch_path),
  m_patch(NULL),
  m_process_order(NULL),
  m_acquired_mutex(false)
{
}


void
EnablePatchEvent::prepare()
{
	m_patch = om->object_store()->find_patch(m_patch_path);
	
	if (m_patch != NULL) {
		m_patch->node_remove_mutex().soft_lock();
		m_acquired_mutex = true;

		/* Any event that requires a new process order will set the patch's
		 * process order to NULL if it is executed when the patch is not
		 * active.  So, if the PO is NULL, calculate it here */
		if (m_patch->process_order() == NULL)
			m_process_order = Om::find_process_order(m_patch);
	}

	SlowEvent::prepare();
}


void
EnablePatchEvent::execute(uint sample_offset)
{
	if (m_patch != NULL) {
		m_patch->process(true);

		if (m_patch->process_order() == NULL)
			m_patch->process_order(m_process_order);
	}
	
	SlowEvent::execute(sample_offset);
}


void
EnablePatchEvent::post_process()
{
	if (m_patch != NULL) {
		if (m_acquired_mutex)
			m_patch->node_remove_mutex().soft_unlock();
		m_request->respond_ok();
		om->osc_sender()->send_patch_enable(m_patch_path);
	} else {
		m_request->respond_error("Patch not found");
	}
}


} // namespace Om

