/* This file is part of Om.  Copyright (C) 2005 Dave Robillard.
 * 
 * Om is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 * 
 * Om is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#ifndef PATCHCONTROLLER_H
#define PATCHCONTROLLER_H

#include <string>
#include <gtkmm.h>
#include "OmGtkObject.h"

namespace LibOmClient {
class PatchModel;
class NodeModel;
class PortModel;
class MetadataModel;
class ControlModel;
class ConnectionModel;
}

using std::string;
using namespace LibOmClient;

namespace OmGtk {
	
class PatchWindow;
class SubpatchModule;
class Controller;
class OmPatchBayArea;
class NodeControlWindow;
class ControlPanel;


/** Controller for a patch.
 *
 * A patch is different from a port or node because there are two
 * representations in the Gtk client - the window and the module in the parent
 * patch (if applicable).  So, this is a master class that contains both of
 * those representations, and acts as the recipient of all patch related
 * events (ie it is the OmGtkObject for a patch).
 * 
 * \ingroup OmGtk
 */
class PatchController : public OmGtkObject
{
public:
	PatchController(PatchModel* patch_model, Controller* controller, PatchController* parent, bool load_widgets=true);
	virtual ~PatchController();

	void new_node(NodeModel* const nm);
	void remove_node(const string& name);
	void new_port(PortModel* const pm);
	void remove_port(const string& path);
	void connection(ConnectionModel* const cm);
	void disconnection(const string& port1_path, const string& port2_path);
	void metadata_update(const MetadataModel* const mm);

	void new_subpatch(SubpatchModule* module);

	void get_new_module_location(int& x, int& y);

	void show_control_window();
	void claim_control_panel();

	SubpatchModule*    module() const             { return m_module; }
	void               module(SubpatchModule* sm);
	PatchWindow*       window() const             { return m_window; }
	void               window(PatchWindow* pw)    { m_window = pw; }
	NodeControlWindow* control_window()          { return m_control_window; }
	ControlPanel*      control_panel()            { return m_control_panel; }
	
	void path(const string& new_path);
	void enabled(bool e);
	
	PatchController*      parent()                      { return m_parent; }
	OmPatchBayArea*       patch_bay()                   { return m_patch_bay; }
	void                  patch_bay(OmPatchBayArea* pb) { m_patch_bay = pb; }
	PatchModel*           model()                       { return m_patch_model; }

private:
	PatchController* m_parent;
	PatchModel*      m_patch_model;
	OmPatchBayArea*  m_patch_bay;
	Controller*      m_controller;

	PatchWindow*     m_window;
	SubpatchModule*  m_module;

	/** Control window that will be shown when double-clicking a SubpatchModule */
	NodeControlWindow* m_control_window;

	/** Control panel for the main window */
	ControlPanel* m_control_panel;

	/** Invisible bin used to store control panel when not shown by main window */
	Gtk::Alignment m_control_panel_bin;

	// Coordinates for next added plugin (used by canvas menu)
	// 0 means "not set", ie guess at the best location
	int m_module_x;
	int m_module_y;
};


} // namespace OmGtk

#endif // PATCHCONTROLLER_H
