/* Copyright 2011 Canonical, Ltd. This software is licensed under the GNU
 * Lesser General Public License version 3 or later (see the file COPYING).
 */

#include "geissingleton.h"

#include <QApplication>
#include <QDeclarativeItem>
#include <QDesktopWidget>
#include <QGraphicsScene>
#include <QGraphicsView>
#include <QGraphicsWidget>
#include <QSocketNotifier>

#include <typeinfo>

#include "attributes.h"
#include "continuousgesturearea.h"
#include "device.h"
#include "gesture.h"
#include "gesturearea.h"
#include "gestureevent.h"

using geis_attributes::GetAttribute;

/******************************************************************************
 * Functions for handling the GEIS singleton instance
 ******************************************************************************/
namespace {

Q_GLOBAL_STATIC(GeisSingleton, g_geis_singleton)

}  // namespace

GeisSingleton* GeisSingleton::Instance() {
  return g_geis_singleton();
}

GeisSingleton::GeisSingleton(QObject* parent)
  : QObject(parent),
    initialized_(false) {
  geis_ = geis_new(GEIS_INIT_TRACK_DEVICES,
                   GEIS_INIT_TRACK_GESTURE_CLASSES,
                   NULL);
  if (!geis_)
    qCritical("Failed to initialize geis instance");

  int fd;
  geis_get_configuration(geis_, GEIS_CONFIGURATION_FD, &fd);

  notifier_ = new QSocketNotifier(fd, QSocketNotifier::Read, this);
  connect(notifier_, SIGNAL(activated(int)), SLOT(GeisEventSlot()));
}

/******************************************************************************/

void GeisSingleton::RegisterGestureArea(GestureArea* area) {
  if (!window_map_.contains(area->window_id(), area))
    window_map_.insert(area->window_id(), area);
}

void GeisSingleton::UnregisterGestureArea(GestureArea* area) {
  window_map_.remove(area->window_id(), area);
}

void GeisSingleton::GeisEventSlot() {
  GeisStatus status;

  status = geis_dispatch_events(geis_);
  if (status != GEIS_STATUS_SUCCESS) {
    qWarning("Failed to dispatch geis events");
    return;
  }

  GeisEvent event;
  for (status = geis_next_event(geis_, &event);
       status == GEIS_STATUS_SUCCESS || status == GEIS_STATUS_CONTINUE;
       status = geis_next_event(geis_, &event)) {
    switch (geis_event_type(event)) {
      case GEIS_EVENT_CLASS_AVAILABLE:
      case GEIS_EVENT_CLASS_CHANGED:
      case GEIS_EVENT_CLASS_UNAVAILABLE:
        HandleClassEvent(event);
        break;

      case GEIS_EVENT_DEVICE_AVAILABLE:
      case GEIS_EVENT_DEVICE_UNAVAILABLE:
        HandleDeviceEvent(event);
        break;

      case GEIS_EVENT_INIT_COMPLETE:
        initialized_ = true;
        emit Initialized();
        break;

      case GEIS_EVENT_GESTURE_BEGIN:
        HandleGestureBeginEvent(event);
        break;

      case GEIS_EVENT_GESTURE_UPDATE:
        HandleGestureUpdateEvent(false, event);
        break;

      case GEIS_EVENT_GESTURE_END:
        HandleGestureUpdateEvent(true, event);
        break;

      default:
        break;
    }

    geis_event_delete(event);
  }
}

void GeisSingleton::HandleClassEvent(GeisEvent event) {
  GeisGestureClass gesture_class = GetAttribute< GeisGestureClass, GeisEvent >(
    event,
    GEIS_EVENT_ATTRIBUTE_CLASS);

  GeisString name = geis_gesture_class_name(gesture_class);
  if (!name) {
    qCritical("Failed to get name of gesture class");
    return;
  }

  GestureArea::Primitive primitive;
  if (QString::compare(name, GEIS_GESTURE_DRAG) == 0)
    primitive = GestureArea::kDrag;
  else if (QString::compare(name, GEIS_GESTURE_PINCH) == 0)
    primitive = GestureArea::kPinch;
  else if (QString::compare(name, GEIS_GESTURE_ROTATE) == 0)
    primitive = GestureArea::kRotate;
  else if (QString::compare(name, GEIS_GESTURE_TAP) == 0)
    primitive = GestureArea::kTap;
  else
    return;

  switch (geis_event_type(event)) {
    case GEIS_EVENT_CLASS_AVAILABLE:
      geis_gesture_class_ref(gesture_class);
      classes_[primitive] = gesture_class;
      break;

    case GEIS_EVENT_CLASS_UNAVAILABLE:
      classes_.remove(primitive);
      geis_gesture_class_unref(gesture_class);
      break;

    default:
      break;
  }
}

void GeisSingleton::HandleDeviceEvent(GeisEvent event) {
  GeisDevice geis_device = GetAttribute< GeisDevice, GeisEvent >(
    event,
    GEIS_EVENT_ATTRIBUTE_DEVICE);

  if (geis_event_type(event) == GEIS_EVENT_DEVICE_AVAILABLE) {
    Device* device = new Device(geis_device);
    device_map_[device->id()] = device;
  } else if (geis_event_type(event) == GEIS_EVENT_DEVICE_UNAVAILABLE) {
    GeisInteger id = geis_device_id(geis_device);
    delete device_map_[id];
    device_map_.remove(id);
  }
}

namespace {

/* Get a list of gesture frames from the groupset. The groupset mechanism
 * doesn't really work, so we flatten everything down into a list. The list
 * must be in order from most touches to least.
 */
void GetGestureFrameList(GeisGroupSet group_set, QList< GeisFrame >* frames) {
  QMultiMap< int, GeisFrame > frames_by_touch_count;
  QList< GeisFrame > handled_gestures;
  for (GeisSize i = 0; i < geis_groupset_group_count(group_set); ++i) {
    GeisGroup group = geis_groupset_group(group_set, i);

    for (GeisSize j = 0; j < geis_group_frame_count(group); ++j) {
      GeisFrame frame = geis_group_frame(group, j);

      if (handled_gestures.contains(frame))
        continue;

      frames_by_touch_count.insert(geis_frame_touchid_count(frame), frame);
      handled_gestures.append(frame);
    }
  }

  for (QMultiMap< int, GeisFrame >::ConstIterator it =
         frames_by_touch_count.constBegin();
       it != frames_by_touch_count.constEnd();
       ++it)
    frames->prepend(*it);
}

QPointF PointForTouch(GeisTouch touch) {
  GeisFloat x = GetAttribute< GeisFloat, GeisTouch >(touch,
                                                     GEIS_TOUCH_ATTRIBUTE_X);
  GeisFloat y = GetAttribute< GeisFloat, GeisTouch >(touch,
                                                     GEIS_TOUCH_ATTRIBUTE_Y);

  return QPointF(x, y);
}

QRectF RectForTouches(const QSet< GeisTouch >& touches) {
  QPointF min(0, 0);
  QPointF max(0, 0);
  bool first = true;
  foreach (GeisTouch touch, touches) {
    QPointF point(PointForTouch(touch));

    if (first) {
      min = point;
      max = point;
      first = false;
    } else {
      if (point.x() < min.x())
        min.setX(point.x());
      else if (point.x() > max.x())
        max.setX(point.x());
      if (point.y() < min.y())
        min.setY(point.y());
      else if (point.y() > max.y())
        max.setY(point.y());
    }
  }

  return QRectF(min, max);
}

QRectF RectForFocus(GeisFrame frame) {
  GeisFloat x = GetAttribute< GeisFloat, GeisFrame >(
    frame,
    GEIS_GESTURE_ATTRIBUTE_FOCUS_X);
  GeisFloat y = GetAttribute< GeisFloat, GeisFrame >(
    frame,
    GEIS_GESTURE_ATTRIBUTE_FOCUS_Y);

  return QRectF(x, y, 0, 0);
}

bool ContainsClaimedTouches(const QSet< GeisTouch >& touches,
                            const QSet< GeisTouch >& claimed_touches) {
  foreach (GeisTouch touch, touches)
    if (claimed_touches.contains(touch))
      return true;

  return false;
}

bool ContainsOtherUnclaimedTouches(const GestureArea* area,
                                   const QSet< GeisTouch >& touches,
                                   const QSet< GeisTouch >& unclaimed_touches) {
  foreach (GeisTouch touch, unclaimed_touches) {
    if (touches.contains(touch))
      continue;

    if (area->contains(PointForTouch(touch)))
      return true;
  }

  return false;
}

}  // namespace

QGraphicsScene* GeisSingleton::FindSceneForTouches(WId window,
                                                   const QRectF& touches) {
  foreach (const GestureArea* area, window_map_.values(window)) {
    QGraphicsScene* scene = area->scene();
    QGraphicsView* view = scene->views().first();
    QRectF bounds(view->mapToGlobal(QPoint(0, 0)), view->size());

    /* touches may be null if it consists of only one point */
    if (touches.isValid()) {
      if (bounds.contains(touches))
        return scene;
    } else if (bounds.contains(touches.topLeft())) {
      return scene;
    }
  }

  return NULL;
}

void GeisSingleton::HandleGestureBeginEvent(GeisEvent event) {
  GeisTouchSet touch_set = GetAttribute< GeisTouchSet, GeisEvent >(
    event,
    GEIS_EVENT_ATTRIBUTE_TOUCHSET);
  GeisGroupSet group_set = GetAttribute< GeisGroupSet, GeisEvent >(
    event,
    GEIS_EVENT_ATTRIBUTE_GROUPSET);

  QList< GeisFrame > frames;
  GetGestureFrameList(group_set, &frames);

  QSet< GeisTouch > unclaimed_touches;
  for (GeisSize i = 0; i < geis_touchset_touch_count(touch_set); ++i)
    unclaimed_touches.insert(geis_touchset_touch(touch_set, i));

  QSet< GeisTouch > claimed_touches;
  foreach (GeisFrame frame, frames) {
    GeisInteger device_id = GetAttribute< GeisInteger, GeisFrame >(
      frame,
      GEIS_GESTURE_ATTRIBUTE_DEVICE_ID);
    WId window = GetAttribute< GeisInteger, GeisFrame >(
      frame,
      GEIS_GESTURE_ATTRIBUTE_EVENT_WINDOW_ID);

    QSet< GeisTouch > touches;
    for (GeisSize i = 0; i < geis_frame_touchid_count(frame); ++i) {
      GeisTouchId id = geis_frame_touchid(frame, i);
      GeisTouch touch = geis_touchset_touch_by_id(touch_set, id);
      touches.insert(touch);
    }

    if (ContainsClaimedTouches(touches, claimed_touches)) {
      geis_gesture_reject(geis_frame_id(frame));
      continue;
    }

    QList< GestureArea::Primitive > primitives;
    for (QHash< GestureArea::Primitive, GeisGestureClass >::ConstIterator it =
           classes_.constBegin();
         it != classes_.constEnd();
         ++it)
      if (geis_frame_is_class(frame, it.value()))
        primitives.append(it.key());

    QList< QGraphicsItem* > items;
    if (window == QApplication::desktop()->winId())
      foreach (GestureArea* area, window_map_.values(window))
        items.append(area);

    QRectF touches_rect;
    if (device_map_[device_id]->device_type() == Device::TouchScreen)
      touches_rect = RectForTouches(touches);
    else
      touches_rect = RectForFocus(frame);

    QGraphicsScene* scene = FindSceneForTouches(window, touches_rect);
    if (!scene && items.empty()) {
      geis_gesture_reject(geis_frame_id(frame));
      continue;
    }

    if (scene) {
      QPoint offset(scene->views().first()->mapFromGlobal(QPoint(0, 0)));
      touches_rect.translate(offset);

      if (touches_rect.isValid())
        items.append(scene->items(touches_rect));
      else
        items.append(scene->items(touches_rect.topLeft()));
    }

    bool gesture_handled = false;
    foreach (QGraphicsItem* item, items) {
      GestureArea* area = qobject_cast< GestureArea* >(item);
      if (!area)
        continue;

      if (touches.count() != area->subscription()->touches()->start())
          continue;

      if (!area->subscription()->_UNITY_globalGesture()) {
        /* We have an area that intersects with the box, now check if the
         * area contains the box.
         */
        if (touches_rect.isValid() && !area->shape().contains(touches_rect))
          continue;

        if (ContainsOtherUnclaimedTouches(area, touches, unclaimed_touches))
          continue;
      }

      Device::DeviceType type = device_map_[device_id]->device_type();
      if (area->subscription()->devices() != Gesture::All &&
          ((type == Device::TouchScreen &&
            area->subscription()->devices() != Gesture::TouchScreens) ||
           (type == Device::TouchPad &&
            area->subscription()->devices() != Gesture::TouchPads) ||
           (type == Device::Independent &&
            area->subscription()->devices() != Gesture::Independents)))
        continue;

      if (!primitives.contains(area->primitive()))
        continue;

      GestureEvent event(area, frame, touch_set, device_map_[device_id]);
      if (area->IsGestureEventHandled(&event)) {
        GestureArea::Primitive primitive = area->primitive();

        primitives.removeAll(primitive);
        gesture_handled = true;

        /* Don't keep instantaneous gestures around as active */
        if (qobject_cast<ContinuousGestureArea*>(area))
          active_gestures_[geis_frame_id(frame)][primitive] = area;
      }
    }

    if (gesture_handled)
      /* FIXME: Need a way to accept a gesture frame */
      claimed_touches.unite(touches);
    else
      geis_gesture_reject(geis_frame_id(frame));
  }
}

void GeisSingleton::HandleGestureUpdateEvent(bool end, GeisEvent event) {
  GeisTouchSet touch_set = GetAttribute< GeisTouchSet, GeisEvent >(
    event,
    GEIS_EVENT_ATTRIBUTE_TOUCHSET);
  GeisGroupSet group_set = GetAttribute< GeisGroupSet, GeisEvent >(
    event,
    GEIS_EVENT_ATTRIBUTE_GROUPSET);

  QList< GeisFrame > frames;
  GetGestureFrameList(group_set, &frames);

  foreach (GeisFrame frame, frames) {
    GeisGestureId id = geis_frame_id(frame);
    GeisInteger device_id = GetAttribute< GeisInteger, GeisFrame >(
      frame,
      GEIS_GESTURE_ATTRIBUTE_DEVICE_ID);

    QList< GestureArea::Primitive > primitives;
    for (QHash< GestureArea::Primitive, GeisGestureClass >::ConstIterator it =
           classes_.constBegin();
         it != classes_.constEnd();
         ++it)
      if (geis_frame_is_class(frame, it.value()))
        primitives.append(it.key());

    /* Tap events only send update events. Handle them like begin events. */
    if (primitives.length() == 1 && primitives.first() == GestureArea::kTap &&
        frames.length() == 1) {
      HandleGestureBeginEvent(event);
      return;
    }

    foreach (GestureArea::Primitive primitive, primitives) {
      if (!active_gestures_.contains(id) ||
          !active_gestures_[id].contains(primitive))
        continue;

      GestureArea* area = active_gestures_[geis_frame_id(frame)][primitive];
      GestureEvent event(area, frame, touch_set, device_map_[device_id]);
      area->HandleGestureUpdateEvent(end, &event);

      if (end) {
        active_gestures_[id].remove(primitive);
        if (active_gestures_[id].empty())
          active_gestures_.remove(id);
      }
    }
  }
}

GeisSingleton::~GeisSingleton() {
  delete notifier_;

  foreach (GeisGestureClass gesture_class, classes_)
    geis_gesture_class_unref(gesture_class);

  foreach (const Device* device, device_map_)
    delete device;

  if (geis_)
    geis_delete(geis_);
}
