/***************************************************************************
                          jdxblock.h  -  description
                             -------------------
    begin                : Sun Jun 6 2004
    copyright            : (C) 2000-2014 by Thies H. Jochimsen
    email                : thies@jochimsen.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef JDXBLOCK_H
#define JDXBLOCK_H

#include <tjutils/tjstatic.h>
#include <tjutils/tjlist.h>

#include <odinpara/jdxbase.h>
#include <odinpara/jdxtypes.h>

/**
  * @addtogroup jcampdx
  * @{
  */


typedef List<JcampDxClass,JcampDxClass*,JcampDxClass&> JDXList; // alias


/**
  *
  *  This class represents a block of parameters. Parameters can be
  *  added or removed from the list of parameters within the block.
  *  The whole block may be written to or read from a file.
  */
class JcampDxBlock : public virtual JcampDxClass, public JDXList, public StaticHandler<JcampDxBlock> {

 public:

/**
  *
  *  Constructs an empty block of parameters with the label 'title' and the specified mode
  */
  JcampDxBlock(const STD_string& title="Parameter List",compatMode mode=notBroken);


/**
  *
  *  Copy constructor
  */
  JcampDxBlock(const JcampDxBlock& block);


/**
  *
  *  Destructor
  */
  ~JcampDxBlock();


/**
  *
  *  Assignment operator
  */
  JcampDxBlock& operator = (const JcampDxBlock& block);

/**
  *
  *  Merges all parameters found in the specified 'block' into this block
  */
  JcampDxBlock& merge(JcampDxBlock& block, bool onlyUserPars=true);

/**
  *
  *  Removes all parameters found in the specified 'block' from this block
  */
  JcampDxBlock& unmerge(JcampDxBlock& block);


/**
  *
  *  Print the value of the parameter that has the label 'parameterName' as a formated string.
  *  If applicable, append the unit of the number in case 'append_unit' is set to 'true'.
  */
  STD_string printval(const STD_string& parameterName, bool append_unit=false) const;

/**
  *
  * Set the value of the parameter that has the label 'parameterName' to 'value'.
  * Returns 'true' if sucessful.
  */
  bool parseval(const STD_string& parameterName, const STD_string& value);


/**
  *
  *  Parse values of thius block from the string 'src' in JCAMP-DX format,
  * returns the number of parameters sucessfully parsed.
  */
  int parseblock(const STD_string& source);

/**
  *
  *  Returns the current number of parameters in the block
  */
  unsigned int numof_pars() const;


/**
  * Returns a pointer to parameter 'label', or zero if not found
  */
  JcampDxClass* get_parameter(const STD_string& ldrlabel);


/**
  *
  *  Returns true if a parameter with the given label already exists
  *  in the block
  */
  bool parameter_exists(const STD_string& ldrlabel) const;

/**
  *
  *  Prefixes all parameters labels with the given string. If the prefix
  *  is already at the beginning of the label, it is not added again.
  */
  JcampDxBlock& set_prefix(const STD_string& prefix);

/**
  *
  *  If embedded is true, the block's widget will be displayed in a subwidget
  *  otherwise an 'Edit' button that will open a subdialog is generated
  */
  JcampDxBlock& set_embedded(bool embedded) {embed=embedded; return *this;}

/**
  *
  *  Returns whether the block's widget will be displayed in a subwidget
  *  or an 'Edit' button that will open a subdialog is generated
  */
  bool is_embedded() const {return embed;}




/**
  * Returns the i'th parameter in the block
  */
  JcampDxClass& operator [] (unsigned int i);

/**
  * Returns the const i'th parameter in the block
  */
  const JcampDxClass& operator [] (unsigned int i) const;

/**
  * Returns the first parameter in the block with the given 'id'
  */
  JcampDxClass& get_parameter_by_id(int id);


/**
  * Makes this become a copy of 'src', including its parameters by creating temporary objects.
  * These temporary objects will be deleted automatically upon destruction.
  */
  JcampDxBlock& create_copy(const JcampDxBlock& src);

/**
  * Appends a deep copy of src (not a reference).
  * This copy will be deleted automatically upon destruction.
  */
  JcampDxBlock& append_copy(const JcampDxClass& src);

/**
  * Copy parameter values from 'src' to the equivalent parameters in this block
  */
  JcampDxBlock& copy_ldr_vals(const JcampDxBlock& src);


/**
  * Compares 'rhs' with 'this' with respect to whether all parameters with
  * the same label mutually exist in the other block
  * and whether these parameters have the same type, label and value.
  */
  bool operator == (const JcampDxBlock& rhs) const {return !( ((*this)<rhs) || ((rhs<(*this))) );}



/**
  * Comparison operator which
  * - (1st) compares number of parameters
  * - (2nd) the type of parameters existing in both blocks
  * - (3rd) the value of parameters existing in both blocks
  */
  bool operator < (const JcampDxBlock& rhs) const {return compare(rhs);}


/**
  * Equivalent to operator <. In addition, a list of parameters
  * can be given in 'exclude' which are excluded from the comparison.
  * When comparing float or double numbers, parametrs are considered equal
  * if they differe by no more than the given 'accuracy'.
  */
  bool compare(const JcampDxBlock& rhs, const STD_list<STD_string>* exclude=0, double accuracy=0.0) const;


#ifndef NO_CMDLINE
  JcampDxBlock& parse_cmdline_options(int argc, char *argv[], bool modify=true);
  STD_map<STD_string,STD_string> get_cmdline_options() const;
  STD_string get_cmdline_usage(const STD_string& lineprefix="") const;
#endif


  // functions for StaticHandler
  static void init_static();
  static void destroy_static();

  // overwriting virtual functions from JcampDxClass
  STD_string print() const;
  JcampDxClass& set_compatmode(compatMode compat_mode);
  JcampDxClass& set_parmode(parameterMode parameter_mode);
  JcampDxClass& set_filemode(fileMode file_mode);
  STD_string get_parx_code(parxCodeType type, const ParxEquiv& equiv=ParxEquiv()) const;
  STD_ostream& print2stream(STD_ostream& os) const;
  bool parsevalstring (const STD_string&) {return true;}
  bool parse(STD_string& parstring);
  STD_string printvalstring() const {return "";}
  JcampDxClass* create_copy() const {JcampDxBlock* result=new JcampDxBlock; result->create_copy(*this); return result;}
  const char* get_typeInfo() const {return "JcampDxBlock";}
  JcampDxBlock* cast(JcampDxBlock*) {return this;}
  int load(const STD_string &filename);
  int write(const STD_string &filename) const;
  STD_string get_jdx_prefix() const {return "";}
  STD_string get_jdx_postfix() const {return "";}

 protected:

/**
  *
  *  Use this functions to append members of derived classes
  */
  JcampDxBlock&  append_member(JcampDxClass& ldr,const STD_string ldrlabel="");




 private:
  friend class JDXwidget;

  static STD_string extract_parlabel(const STD_string& parstring);

  STD_string print_header() const;
  STD_string print_tail() const;


  JDXList::constiter ldr_exists(const STD_string& label) const;
  int parse_ldr_list(STD_string& parstring);

  STD_list<JcampDxClass*>* garbage;
  bool embed;

};



/** @}
  */



#endif


