## Copyright (C) 1999 Daniel Heiserer 2001 Laurent Mazet
##
## This program is free software; it is distributed in the hope that it
## will be useful, but WITHOUT ANY WARRANTY; without even the implied
## warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
## the GNU General Public License for more details.
##
## You should have received a copy of the GNU General Public License
## along with this file; see the file COPYING.  If not, write to the
## Free Software Foundation, 59 Temple Place - Suite 330, Boston, MA
## 02111-1307, USA.

## -*- texinfo -*-
## @deftypefn {Function File} {} print (@var{filename}, @var{options})
##
## Print a graph, or save it to a file
##
## @var{filename} defines the file name of the output file. If no
## filename is specified, output is sent to the printer.
##
## @var{options}:
## @table @code
## @item -P@var{printer}
##   Set the @var{printer} name to which the graph is sent if no
##   @var{filename} is specified.
## @item -color
## @itemx -mono
##   Monochrome or colour lines.
## @item -solid
## @itemx -dashed
##   Solid or dashed lines.
## @item -portrait
## @itemx -landscape
##   Plot orientation, as returned by "orient".
## @item -d@var{device}
##   Output device, where @var{device} is one of:
##   @table @code
##   @item ps
##   @itemx ps2
##   @itemx psc
##   @itemx psc2
##     Postscript (level 1 and 2, mono and color)
##   @item eps
##   @itemx eps2
##   @itemx epsc
##   @itemx epsc2
##     Encapsulated postscript (level 1 and 2, mono and color)
##   @item ill
##   @itemx aifm
##     Adobe Illustrator
##   @item cdr
##   @itemx corel
##     CorelDraw
##   @item hpgl
##     HP plotter language
##   @item fig
##     XFig
##   @item dxf
##     AutoCAD
##   @item mf
##     Metafont
##   @item png
##     Portable network graphics
##   @item pbm
##     PBMplus
##   @end table
##
##   Other devices are supported by "convert" from ImageMagick.  Type
##   system("convert") to see what formats are available.
##
##   If the device is omitted, it is inferred from the file extension,
##   or if there is no filename it is sent to the printer as postscript.
##
## @item -F@var{fontname}
## @itemx -F@var{fontname}:@var{size}
## @itemx -F:@var{size}
##   @var{fontname} set the postscript font (for use with postscript,
##   aifm, corel and fig). By default, 'Helvetica' is set for PS/Aifm,
##   and 'SwitzerlandLight' for Corel. It can also be 'Times-Roman'.
##   @var{size} is given in points. @var{fontname} is ignored for the
##   fig device.
## @end table
##
## The filename and options can be given in any order.
##
## If you are using Octave 2.1.x or above, command("print") will change 
## print from a function to a command, so instead of typing
##    print("-FTimes-Roman:14", "-dashed", "-depsc", "out.ps")
## you can type
##    print -FTimes-Roman:14 -dashed -depsc out.ps
##
## See also: orient, command
## @end deftypefn

## Author: Daniel Heiserer <Daniel.heiserer@physik.tu-muenchen.de>
## 2001-03-23  Laurent Mazet <mazet@crm.mot.com>
##     * simplified interface: guess the device from the extension
##     * font support
## 2001-03-25  Paul Kienzle <pkienzle@kienzle.powernet.co.uk>
##     * add unwind_protect
##     * use tmpnam to generate temporary name
##     * move "set term" before "set output" as required by gnuplot
##     * more options, and flexible options
## 2001-03-29  Laurent Mazet <mazet@crm.mot.com>
##     * add solid and dashed options
##     * change PBMplus device
##     * add Corel device
##     * take care of the default terminal settings to restore them.
##     * add color, mono, dashed and solid support for printing and convert.
##     * add orientation for printing.
##     * create a .ps for printing (avoid some filtering problems).
##     * default printing is mono, default convert is color.
##     * add font size support.
## 2001-03-30  Laurent Mazet <mazet@crm.mot.com>
##     * correct correl into corel
##     * delete a irrelevant test
##     * check for convert before choosing the ouput device
## 2001-03-31  Paul Kienzle <pkienzle@kienzle.powernet.co.uk>
##     * use -Ffontname:size instead of -F"fontname size"
##     * add font size support to fig option
##     * update documentation
## 2003-10-01  Laurent Mazet <mazet@crm.mot.com>
##     * clean documentation

function print(varargin)

  ## take care of the settings we had before
  origterm = gget("terminal");
  origout = gget("output");
  _automatic_replot = automatic_replot;

  ## take care of the default terminal settings to restore them.
  terminal_default = "";
  
  orientation = orient;
  use_color = 0; # 0=default, -1=mono, +1=color
  force_solid = 0; # 0=default, -1=dashed, +1=solid
  fontsize = font = name = devopt = printer = "";
  
  va_arg_cnt = 1;

  for i=1:nargin
    arg = nth (varargin, va_arg_cnt++);
    if isstr(arg)
      if strcmp(arg, "-color")
	use_color = 1;
      elseif strcmp(arg, "-mono")
	use_color = -1;
      elseif strcmp(arg, "-solid")
        force_solid = 1;
      elseif strcmp(arg, "-dashed")
        force_solid = -1;
      elseif strcmp(arg, "-portrait")
	orientation = "portrait";
      elseif strcmp(arg, "-landscape")
	orientation = "landscape";
      elseif length(arg) > 2 && arg(1:2) == "-d"
	devopt = arg(3:length(arg));
      elseif length(arg) > 2 && arg(1:2) == "-P"
	printer = arg;
      elseif length(arg) > 2 && arg(1:2) == "-F"
	idx = rindex(arg, ":");
	if (idx)
	  font = arg(3:idx-1);
	  fontsize = arg(idx+1:length(arg));
	else
	  font = arg(3:length(arg));
	endif
      elseif length(arg) >= 1 && arg(1) == "-"
	error([ "print: unknown option ", arg ]);
      elseif length(arg) > 0
	name = arg;
      endif
    else
      error("print: expects string options");
    endif
  endfor

  doprint = isempty(name);
  if doprint
    if isempty(devopt)
      printname = [ tmpnam, ".ps" ]; 
    else
      printname = [ tmpnam, ".", devopt ];
    endif
    name = printname;
  endif

  if isempty(devopt)
    dot = rindex(name, ".");
    if (dot == 0) 
      error ("print: no format specified");
    else
      dev = tolower(name(dot+1:length(name)));
    endif
  else
    dev = devopt;
  endif

  if strcmp(dev, "ill")
    dev = "aifm";
  elseif strcmp(dev, "cdr")
    dev = "corel";
  endif

  ## check if we have to use convert
  dev_list = [" aifm corel fig png pbm dxf mf hpgl", ...
	      " ps ps2 psc psc2 eps eps2 epsc epsc2 " ];
  convertname = "";
  if isempty(findstr(dev_list , [ " ", dev, " " ]))
    if !isempty(devopt)
      convertname = [ devopt ":" name ];
    else
      convertname = name;
    endif
    dev = "epsc";
    name = [ tmpnam, ".eps" ];
  endif
  
  unwind_protect
    automatic_replot = 0;

    if strcmp(dev, "ps") || strcmp(dev, "ps2") ...
	  || strcmp(dev, "psc")  || strcmp(dev, "psc2") ...
	  || strcmp(dev, "epsc") || strcmp(dev, "epsc2") ... 
	  || strcmp(dev, "eps")  || strcmp(dev, "eps2")
      ## Various postscript options
      gset term postscript;
      terminal_default = gget ("terminal");
      
      if dev(1) == "e"
	options = "eps ";
      else
	options = [ orientation, " " ];
      endif
      options = [ options, "enhanced " ];
      
      if any( dev == "c" ) || use_color > 0
        if force_solid < 0
	  options = [ options, "color dashed " ];
	else
          options = [ options, "color solid " ];
        endif
      else
        if force_solid > 0
	  options = [ options, "mono solid " ];
	else
	  options = [ options, "mono dashed " ];
        endif
      endif

      if !isempty(font)
	options = [ options, "\"", font, "\" " ];
      endif
      if !isempty(fontsize)
	options = [ options, " ", fontsize ];
      endif

      eval (sprintf ("gset term postscript %s;", options));


    elseif strcmp(dev, "aifm") || strcmp(dev, "corel")
      ## Adobe Illustrator, CorelDraw
      eval(sprintf ("gset term %s;", dev));
      terminal_default = gget ("terminal");
      if (use_color >= 0)
	options = " color";
      else
	options = " mono";
      endif
      if !isempty(font)
	options = [ options, " \"" , font, "\"" ];
      endif
      if !isempty(fontsize)
	options = [ options, " ", fontsize ];
      endif

      eval (sprintf ("gset term %s %s;", dev, options));

    elseif strcmp(dev, "fig")
      ## XFig
      gset term fig;
      terminal_default = gget ("terminal");
      options = orientation;
      if (use_color >= 0)
	options = " color";
      else
	options = " mono";
      endif
      if !isempty(fontsize)
	options = [ options, " fontsize ", fontsize ];
      endif
      eval (sprintf ("gset term fig %s;", options));

    elseif strcmp(dev, "png") || strcmp(dev, "pbm")
      ## Portable network graphics, PBMplus
      eval(sprintf ("gset term %s;", dev));
      terminal_default = gget ("terminal");
      if (use_color >= 0)
      	eval (sprintf ("gset term %s color medium;", dev));
      else
	eval (sprintf ("gset term %s mono medium;", dev));
      endif

    elseif strcmp(dev,"dxf") || strcmp(dev,"mf") || strcmp(dev, "hpgl")
      ## AutoCad DXF, METAFONT, HPGL
      eval (sprintf ("gset terminal %s;", dev));
            
    endif;
    
    eval (sprintf ("gset output \"%s\";", name));
    replot;
    
  unwind_protect_cleanup

    ## Restore init state
    if ! isempty (terminal_default)
      eval (sprintf ("gset terminal %s;", terminal_default));
    endif
    eval (sprintf ("gset terminal %s;", origterm));
    if isempty (origout)
      gset output;
    else
      eval (sprintf ("gset output \"%s\";", origout));
    end
    replot;
    
    automatic_replot = _automatic_replot ;

  end_unwind_protect

  if !isempty(convertname)
    command = [ "convert '", name, "' '", convertname, "'" ];
    [output, errcode] = system (command);
    unlink (name);
    if (errcode)
      error ("print: could not convert");
    endif
  endif
  if doprint
    system(sprintf ("lpr %s '%s'", printer, printname));
    unlink(printname);
  endif
  
endfunction
