/** OpenCP Module Player
 * copyright (c) '06-'09 Stian Skjelstad <stian@nixia.no>
 *
 * CoreAudio (Darwin/Mac OS/OSX) Player device
 */

#include "config.h"
#include <stdio.h>
#include <unistd.h>
#include <math.h>
#include <pthread.h>
#include <CoreAudio/CoreAudio.h>
#include <CoreServices/CoreServices.h>
#include <AudioUnit/AudioUnit.h>
#include <AudioToolbox/AudioToolbox.h>
#include "types.h"
#include "boot/plinkman.h"
#include "dev/imsdev.h"
#include "dev/player.h"
#include "stuff/imsrtns.h"

static AudioUnit theOutputUnit;
static pthread_mutex_t mutex = PTHREAD_MUTEX_INITIALIZER;
struct sounddevice plrCoreAudio;
static int needfini=0;

static unsigned int CoreAudioRate;
static void *playbuf=0;
static int buflen;
volatile static int kernpos, cachepos, bufpos; /* in bytes */
volatile static int cachelen, kernlen; /* to make life easier */

volatile static uint32_t playpos; /* how many samples have we done totally */

static const char *OSStatus_to_string(OSStatus status)
{
	switch (status)
 	{
   		case kAudioHardwareNoError:
			return "kAudioHardwareNoError";
		case kAudioHardwareNotRunningError:
			return "kAudioHardwareNotRunningError";
		case kAudioHardwareUnspecifiedError:
			return "kAudioHardwareUnspecifiedError";
		case kAudioHardwareUnknownPropertyError:
			return "kAudioHardwareUnknownPropertyError";
		case kAudioHardwareBadPropertySizeError:
			return "kAudioHardwareBadPropertySizeError";
		case kAudioHardwareIllegalOperationError:
			return "kAudioHardwareIllegalOperationError";
		case kAudioHardwareBadDeviceError:
			return "kAudioHardwareBadDeviceError";
		case kAudioHardwareBadStreamError:
			return "kAudioHardwareBadStreamError";
		case kAudioHardwareUnsupportedOperationError:
			return "kAudioHardwareUnsupportedOperationError";
		case kAudioDeviceUnsupportedFormatError:
			return "kAudioDeviceUnsupportedFormatError";
		case kAudioDevicePermissionsError:
			return "kAudioDevicePermissionsError";
		default:
			return "unknown";
	}
}

/*
static void print_format(const char* str,AudioStreamBasicDescription *f){
    uint32_t flags=(uint32_t) f->mFormatFlags;
    fprintf(stderr, "%s %7.1fHz %dbit [%c%c%c%c] %s %s %s%s%s%s\n",
            str, f->mSampleRate, (int)f->mBitsPerChannel,
            (int)(f->mFormatID & 0xff000000) >> 24,
            (int)(f->mFormatID & 0x00ff0000) >> 16,
            (int)(f->mFormatID & 0x0000ff00) >>  8,
            (int)(f->mFormatID & 0x000000ff) >>  0,
            (flags&kAudioFormatFlagIsFloat) ? "float" : "int",
            (flags&kAudioFormatFlagIsBigEndian) ? "BE" : "LE",
            (flags&kAudioFormatFlagIsSignedInteger) ? "S" : "U",
            (flags&kAudioFormatFlagIsPacked) ? " packed" : "",
            (flags&kAudioFormatFlagIsAlignedHigh) ? " aligned" : "",
            (flags&kAudioFormatFlagIsNonInterleaved) ? " non-interleaved" : " interleaved" );

    fprintf(stderr, "%5d mBytesPerPacket\n",
            (int)f->mBytesPerPacket);
    fprintf(stderr, "%5d mFramesPerPacket\n",
            (int)f->mFramesPerPacket);
    fprintf(stderr, "%5d mBytesPerFrame\n",
            (int)f->mBytesPerFrame);
    fprintf(stderr, "%5d mChannelsPerFrame\n",
            (int)f->mChannelsPerFrame);

}

static void ostype2string (OSType inType, char *outString)
{
   unsigned char   theChars[4];
   unsigned char *theString = (unsigned char *)outString;
   unsigned char *theCharPtr = theChars;
   int                       i;

   // extract four characters in big-endian order
   theChars[0] = 0xff & (inType >> 24);
   theChars[1] = 0xff & (inType >> 16);
   theChars[2] = 0xff & (inType >> 8);
   theChars[3] = 0xff & (inType);

   for (i = 0; i < 4; i++) {
      if((*theCharPtr >= ' ') && (*theCharPtr <= 126)) {
         *theString++ = *theCharPtr;
      } else {
         *theString++ = ' ';
      }

      theCharPtr++;
   }

   *theString = 0;
}*/

static OSStatus theRenderProc(void *inRefCon, AudioUnitRenderActionFlags *inActionFlags, const AudioTimeStamp *inTimeStamp, UInt32 inBusNumber, UInt32 inNumFrames, AudioBufferList *ioData)
{
	int i, i2;

	pthread_mutex_lock(&mutex);

	i = cachelen>>2; /*stereo + 16it */
	if (i > ioData->mBuffers[0].mDataByteSize)
		i = ioData->mBuffers[0].mDataByteSize;

	kernlen = ioData->mBuffers[0].mDataByteSize = i;
	playpos += i;

	if ((i+kernpos)>buflen)
	{
		i2 = ( i + kernpos ) % buflen;
		i = i - i2;
	} else {
		i2 = 0;
	}

	memcpy(ioData->mBuffers[0].mData, playbuf+kernpos, i);
	if (i2)
		memcpy(ioData->mBuffers[0].mData+i, playbuf, i2);

	kernpos = (kernpos+i+i2)%buflen;

	pthread_mutex_unlock(&mutex);

	return noErr;
}

/* stolen from devposs */
static int CoreAudioGetBufPos(void)
{
	int retval;

	pthread_mutex_lock(&mutex);
	if (kernpos==bufpos)
	{
		if (cachelen|kernlen)
		{
			retval=kernpos;
			pthread_mutex_unlock(&mutex);
			return retval;
		}
	}
	retval=(kernpos+buflen-4 /* 1 sample = 4 bytes*/)%buflen;
	pthread_mutex_unlock(&mutex);
	return retval;
}

static int CoreAudioGetPlayPos(void)
{
	int retval;

	pthread_mutex_lock(&mutex);
	retval=kernpos;
	pthread_mutex_unlock(&mutex);
        return retval;
}

static void CoreAudioIdle(void)
{
}

static void CoreAudioAdvanceTo(unsigned int pos)
{
	pthread_mutex_lock(&mutex);

	cachelen+=(pos-bufpos+buflen)%buflen;
	bufpos=pos;

	pthread_mutex_unlock(&mutex);
}

static uint32_t CoreAudioGetTimer(void)
{
	long retval;

	pthread_mutex_lock(&mutex);

	retval=playpos-kernlen;

	pthread_mutex_unlock(&mutex);

	return imuldiv(retval, 65536>>(2/*stereo+bit16*/), plrRate);
}

static void CoreAudioStop(void)
{
	OSErr status;

	/* TODO, forceflush */

	if (playbuf)
	{
		free(playbuf);
		playbuf=0;
	}

	plrGetBufPos=0;
	plrGetPlayPos=0;
	plrIdle=0;
	plrAdvanceTo=0;
	plrGetTimer=0;

	status=AudioOutputUnitStop(theOutputUnit);
	if (status)
		fprintf(stderr, "[CoreAudio] AudioOutputUnitStop returned %d (%s)\nn", (int)status, OSStatus_to_string(status));
}

static int CoreAudioPlay(void **buf, unsigned int *len)
{
	OSErr status;

	if ((*len)<(plrRate&~3))
		*len=plrRate&~3;
	if ((*len)>(plrRate*4))
		*len=plrRate*4;
	playbuf=*buf=malloc(*len);

	memset(*buf, 0x80008000, (*len)>>2);
	buflen = *len;

	kernpos=0;
	cachepos=0;
	bufpos=0;
	cachelen=0;
	kernlen=0;

	playpos=0;

	plrGetBufPos=CoreAudioGetBufPos;
	plrGetPlayPos=CoreAudioGetPlayPos;
	plrIdle=CoreAudioIdle;
	plrAdvanceTo=CoreAudioAdvanceTo;
	plrGetTimer=CoreAudioGetTimer;

	status=AudioOutputUnitStart(theOutputUnit);
	if (status)
	{
  		fprintf(stderr, "[CoreAudio] AudioOutputUnitStart returned %d (%s)\n", (int)status, OSStatus_to_string(status));
		free(*buf);
		*buf = playbuf = 0;
		plrGetBufPos = 0;
		plrGetPlayPos = 0;
		plrIdle = 0;
		plrAdvanceTo = 0;
		plrGetTimer = 0;
		return 0;
	}
	return 1;
}

static void CoreAudioSetOptions(unsigned int rate, int opt)
{
	plrRate=CoreAudioRate; /* fixed */
	plrOpt=PLR_STEREO|PLR_16BIT|PLR_SIGNEDOUT; /* fixed fixed fixed */
}

static int CoreAudioInit(const struct deviceinfo *c)
{
	plrSetOptions=CoreAudioSetOptions;
	plrPlay=CoreAudioPlay;
	plrStop=CoreAudioStop;
	return 1;
}

static void CoreAudioClose(void)
{
	plrSetOptions=0;
	plrPlay=0;
	plrStop=0;
}

static int CoreAudioDetect(struct deviceinfo *card)
{
#if 0
	static AudioDeviceID gOutputDeviceID; /* TODO */
#endif

	AudioStreamBasicDescription inDesc;
	const int channels=2;
	OSStatus status;
/*	UInt32 propertySize; */

	ComponentDescription desc;
	Component comp;

	UInt32 /*maxFrames,*/ size;

	AURenderCallbackStruct renderCallback;

#if 0
	propertySize = sizeof(gOutputDeviceID);
	status = AudioHardwareGetProperty(kAudioHardwarePropertyDefaultOutputDevice,
	                                  &propertySize,
	                                  &gOutputDeviceID);
	if (status) {
		fprintf(stderr, "[CoreAudio] Could not get default output Device (status = %d (%s))\n", (int)status, OSStatus_to_string(status));
		goto errorout;
	}

	if (gOutputDeviceID == kAudioDeviceUnknown)
	{
		fprintf(stderr, "[CoreAudio] Unable to find a default output device\n");
		goto errorout;
	}
#endif
	desc.componentType = kAudioUnitType_Output;
	desc.componentSubType = kAudioUnitSubType_DefaultOutput;
	desc.componentManufacturer = kAudioUnitManufacturer_Apple;
	desc.componentFlags = 0;
	desc.componentFlagsMask = 0;

	comp = FindNextComponent (comp, &desc);
	if ( !comp ) {
		fprintf(stderr, "[CoreAudio] Unable to find the Output Unit component\n");
		goto errorout;
	}

	status = OpenAComponent (comp, &theOutputUnit);
	if (status) {
		fprintf(stderr, "[CoreAudio] Unable to open Output Unit component (status = %d (%s))\n", (int)status, OSStatus_to_string(status));
		goto errorout;
	}

	size = sizeof(inDesc);
	status = AudioUnitGetProperty (theOutputUnit, kAudioUnitProperty_StreamFormat, kAudioUnitScope_Input, 0, &inDesc, &size);
	if (status) {
		fprintf(stderr, "[CoreAudio] Unable to get the input (default) format (status = %d (%s))\n", (int)status, OSStatus_to_string(status));
		goto errorout_component;
	}

	/*print_format("default: ", &inDesc);*/

	CoreAudioRate = inDesc.mSampleRate;
/*	inDesc.mSampleRate = 44100; Use default here */
	inDesc.mFormatID=kAudioFormatLinearPCM;
	inDesc.mChannelsPerFrame=channels;
	inDesc.mBitsPerChannel=16;
	inDesc.mFormatFlags = kAudioFormatFlagIsSignedInteger|kAudioFormatFlagIsPacked; /* float is possible */
#ifdef WORDS_BIGENDIAN
	inDesc.mFormatFlags |= kAudioFormatFlagIsBigEndian;
#endif
	inDesc.mFramesPerPacket=1;
	inDesc.mBytesPerPacket = inDesc.mBytesPerFrame = inDesc.mFramesPerPacket*channels*(inDesc.mBitsPerChannel/8);
	
	/*print_format("target: ", &inDesc);*/

	status = AudioUnitSetProperty (theOutputUnit, kAudioUnitProperty_StreamFormat, kAudioUnitScope_Input, 0, &inDesc, sizeof(inDesc));
	if (status) {
		fprintf(stderr, "[CoreAudio] Unable to set the input format (status = %d (%s))\n", (int)status, OSStatus_to_string(status));
		goto errorout_component;
	}

#if 0
	size = sizeof(maxFrames);
	status = AudioUnitGetProperty (theOutputUnit, kAudioDevicePropertyBufferSize, kAudioUnitScope_Input, 0, &maxFrames, &size);
	if (status) {
		fprintf(stderr, "[CoreAudio] AudioUnitGetProperty returned %d (%s) when getting kAudioDevicePropertyBufferSize\n", (int)status, OSStatus_to_string(status));
		goto errorout_uninit;
	}

	fprintf(stderr, "[CoreAudio] maxFrames=%d\n", (int)maxFrames);
#endif

	renderCallback.inputProc = theRenderProc;
	renderCallback.inputProcRefCon = 0;
	status = AudioUnitSetProperty(theOutputUnit, kAudioUnitProperty_SetRenderCallback, kAudioUnitScope_Input, 0, &renderCallback, sizeof(AURenderCallbackStruct));
        if (status) {
		fprintf(stderr, "[CoreAudio] Unable to set the render callback (status = %d (%s))\n", (int)status, OSStatus_to_string(status));
		goto errorout_component;
	}

	status = AudioUnitInitialize (theOutputUnit);
	if (status) {
		fprintf(stderr, "[CoreAudio] Unable to initialize Output Unit component (status = %d (%s))\n", (int)status, OSStatus_to_string(status));
		goto errorout_component;
	}
	/* ao is now created, the above is needed only ONCE */

	card->dev=&plrCoreAudio;
	card->port=0;
	card->port2=0;
	card->subtype=-1;
	card->mem=0;
	card->chan=2;

	needfini=1;

	return 1;

#if 0
errorout_uninit:
	AudioUnitUninitialize (theOutputUnit);
#endif
errorout_component:
	CloseComponent (theOutputUnit);
errorout:
	return 0;
}

static void __attribute__((destructor))fini(void)
{
	if (needfini)
	{
		AudioUnitUninitialize (theOutputUnit);
		CloseComponent (theOutputUnit);
	}
}

struct sounddevice plrCoreAudio={SS_PLAYER, "CoreAudio player", CoreAudioDetect,  CoreAudioInit,  CoreAudioClose};
char *dllinfo="driver plrCoreAudio";
struct linkinfostruct dllextinfo = {"devpalsa", "OpenCP Player Device: CoreAudio (c) 2006 Stian Skjelstad", DLLVERSION, 0};
