(* $Id: netcgi_jserv_app.mli,v 1.5 2003/02/04 01:39:14 stolpmann Exp $
 * ----------------------------------------------------------------------
 *
 *)


(* Support for the JSERV protocol (see http://java.apache.org/jserv/).
 * See also the explanations under examples/jserv.
 *)


(* This module contains a "ready to use" application server framework
 * for single-threaded and multi-process servers.
 *)

type request_handler =
    { req_activate       : Netcgi_types.cgi_activation -> unit;
      req_processing     : string -> Netmime.mime_header -> 
	                                             Netcgi.argument_processing;
      req_operating_type : Netcgi.operating_type;
    }

  (*
   * req_activate is the request handler. It can be programmed in the
   * style of a CGI.
   *
   * req_processing, req_operating_type: See Netcgi.
   *)

type server_type =
    [ `Sequential   of        (string * request_handler) list
    | `Forking      of int * ((string * request_handler) list)
    | `Process_pool of int * ((string * request_handler) list)
    ]

  (* `Sequential: The server processes the requests sequentially.
   * `Forking: The server forks for every request, and processes it in the
   *    child process. The integer is the maximum number of children 
   *    processes; if it is exceeded, an error message is displayed immediately
   *    without forking.
   * `Process_pool: The server forks a fixed number of times (the integer).
   *    The children will process the requests concurrently. If more requests
   *    arrive than children are available, the requests must wait until
   *    a child becomes free again.
   *
   * The   (string * request_handler) list   of all server types is the list of
   * servlets. The string is the name of the servlet (which is part of 
   * the URL), and the request_handler is the corresponding handler.
   *)

type protocol_type =
    [ `Ajp_1_2
    ]
  (* Selects the protocol. *)

type jserv_config =
    { js_backlog : int;
      js_reuseaddr : bool;
      js_cgiconfig : Netcgi_env.cgi_config;
      js_init_process : unit -> unit;
      js_fini_process : unit -> unit;    
      js_idle_worker : unit -> unit;
      js_idle_master : unit -> unit;
    }

  (* The configuration XXX *)
  (* js_init_process is called when the process is initialized, and
   * js_fini_process is called just before the process exits. This
   * means:
   * For servers of type `Sequential: js_init_process is called once before
   *   the server begins accepting connections, and js_fini_process is called
   *   once after the server has got the shutdown notification.
   * For servers of type `Forking and `Process_pool: js_init_process is called 
   *   just after the process has forked, and js_fini_process is called
   *   just before the child process exits.
   *
   * js_idle_worker is called every second while the working process is
   *   idle.
   * js_idle_master is called every second while the master process waits
   *   for new connections. - This means:
   * For servers of type `Sequential: both js_idle_worker and js_idle_master
   *   are called every second while the server waits for new connections.
   * For servers of type `Forking: js_idle_master is called every second
   *   while the master process waits for new connections. js_idle_worker
   *   is not called at all, as the working children processes are never
   *   idle.
   * For servers of type `Process_pool: js_idle_master is called every second
   *   while the master process waits for new connections. js_idle_worker
   *   is called by every child process every second while the process waits
   *   for a new request.
   *
   * js_idle_master is useful for tasks that must be carried out periodically
   * for the whole system (e.g. log file rotation every day). 
   * js_idle_worker is useful for background tasks that modify the state of the
   * worker processes (e.g. close database handles after a period of
   * inactivity).
   * 
   * Exceptions: if raised in js_idle_worker, exceptions are logged.
   * The exceptions raised in js_idle_master are not caught, and fall through.
   *)

val std_config : jserv_config

  (* The standard configuration:
   *
   * - small backlog
   * - ports are reused
   * - default CGI configuration
   * - callback functions are all (fun () -> ()).
   *)

val logger : (string -> string -> unit) ref

  (* This variable contains the logger function. The function is called to
   * log error conditions. The first passed string is the servlet name,
   * or "?" if not available. The second passed string is the message.
   * By default, the messages are written to stderr.
   *)


val run :
      ?config:jserv_config ->
      server_type ->
      protocol_type ->
      (* props: *) (string * string) list ->
      Netcgi_jserv.auth option -> 
      (* addr: *) Unix.inet_addr -> 
      (* port: *) int ->
	unit
  (* Starts the server. The last four arguments are compatible with the 
   * function accepted by Netcgi_jserv.jvm_emu_main, so [run] can be
   * used as follows:
   *
   * Netcgi_jserv.jvm_emu_main (run ~config srvtype prttype)
   *
   * The server runs until it gets a shutdown notification from Apache.
   * 
   * Restart notifications are currently ignored.
   *
   * Another remark about `Process_pool. The signal handler for SIGALRM
   * is temporarily redefined while a process waits for a lock. The old
   * handler is suspended until the lock can be acquired.
   *)


(* ======================================================================
 * History:
 * 
 * $Log: netcgi_jserv_app.mli,v $
 * Revision 1.5  2003/02/04 01:39:14  stolpmann
 * 	New feature: js_idle_worker, js_idle_master options.
 * 	Fix: locking code for `Process_pool takes into account that
 * file positions are shared by file descriptors after fork
 *
 * Revision 1.4  2003/01/27 21:52:55  stolpmann
 * 	The logger function can be changed (exported).
 *
 * Revision 1.3  2002/03/24 22:31:07  stolpmann
 * 	New: `Process_pool
 *
 * Revision 1.2  2002/02/04 15:21:31  stolpmann
 * 	Updated comments.
 *
 * Revision 1.1  2002/02/03 21:27:14  stolpmann
 * 	Initial revision.
 *
 * 
 *)
