(* $Id: netcgi_fcgi_10.ml,v 1.1 2003/10/07 17:39:32 stolpmann Exp $
 * ----------------------------------------------------------------------
 *
 *)

(* This code is copyright 2003 Eric Stokes, and may be used under
either, the GNU GPL, or the same license as ocamlnet *)

open Unix

(* protocal defines *)
let fcgi_begin_request     = 1;;
let fcgi_abort_request     = 2;;
let fcgi_end_request       = 3;;
let fcgi_params            = 4;;
let fcgi_stdin             = 5;;
let fcgi_stdout            = 6;;
let fcgi_stderr            = 7;;
let fcgi_data              = 8;;
let fcgi_get_values        = 9;;
let fcgi_get_values_result = 10;;
let fcgi_unknown_type      = 11;;
let fcgi_maxtype           = fcgi_unknown_type;;
let fcgi_null_request_id   = 0;;
let fcgi_keep_conn         = 1;;
let fcgi_responder         = 1;;
let fcgi_authorizer        = 2;;
let fcgi_filter            = 3;;
let fcgi_request_complete  = 0;;
let fcgi_cant_mpx_conn     = 1;;
let fcgi_overloaded        = 2;;
let fcgi_unknown_role      = 3;;


(* protocal header *)
type fcgiHeader = {version: int; rtype: int; requestid: int; contentlen: int; padlen: int}

(* begin request *)
type fcgiBeginRequestBody = {role: int; flags: int}

(* end request *)
type fcgiEndRequestBody = {astatus: int; pstatus: int}

(* fcgi params will return an asociation list *)
(* fcgi stdin will return a string *)

(* a full request record for the responder roll *)
type fcgiRequest = {id: int; 
		    app_type: int;
		    params: (string * string) list; 
		    stdin: string;
		    data: string;
		    con: Unix.file_descr}


(* debug print header *)
let print_header hd =
  print_string "ver:";print_int hd.version;print_endline "";
  print_string "typ:";print_int hd.rtype;print_endline "";
  print_string "id:";print_int hd.requestid;print_endline "";
  print_string "clen:";print_int hd.contentlen;print_endline "";
  print_string "plen:";print_int hd.padlen;print_endline ""

let print_packet (hd, data) =
  print_header hd; print_endline data

let print_request rq =
  let print_params prms =
    List.iter (fun x -> print_string (fst x);print_string ": ";print_endline (snd x)) prms 
  in
  print_string "requestid: ";print_int rq.id;print_endline "";
  print_string "role: ";print_int rq.app_type;print_endline "";
  print_endline "params"; print_params rq.params;
  print_endline "stdin"; print_endline rq.stdin;
  print_endline "data"; print_endline rq.data


(*********************************************
  general encodeing, and decodeing functions 
**********************************************)
let byte_from_int1 i = (* one byte integer encoding *)
  (if i > 255 then
     failwith "out of range");
  let buf = String.make 1 'c' in
    String.set buf 0 (Char.chr i); buf

let int_from_byte1 s off = (* one byte integer decoding *)
  Char.code (String.get s off)

let byte_from_int2 i = (* two byte integer encodeing *)
  (if i > 65535 then
     failwith "out of range");
  let buf = String.make 2 'c' in
    String.set buf 0 (Char.chr (i lsr 8));
    String.set buf 1 (Char.chr (i land 255)); buf
      
let int_from_byte2 s off = (* two byte integer decodeing *)
  ((int_from_byte1 s off) lsl 8) + int_from_byte1 s (off + 1)

let byte_from_int4 i = (* four byte integer encodeing *)
  let buf = String.make 4 'c' in
    String.set buf 0 (Char.chr (i lsr 24));
    String.set buf 1 (Char.chr ((i lsr 16) land 255));
    String.set buf 2 (Char.chr ((i lsr 8) land 255));
    String.set buf 3 (Char.chr (i land 255)); buf

let int_from_byte4 data off = (* four byte integer decodeing *)
      let byte0 = (Char.code (String.get data off)) in
      let byte1 = (Char.code (String.get data (off + 1))) in
      let byte2 = (Char.code (String.get data (off + 2))) in
      let byte3 = (Char.code (String.get data (off + 3))) in
	((+) ((+) ((+) ((lsl) byte0 24) ((lsl) byte1 16)) ((lsl) byte2 8)) byte3)

(*************************************************** 
 fastcgi structure encodeing and decodeing functions 
****************************************************)
let encode_fcgi_header hdr = (* encodeing of fcgi header *)
  (byte_from_int1 hdr.version)    ^
  (byte_from_int1 hdr.rtype)      ^
  (byte_from_int2 hdr.requestid)  ^ 
  (byte_from_int2 hdr.contentlen) ^
  (byte_from_int1 hdr.padlen)     ^
  (String.make 1 'c')

let decode_fcgi_header buf off =
  let version   = int_from_byte1 buf off in
  let rectype   = int_from_byte1 buf (off + 1) in
  let requestid = int_from_byte2 buf (off + 2) in
  let datalen   = int_from_byte2 buf (off + 4) in
  let padlen    = int_from_byte1 buf (off + 6) in
    {version=version;
     rtype=rectype;
     requestid=requestid;
     contentlen=datalen;
     padlen=padlen}

let decode_fcgi_begin_request data off =
  {role=int_from_byte2 data off; flags=int_from_byte1 data (off + 2)}

let decode_fcgi_param data off =
  let get_data data off nmlen vlen = 
    ((String.sub data off nmlen), (String.sub data (off + nmlen) vlen))
  in
    if ((lsr) (Char.code (String.get data off)) 7) = 1 then
      (let nmlen = int_from_byte4 data off in
	 if ((lsr) (Char.code (String.get data (off + 4))) 7) = 1 then
	   get_data data (off + 8) nmlen (int_from_byte4 data (off + 4))
	 else
	   get_data data (off + 5) nmlen (int_from_byte1 data (off + 4)))
    else
      (let nmlen = int_from_byte1 data off in
	 if ((lsr) (Char.code (String.get data (off + 1))) 7) = 1 then
	   get_data data (off + 5) nmlen (int_from_byte4 data (off + 1))
	 else
	   get_data data (off + 2) nmlen (int_from_byte1 data (off + 1)))

let encode_fcgi_data data id d_type = (* encode stdout, or stderr data for requestid id *)
  (encode_fcgi_header {version=1;
		       rtype=d_type;
		       requestid=id;
		       contentlen=(String.length data);
		       padlen=0}) ^ data

let encode_fcgi_end_request req id =
  match req with
      {astatus=astat; pstatus=pstat} ->
	(encode_fcgi_header {version=1;
			     rtype=fcgi_end_request;
			     requestid=id;
			     contentlen=8;
			     padlen=0}) ^
	(byte_from_int4 astat) ^ (byte_from_int1 pstat) ^ (String.make 3 'c')
  
(***************************************************
  functions which read or write fcgi structures
***************************************************)
let fcgi_read_header con = (* read the packet header *)
  let buf = String.make 8 'c' in
  let read = read con buf 0 8 in
    decode_fcgi_header buf 0

let fcgi_read_packet con = (* read one full packet *)
  let header = fcgi_read_header con in
  let buf = String.make header.contentlen 'c' in
  let read = read con buf 0 header.contentlen in
    (header, buf)

let fcgi_read_begin_request con =
  let (header, data) = fcgi_read_packet con in
    match header with
	{rtype=fcgi_begin_request;contentlen=0} -> failwith "Protocal Error"
      | {rtype=fcgi_begin_request;requestid=id;contentlen=8} ->
	  (id, (decode_fcgi_begin_request data 0))
      | {rtype=_} -> failwith "Invalid FCGI Structure"

let fcgi_read_params con =
  let rec fcgi_read_params con param =
    let (header, data) = fcgi_read_packet con in
      match header with
	  {rtype=v;contentlen=0} when v = fcgi_params -> param
	| {rtype=v} when v = fcgi_params -> 
	    fcgi_read_params con ((decode_fcgi_param data 0) :: param)
	| {rtype=_} -> failwith "Invalid FCGI Structure"
  in
    fcgi_read_params con []

let fcgi_read_stream type_t con =
  let rec read_stream type_t con data =
    let (header, payload) = fcgi_read_packet con in
      match header with
	  {rtype=t;contentlen=0} when t = type_t -> data
	| {rtype=t} when t = type_t -> read_stream type_t con (data ^ payload)
	| {rtype=_} -> failwith "Invalid FCGI Structure"
  in
    read_stream type_t con ""

let fcgi_read_stdin con =
  fcgi_read_stream fcgi_stdin con

let fcgi_read_data con =
  fcgi_read_stream fcgi_data con

(* read a full responder request *)
let fcgi_read_responder_request con id =
  let params  = fcgi_read_params con in
  let stdin = fcgi_read_stdin con in 
     {id=id;
      app_type=fcgi_responder;
      params=params;
      stdin=stdin;
      data="";
      con=con}

(* read full authorizer request (note, this is the same as the responder)*)
let fcgi_read_authorizer_request con id =
  let params  = fcgi_read_params con in
  let stdin = fcgi_read_stdin con in 
     {id=id;
      app_type=fcgi_authorizer;
      params=params;
      stdin=stdin;
      data="";
      con=con}

let fcgi_read_filter_request con id =
  let params = fcgi_read_params con in
  let stdin = fcgi_read_stdin con in
  let data = fcgi_read_data con in
    {id=id;
     app_type=fcgi_filter;
     params=params;
     stdin=stdin;
     data=data;
     con=con}

(* read a request *)
let fcgi_read_request con = 
  let (id, beginreq) = fcgi_read_begin_request con in
    match beginreq with
	{role=v} when v = fcgi_responder  -> fcgi_read_responder_request con id
      | {role=v} when v = fcgi_authorizer -> fcgi_read_authorizer_request con id
      | {role=v} when v = fcgi_filter     -> fcgi_read_filter_request con id
      | {role=_}                          -> failwith "FCGI Role Not Supported"

let fcgi_write_end_request req endreq =
  ignore (write req.con (encode_fcgi_end_request endreq req.id) 0 16)

let fcgi_write_stdout req data =
  ignore (write req.con (encode_fcgi_data data req.id fcgi_stdout) 0 (8 + (String.length data)))

let fcgi_write_stderr req data = 
  ignore (write req.con (encode_fcgi_data data req.id fcgi_stderr) 0 (8 + (String.length data)))

(* check that the ip address is correct *)
let check_addr addr = true

(* veryify that we are talking to fastcgi *)
let fcgi_verify unit = 
  try
    let socaddr = getpeername Unix.stdin in false
  with Unix_error(ENOTCONN, _, _) -> true
    | Unix_error(_, _, _) -> false  

(* accept a conn from fastcgi *)
let fcgi_accept unit =
  try
    let (con, addr) = accept Unix.stdin in
      (if (fcgi_verify ()) = false then
	 failwith "you are not fastcgi");
      (if (check_addr addr) = false then
	 failwith "invalid address");
      fcgi_read_request con
  with Unix_error(e, s, t) -> 
    prerr_endline (error_message e); 
    failwith "An Unrecoverable error has occurred";;

(* destroy a connection when we're done with it *)
let fcgi_destroy req = close req.con

(* ======================================================================
 * History:
 * 
 * $Log: netcgi_fcgi_10.ml,v $
 * Revision 1.1  2003/10/07 17:39:32  stolpmann
 * 	Imported Eric's patch for fastcgi
 *
 * 
 *)
