(* ocamlgsl - OCaml interface to GSL                        *)
(* Copyright () 2002 - Olivier Andrieu                     *)
(* distributed under the terms of the GPL version 2         *)


type t
type accel
type interp_type =
  | LINEAR
  | POLYNOMIAL
  | CSPLINE
  | CSPLINE_PERIODIC
  | AKIMA
  | AKIMA_PERIODIC

external alloc : interp_type -> int -> t 
    = "ml_gsl_interp_alloc"

external free : t -> unit
    = "ml_gsl_interp_free"

external _init : t -> float array -> float array -> int -> unit
    = "ml_gsl_interp_init"

let init i x y = 
  let lx = Array.length x in
  let ly = Array.length y in
  if lx <> ly
  then invalid_arg "Gsl_interp: init" ;
  _init i x y lx

external name : t -> string 
    = "ml_gsl_interp_name"

external min_size : t -> int
    = "ml_gsl_interp_min_size"

external accel_alloc : unit -> accel
    = "ml_gsl_interp_accel_alloc"

external accel_free : accel -> unit
    = "ml_gsl_interp_accel_free"

external i_eval : t -> float array -> float array 
  -> float -> accel -> float
      = "ml_gsl_interp_eval"

external i_eval_deriv : t -> float array -> float array 
  -> float -> accel -> float
      = "ml_gsl_interp_eval_deriv"

external i_eval_deriv2 : t -> float array -> float array 
  -> float -> accel -> float
      = "ml_gsl_interp_eval_deriv2"

external i_eval_integ : t -> float array -> float array 
  -> float -> float -> accel -> float
      = "ml_gsl_interp_eval_integ_bc" "ml_gsl_interp_eval_integ"


(* Higher level functions *)
type interp = {
    interp : t ;
    accel  : accel ;
    xa     : float array ;
    ya     : float array ;
    size   : int ;
    i_type : interp_type ;
  }

let make i_type x y =
  let len = Array.length x in
  let ly = Array.length y in
  if len <> ly
  then invalid_arg "Gsl_interp.make" ;
  let t = alloc i_type len in
  let a = accel_alloc () in
  init t x y ;
  Gc.finalise free t ;
  Gc.finalise accel_free a ;
  { interp=t; accel = a ;
    xa=x; ya=y; size=len; i_type=i_type }

let eval interp x =
  i_eval interp.interp interp.xa interp.ya x interp.accel

external eval_array : interp -> float array -> float array -> unit
    = "ml_gsl_interp_eval_array"

let eval_deriv interp x =
  i_eval_deriv interp.interp interp.xa interp.ya x interp.accel

let eval_deriv2 interp x =
  i_eval_deriv2 interp.interp interp.xa interp.ya x interp.accel

let eval_integ interp a b  =
  i_eval_integ interp.interp interp.xa interp.ya a b interp.accel
