/*
 * Copyright 2007 Samuel Mimram
 *
 * This file is part of ocaml-vorbis.
 *
 * ocaml-vorbis is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * ocaml-vorbis is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Ocaml-vorbis; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * Libvorbis bindings for OCaml.
 *
 * @author Samuel Mimram
 */

#define CAML_NAME_SPACE
#include <caml/alloc.h>
#include <caml/callback.h>
#include <caml/custom.h>
#include <caml/fail.h>
#include <caml/memory.h>
#include <caml/misc.h>
#include <caml/mlvalues.h>
#include <caml/signals.h>

#include <vorbis/vorbisfile.h>
#include <vorbis/vorbisenc.h>
#include <vorbis/codec.h>

#include <string.h>
#include <stdio.h>
#include <time.h>
#include <assert.h>

#include <ocaml-ogg.h>

static void raise_err(int ret)
{
  switch(ret)
  {
    case OV_HOLE:
      caml_raise_constant(*caml_named_value("vorbis_exn_hole_in_data"));

    case OV_EREAD:
      caml_raise_constant(*caml_named_value("vorbis_exn_read_error"));

    case OV_EFAULT:
      caml_raise_constant(*caml_named_value("vorbis_exn_internal_fault"));

    case OV_ENOTVORBIS:
      caml_raise_constant(*caml_named_value("vorbis_exn_not_vorbis"));

    case OV_EBADHEADER:
      caml_raise_constant(*caml_named_value("vorbis_exn_bad_header"));

    case OV_EVERSION:
      caml_raise_constant(*caml_named_value("vorbis_exn_version_mismatch"));

    case OV_EBADLINK:
      caml_raise_constant(*caml_named_value("vorbis_exn_bad_link"));

    case OV_EINVAL:
      caml_raise_constant(*caml_named_value("vorbis_exn_invalid"));

    case OV_EIMPL:
      caml_raise_constant(*caml_named_value("vorbis_exn_not_implemented"));

    default:
      caml_raise_with_arg(*caml_named_value("vorbis_exn_unknown_error"), Val_int(ret));
  }
}

static void check_err(int ret)
{
  if (ret)
    raise_err(ret);
}


/***** Decoding *****/


/*
CAMLprim value ocaml_vorbis_feed(value sync_state, value buffer, value _offs, value _len)
{
  ogg_sync_state *oy = Sync_state_val(sync_state);
  int offs = Int_val(_offs);
  int len = Int_val(_len);
  char *outbuf = ogg_sync_buffer(oy, len);

  memcpy(outbuf, String_val(buffer) + offs, len);
  ogg_sync_wrote(oy, len);

  return Val_unit;
}

CAMLprim value ocaml_vorbis_synthesis_init(value unit)
{
  vorbis_info *vi = malloc(sizeof(vorbis_info));
  value ans;

  vorbis_info_init(vi);
  // TODO!
  ans = new_dsp_state();
  vorbis_analysis_init(Dsp_state_val(ans), vi);

  return ans;
}
*/


/***** Encoding *****/


typedef struct
{
  vorbis_dsp_state vd;
  vorbis_block vb;
  vorbis_info vi;
  ogg_packet op;
} encoder_t;

#define Encoder_val(v) (*((encoder_t**)Data_custom_val(v)))
#define Dsp_state_val(v) (&Encoder_val(v)->vd)

static void finalize_encoder(value e)
{
  encoder_t *enc = Encoder_val(e);

  vorbis_block_clear(&enc->vb);
  vorbis_dsp_clear(&enc->vd);
  vorbis_info_clear(&enc->vi);
  free(enc);
}

static struct custom_operations encoder_ops =
{
  "ocaml_vorbis_encoder",
  finalize_encoder,
  custom_compare_default,
  custom_hash_default,
  custom_serialize_default,
  custom_deserialize_default
};

CAMLprim value ocaml_vorbis_analysis_init(value channels, value rate, value max_bitrate, value nominal_bitrate, value min_bitrate)
{
  encoder_t *enc = malloc(sizeof(encoder_t));
  value ans;

  vorbis_info_init(&enc->vi);
  /* TODO: clear info on error */
  check_err(vorbis_encode_init(&enc->vi, Int_val(channels), Int_val(rate), Int_val(max_bitrate), Int_val(nominal_bitrate), Int_val(min_bitrate)));
  vorbis_analysis_init(&enc->vd, &enc->vi);
  vorbis_block_init(&enc->vd, &enc->vb);
  ans = caml_alloc_custom(&encoder_ops, sizeof(encoder_t*), 1, 0);
  Encoder_val(ans) = enc;

  return ans;
}

CAMLprim value ocaml_vorbis_analysis_init_vbr(value channels, value rate, value quality)
{
  encoder_t *enc = malloc(sizeof(encoder_t));
  value ans;

  vorbis_info_init(&enc->vi);
  /* TODO: clear info on error */
  check_err(vorbis_encode_init_vbr(&enc->vi, Int_val(channels), Int_val(rate), Double_val(quality)));
  vorbis_analysis_init(&enc->vd, &enc->vi);
  vorbis_block_init(&enc->vd, &enc->vb);
  ans = caml_alloc_custom(&encoder_ops, sizeof(encoder_t*), 1, 0);
  Encoder_val(ans) = enc;

  return ans;
}

CAMLprim value ocaml_vorbis_reset(value vdsp)
{
  encoder_t *enc = Encoder_val(vdsp);

  vorbis_block_clear(&enc->vb);
  vorbis_dsp_clear(&enc->vd);
  vorbis_info_clear(&enc->vi);

  vorbis_analysis_init(&enc->vd, &enc->vi);
  vorbis_block_init(&enc->vd, &enc->vb);

  return Val_unit;
}

CAMLprim value ocaml_vorbis_analysis_headerout(value vdsp, value vogg, value comments)
{
  vorbis_dsp_state *vd = Dsp_state_val(vdsp);
  ogg_stream_state *os = Stream_state_val(vogg);
  vorbis_comment vc;
  ogg_packet header, header_comm, header_code;
  int i;

  vorbis_comment_init(&vc);
  for(i = 0; i < Wosize_val(comments); i++)
    vorbis_comment_add_tag(&vc, String_val(Field(Field(comments, i), 0)), String_val(Field(Field(comments, i), 1)));
  vorbis_analysis_headerout(vd, &vc, &header, &header_comm, &header_code);
  vorbis_comment_clear(&vc);
  ogg_stream_packetin(os, &header);
  ogg_stream_packetin(os, &header_comm);
  ogg_stream_packetin(os, &header_code);

  return Val_unit;
}

CAMLprim value ocaml_vorbis_encode_float(value vdsp, value vogg, value data, value _offs, value _len)
{
  CAMLparam3(vdsp, vogg, data);
  encoder_t *enc = Encoder_val(vdsp);
  vorbis_block *vb = &enc->vb;
  vorbis_dsp_state *vd = Dsp_state_val(vdsp);
  ogg_stream_state *os = Stream_state_val(vogg);
  ogg_packet *op = &enc->op;
  int offs = Int_val(_offs);
  int len = Int_val(_len);
  float **vorbis_buffer;
  int c, i;
  value datac;

  /* TODO: check for consistency of data */

  vorbis_buffer = vorbis_analysis_buffer(vd, len);
  for(c = 0; c < Wosize_val(data); c++)
  {
    datac = Field(data, c);
    for(i = 0; i < len; i++)
      vorbis_buffer[c][i] = Double_field(datac, i + offs);
  }

  caml_enter_blocking_section();
  vorbis_analysis_wrote(vd, len);

  /* TODO: split the encoding part? */

  while(vorbis_analysis_blockout(vd, vb) == 1)
  {
    /* Analysis, assume we want to use bitrate management. */
    vorbis_analysis(vb, NULL);
    vorbis_bitrate_addblock(vb);

    /* Weld packets into the bitstream. */
    while(vorbis_bitrate_flushpacket(vd, op))
      ogg_stream_packetin(os, op);
  }
  caml_leave_blocking_section();

  CAMLreturn(Val_unit);
}

/*
CAMLprim value ocaml_vorbis_eos(value vdsp, value vogg)
{
  CAMLparam2(vdsp, vogg);
  encoder_t *enc = Encoder_val(vdsp);
  ogg_stream_state *os = Stream_state_val(vogg);

  caml_enter_blocking_section();
  // Tell libvorbis that this is the end of the stream
  vorbis_analysis_wrote(&enc->vd, 0);
  while(vorbis_analysis_blockout(&enc->vd, &enc->vb) == 1)
  {
    vorbis_analysis(&enc->vb, NULL);
    vorbis_bitrate_addblock(&enc->vb);
    while(vorbis_bitrate_flushpacket(&enc->vd, &enc->op))
      ogg_stream_packetin(os, &enc->op);
  }
  caml_leave_blocking_section();

  CAMLreturn(Val_unit);
}
*/


/***** Decoding *****/


/* This should be malloced since we might want to register *_func as global root. */
typedef struct
{
  OggVorbis_File *ovf;
  int bitstream;
  value read_func;
  value seek_func;
  value close_func;
  value tell_func;
} myvorbis_dec_file_t;

#define Decfile_val(v) (*((myvorbis_dec_file_t**)Data_custom_val(v)))

static void finalize_dec_file(value _df)
{
  myvorbis_dec_file_t *df = Decfile_val(_df);

  // TODO: check that close has been done, otherwise df->ovh leaks

  free(df);
}

static struct custom_operations decfile_ops =
{
  "ocaml_vorbis_decfile",
  finalize_dec_file,
  custom_compare_default,
  custom_hash_default,
  custom_serialize_default,
  custom_deserialize_default
};

static size_t read_func_cb(void *ptr, size_t size, size_t nmemb, void *datasource)
{
  myvorbis_dec_file_t *df = datasource;
  value ret;
  int len;

  caml_leave_blocking_section();
  ret = caml_callback(df->read_func, Val_int(size*nmemb));
  len = Int_val(Field(ret,1));
  memcpy(ptr, String_val(Field(ret,0)), len);
  caml_enter_blocking_section();

  return len;
}

static int seek_func_cb(void *datasource, ogg_int64_t offset, int whence)
{
  myvorbis_dec_file_t *df = datasource;
  int cmd;
  int ret;

  switch(whence)
  {
    case SEEK_SET:
      cmd = 0;
      break;

    case SEEK_CUR:
      cmd = 1;
      break;

    case SEEK_END:
      cmd = 2;
      break;

    default:
      assert(0);
  }
  caml_leave_blocking_section();
  ret = Int_val(caml_callback2(df->seek_func, Val_int(offset), Val_int(cmd)));
  caml_enter_blocking_section();

  return ret;
}

static int close_func_cb(void *datasource)
{
  myvorbis_dec_file_t *df = datasource;

  caml_leave_blocking_section();
  caml_callback(df->close_func, Val_unit);
  caml_enter_blocking_section();

  return 0;
}

static long tell_func_cb(void *datasource)
{
  myvorbis_dec_file_t *df = datasource;
  int ret;

  caml_leave_blocking_section();
  ret = Int_val(caml_callback(df->tell_func, Val_unit));
  caml_enter_blocking_section();

  return ret;
}

static ov_callbacks callbacks =
{
  .read_func = read_func_cb,
  .seek_func = seek_func_cb,
  .close_func = close_func_cb,
  .tell_func = tell_func_cb
};

CAMLprim value ocaml_vorbis_open_dec_stream(value read_func, value seek_func, value close_func, value tell_func, value params)
{
  CAMLparam5(read_func, seek_func, close_func, tell_func, params);
  CAMLlocal1(block);
  int ret = 0;
  myvorbis_dec_file_t *df;

  df = malloc(sizeof(myvorbis_dec_file_t));

  df->ovf = (OggVorbis_File*)malloc(sizeof(OggVorbis_File));
  df->bitstream = 0;
  df->read_func = read_func;
  caml_register_global_root(&df->read_func);
  df->seek_func = seek_func;
  caml_register_global_root(&df->seek_func);
  df->close_func = close_func;
  caml_register_global_root(&df->close_func);
  df->tell_func = tell_func;
  caml_register_global_root(&df->tell_func);

  caml_enter_blocking_section();
  ret = ov_open_callbacks(df, df->ovf, NULL, 0, callbacks);
  caml_leave_blocking_section();

  if(ret < 0)
  {
    caml_remove_global_root(&df->tell_func);
    caml_remove_global_root(&df->close_func);
    caml_remove_global_root(&df->seek_func);
    caml_remove_global_root(&df->read_func);
    free(df->ovf);
    free(df);
    raise_err(ret);
  }

  block = caml_alloc_custom(&decfile_ops, sizeof(myvorbis_dec_file_t*),
                            sizeof(myvorbis_dec_file_t)+sizeof(OggVorbis_File),
                            1000000);
  Decfile_val(block) = df;

  CAMLreturn(block);
}

CAMLprim value ocaml_vorbis_decode(value d_f, value be_, value ss_, value signed_, value buf_, value ofs_, value len_)
{
  CAMLparam2(d_f, buf_);

  myvorbis_dec_file_t *df = Decfile_val(d_f);
  int ret = 0;
  int ofs = Int_val(ofs_);
  int len = Int_val(len_);
  int big_endian = Bool_val(be_);
  int sample_size = Int_val(ss_);
  int sign = Bool_val(signed_);
  char* buf;

  if (!df->ovf)
    caml_raise_constant(*caml_named_value("vorbis_exn_invalid_parameters"));
  if (ofs + len > caml_string_length(buf_))
    caml_raise_constant(*caml_named_value("vorbis_exn_invalid_parameters"));
  /* TODO: this buffer could be allocated once when creating the decoder
   * and reused for every decoding pass. This might be useful to reduce
   * load or memory fragmentation if needed.
   */
  buf = malloc(len);

  /* We have to make sure that when a callback is called, the ocaml master lock
   * has been released.  Callbacks are responsible for taking it back if they
   * need to call ocaml code.
   */
  caml_enter_blocking_section();
  ret = ov_read(df->ovf, buf, len, big_endian, sample_size, sign, &df->bitstream);
  caml_leave_blocking_section();

  if (ret <= 0)
  {
    free(buf);
    ret?raise_err(ret):caml_raise_end_of_file();
  }
  memcpy(String_val(buf_) + ofs, buf, ret);
  free(buf);

  CAMLreturn(Val_int(ret));
}

CAMLprim value ocaml_vorbis_decode_byte(value *argv, int argn)
{
  return ocaml_vorbis_decode(argv[0], argv[1], argv[2], argv[3], argv[4], argv[5], argv[6]);
}

CAMLprim value ocaml_vorbis_decode_float(value d_f, value dst, value ofs_, value len_)
{
  CAMLparam2(d_f, dst);

  myvorbis_dec_file_t *df = Decfile_val(d_f);
  int ret = 0;
  int ofs = Int_val(ofs_);
  int len = Int_val(len_);
  float **buf;
  int chans,c,i;

  if (!df->ovf)
    caml_raise_constant(*caml_named_value("vorbis_exn_invalid_parameters"));
  chans = df->ovf->vi->channels;

  if (chans > Wosize_val(dst))
    caml_raise_constant(*caml_named_value("vorbis_exn_invalid_parameters"));
  if (Wosize_val(dst) < 1 || Wosize_val(Field(dst,0)) / Double_wosize - ofs < len)
    caml_raise_constant(*caml_named_value("vorbis_exn_invalid_parameters"));

  /* We have to make sure that when a callback is called, the ocaml master lock
   * has been released.  Callbacks are responsible for taking it back if they
   * need to call ocaml code.
   */
  caml_enter_blocking_section();
  ret = ov_read_float(df->ovf, &buf, len, &df->bitstream);
  caml_leave_blocking_section();

  if (ret <= 0)
    ret?raise_err(ret):caml_raise_end_of_file();

  for (c = 0; c < chans; c++)
    for (i = 0; i < ret; i++)
      Store_double_field(Field(dst, c), i + ofs, buf[c][i]);

  CAMLreturn(Val_int(ret));
}

CAMLprim value ocaml_vorbis_decode_float_alloc(value d_f, value len_)
{
  CAMLparam1(d_f);
  CAMLlocal2(ans, ansc);

  myvorbis_dec_file_t *df = Decfile_val(d_f);
  int ret = 0;
  int len = Int_val(len_);
  float **buf;
  int chans, c, i;

  if (!df->ovf)
    caml_raise_constant(*caml_named_value("vorbis_exn_invalid_parameters"));
  chans = df->ovf->vi->channels;

  /* We have to make sure that when a callback is called, the ocaml master lock
   * has been released.  Callbacks are responsible for taking it back if they
   * need to call ocaml code.
   */
  caml_enter_blocking_section();
  ret = ov_read_float(df->ovf, &buf, len, &df->bitstream);
  caml_leave_blocking_section();

  if (ret <= 0)
    ret?raise_err(ret):caml_raise_end_of_file();

  ans = caml_alloc_tuple(chans);
  for (c = 0; c < chans; c++)
  {
    ansc = caml_alloc(ret * Double_wosize, Double_array_tag);
    Store_field(ans, c, ansc);
    for (i = 0; i < ret; i++)
      Store_double_field(ansc, i, buf[c][i]);
  }

  CAMLreturn(ans);
}

CAMLprim value ocaml_vorbis_close_dec_file(value d_f)
{
  CAMLparam1(d_f);
  myvorbis_dec_file_t* df = Decfile_val(d_f);

  if (df->ovf)
  {
    caml_enter_blocking_section();
    ov_clear(df->ovf);
    caml_leave_blocking_section();
    free(df->ovf);
    df->ovf=NULL;
  }
  if(df->read_func)
  {
    caml_remove_global_root(&df->read_func);
    caml_remove_global_root(&df->seek_func);
    caml_remove_global_root(&df->close_func);
    caml_remove_global_root(&df->tell_func);
    df->read_func=0;
  }

  CAMLreturn(Val_unit);
}

CAMLprim value ocaml_vorbis_get_dec_file_bitstream(value d_f)
{
  myvorbis_dec_file_t *df = Decfile_val(d_f);
  return Val_int(df->bitstream);
}

CAMLprim value ocaml_vorbis_decoder_info(value d_f, value bs)
{
  CAMLparam1(d_f);
  CAMLlocal1(ans);
  myvorbis_dec_file_t *df = Decfile_val(d_f);
  int bitstream = Int_val(bs);
  vorbis_info *vi;

  caml_enter_blocking_section();
  vi = ov_info(df->ovf, bitstream);
  caml_leave_blocking_section();

  assert(vi);
  ans = caml_alloc_tuple(7);
  Store_field(ans, 0, Val_int(vi->version));
  Store_field(ans, 1, Val_int(vi->channels));
  Store_field(ans, 2, Val_int(vi->rate));
  Store_field(ans, 3, Val_int(vi->bitrate_upper));
  Store_field(ans, 4, Val_int(vi->bitrate_nominal));
  Store_field(ans, 5, Val_int(vi->bitrate_lower));
  Store_field(ans, 6, Val_int(vi->bitrate_window));

  CAMLreturn(ans);
}

CAMLprim value ocaml_vorbis_get_dec_file_comments(value d_f, value link_)
{
  CAMLparam2(d_f, link_);
  CAMLlocal2(ans, cmts);

  myvorbis_dec_file_t *df = Decfile_val(d_f);
  int link = Int_val(link_);
  int i;
  vorbis_comment *vc;

  caml_enter_blocking_section();
  vc = ov_comment(df->ovf, link);
  caml_leave_blocking_section();

  if (!vc)
    /* TODO: better error */
    caml_raise_with_arg(*caml_named_value("vorbis_exn_unknown_error"), Val_int(666));

  cmts = caml_alloc_tuple(vc->comments);
  for (i = 0; i < vc->comments; i++)
    Store_field(cmts, i, caml_copy_string(vc->user_comments[i]));
  ans = caml_alloc_tuple(2);
  Store_field(ans, 0, caml_copy_string(vc->vendor));
  Store_field(ans, 1, cmts);

  CAMLreturn(ans);
}

CAMLprim value ocaml_vorbis_decoder_bitrate(value d_f, value bs)
{
  CAMLparam1(d_f);
  myvorbis_dec_file_t *df = Decfile_val(d_f);
  int bitstream = Int_val(bs);
  long ret;

  caml_enter_blocking_section();
  ret = ov_bitrate(df->ovf, bitstream);
  caml_leave_blocking_section();

  CAMLreturn(Val_int(ret));
}

CAMLprim value ocaml_vorbis_decoder_time_total(value d_f, value bs)
{
  CAMLparam1(d_f);
  myvorbis_dec_file_t *df = Decfile_val(d_f);
  int bitstream = Int_val(bs);
  double ret;

  caml_enter_blocking_section();
  ret = ov_time_total(df->ovf, bitstream);
  caml_leave_blocking_section();

  CAMLreturn(caml_copy_double(ret));
}

CAMLprim value ocaml_vorbis_decoder_pcm_total(value d_f, value bs)
{
  CAMLparam1(d_f);
  myvorbis_dec_file_t *df = Decfile_val(d_f);
  int bitstream = Int_val(bs);
  ogg_int64_t ret;

  caml_enter_blocking_section();
  ret = ov_pcm_total(df->ovf, bitstream);
  caml_leave_blocking_section();

  CAMLreturn(Val_int(ret));
}

CAMLprim value ocaml_vorbis_decoder_streams(value d_f)
{
  CAMLparam1(d_f);
  myvorbis_dec_file_t *df = Decfile_val(d_f);
  long ret;

  caml_enter_blocking_section();
  ret = ov_streams(df->ovf);
  caml_leave_blocking_section();

  CAMLreturn(Val_int(ret));
}

CAMLprim value ocaml_vorbis_decoder_serialnumber(value d_f, value bs)
{
  CAMLparam1(d_f);
  myvorbis_dec_file_t *df = Decfile_val(d_f);
  int bitstream = Int_val(bs);
  long ret;

  caml_enter_blocking_section();
  ret = ov_serialnumber(df->ovf, bitstream);
  caml_leave_blocking_section();

  CAMLreturn(Val_int(ret));
}
