(** Represents OCaml and Reason source files *)
open! Stdune
open! Import

module Name : sig
  (** Represents a valid OCaml module name *)
  type t

  val to_dyn : t -> Dyn.t
  include Dune_lang.Conv with type t := t

  val add_suffix : t -> string -> t

  val compare : t -> t -> Ordering.t
  val of_string : string -> t
  val to_string : t -> string

  val uncapitalize : t -> string

  val pp_quote : Format.formatter -> t -> unit

  module Set : sig
    include Set.S with type elt = t
    val to_dyn : t -> Dyn.t
  end

  module Map : Map.S with type key = t

  module Infix : Comparator.OPS with type t = t

  val of_local_lib_name : Lib_name.Local.t -> t

  val to_local_lib_name : t -> Lib_name.Local.t
end

module File : sig
  type t =
    { path    : Path.t
    ; dialect : Dialect.t
    }

  val path : t -> Path.t

  val make : Dialect.t -> Path.t -> t
end

module Kind : sig
  type t = Intf_only | Virtual | Impl | Alias | Impl_vmodule | Wrapped_compat

  include Dune_lang.Conv with type t := t
end

module Source : sig
  (** Only the source of a module, not yet associated to a library *)
  type t

  val name : t -> Name.t

  val make
    :  ?impl:File.t
    -> ?intf:File.t
    -> Name.t
    -> t

  val has : t -> ml_kind:Ml_kind.t -> bool

  val src_dir : t -> Path.t
end


type t

val kind : t -> Kind.t

val to_dyn : t -> Dyn.t

(** [obj_name] Object name. It is different from [name] for wrapped modules. *)
val of_source
  :  ?obj_name:string
  -> visibility:Visibility.t
  -> kind:Kind.t
  -> Source.t
  -> t

val name : t -> Name.t

(** Real unit name once wrapped. This is always a valid module name. *)
val real_unit_name : t -> Name.t

val source : t -> ml_kind:Ml_kind.t -> File.t option

val pp_flags : t -> (unit, string list) Build.t option

val file            : t -> ml_kind:Ml_kind.t -> Path.t option

val obj_name : t -> string

val iter : t -> f:(Ml_kind.t -> File.t -> unit) -> unit

val has : t -> ml_kind:Ml_kind.t -> bool

(** Prefix the object name with the library name. *)
val with_wrapper : t -> main_module_name:Name.t -> t

val map_files : t -> f:(Ml_kind.t -> File.t -> File.t) -> t

(** Set preprocessing flags *)
val set_pp : t -> (unit, string list) Build.t option -> t

val wrapped_compat : t -> t

module Name_map : sig
  type module_
  type t = module_ Name.Map.t

  val decode : src_dir:Path.t -> t Dune_lang.Decoder.t

  val encode : t -> Dune_lang.t list

  val to_dyn : t -> Dyn.t

  val impl_only : t -> module_ list

  val of_list_exn : module_ list -> t

  val singleton : module_ -> t

  val add : t -> module_ -> t

  val by_obj : t -> t
end with type module_ := t

module Obj_map : sig
  type module_
  include Map.S with type key = module_

  val find_exn : 'a t -> module_ -> 'a

  val top_closure
    :  module_ list t
    -> module_ list
    -> (module_ list, module_ list) Result.result
end with type module_ := t

val sources : t -> Path.t list

val visibility : t -> Visibility.t

val encode : t -> Dune_lang.t list

val decode : src_dir:Path.t -> t Dune_lang.Decoder.t

(** [pped m] return [m] but with the preprocessed source paths paths *)
val pped : t -> t

(** [ml_source m] returns [m] but with the OCaml syntax source paths *)
val ml_source : t -> t

val set_src_dir : t -> src_dir:Path.t -> t

(** Represent a module that is generated by Dune itself. We use a
    special ".ml-gen" extension to indicate this fact and hide it from
    [(glob_files *.ml)]. *)
val generated : src_dir:Path.t -> Name.t -> t

(** Represent the generated alias module. *)
val generated_alias : src_dir:Path.Build.t -> Name.t -> t
