package edu.ucsb.cs.nws.nwsprotocol;

import java.io.*;


/**
 * The NwsForecastAPI implements the FORECAST_API calls. It defines the data
 * structures too (measurement and forecast and the like).
 */
public class NwsForecastAPI {
	/** A single forecast generated from a list of Measurements. */
	public static class Forecast {
		/** The forecast of the next value in the series. */
		public double forecast;
		/** An estimation of the accuracy of the forecast. */
		public double error;
		public double MAEerror;
		/** The index of the method that
		 * produced the forecast */
		public int methodUsed;

		/** Produces a Forecast with the specified field values. */
		public Forecast(double forecast,
				double error,
				double MAEerror,
				int methodUsed) {
			this.forecast = forecast;
			this.error = error;
			this.MAEerror = error;
			this.methodUsed = methodUsed;
		}

		/** Produces a Forecast initialized by reading <i>stream</i>. */
		public Forecast(DataInputStream stream) throws Exception {
			forecast = stream.readDouble();
			error = stream.readDouble();
			MAEerror = stream.readDouble();
			methodUsed = stream.readInt();
		}

		/** Returns the fields of the Forecast converted to a
		 * byte array. */
		public byte[] toBytes() {
			byte[][] allBytes = {NwsMessage.toBytes(forecast),
					NwsMessage.toBytes(error),
					NwsMessage.toBytes(MAEerror),
					NwsMessage.toBytes(methodUsed)};
			return NwsMessage.concatenateBytes(allBytes);
		}

	}


	/** A set of forecast information produced from a single measurement. */
	public static class ForecastCollection {
		public static final int FORECAST_TYPE_COUNT = 2;
		public static final int MAE_FORECAST = 0;
		public static final int MSE_FORECAST = 1;

		/** The measurement from which the forecasts were produced. */
		public Measurement measurement;
		/** A pair of forecasts based on the methods which
		 * produce the lowest mean absolute error and mean square
		 * error.  */
		public Forecast[] forecasts;

		/** Produces a ForecastCollection with the specified
		 * field values. */
		public ForecastCollection(Measurement measurement,
					Forecast[] forecasts) {
			this.measurement = measurement;
			this.forecasts = forecasts;
		}

		/** Produces a ForecastCollection initialized by reading
		 * <i>stream</i>. */
		public ForecastCollection(DataInputStream stream) throws Exception {
			this.measurement = new Measurement(stream);
			this.forecasts = new Forecast[FORECAST_TYPE_COUNT];
			for(int i = 0; i < FORECAST_TYPE_COUNT; i++)
				this.forecasts[i] = new Forecast(stream);
		}

		/** Returns the fields of the ForecastCollection
		 * converted to a byte array.  */
		public byte[] toBytes() {
			byte[][] allBytes = {measurement.toBytes(),
				forecasts[MAE_FORECAST].toBytes(),
				forecasts[MSE_FORECAST].toBytes()};
			return NwsMessage.concatenateBytes(allBytes);
		}
	}


	/** A single measurement of resource availability. */
	public static class Measurement {
		/** The time (seconds since midnight 1/1/1970) the
		 * measurement was taken. */
		public double timeStamp;
		/** The observed availability at that time. */
		public double measurement;

		/** Produces a Measurement with the specified field values. */
		public Measurement(	double timeStamp,
					double measurement) {
			this.timeStamp = timeStamp;
			this.measurement = measurement;
		}

		/** Produces a Measurement initialized by reading
		 * <i>stream</i>. */
		public Measurement(DataInputStream stream) throws Exception {
			timeStamp = stream.readDouble();
			measurement = stream.readDouble();
		}

		/** Returns the fields of the Measurement converted to a
		 * byte array. */
		public byte[] toBytes() {
			byte[][] allBytes = {NwsMessage.toBytes(timeStamp),
					NwsMessage.toBytes(measurement)};
			return NwsMessage.concatenateBytes(allBytes);
		}
	}
}
