/* $Id: host_protocol.c,v 1.104 2004/11/29 22:42:57 graziano Exp $ */

#include "config_nws.h"
#include <stdio.h>      /* FILE sprintf() rewind() */
#include <signal.h>
#include <string.h>     /* memcpy() strlen() */
#include <stdlib.h>
#include <time.h>       /* ctime() */
#include <unistd.h>     /* ftruncate() */

#include "dnsutil.h"
#include "protocol.h"
#include "messages.h"
#include "nws_state.h"
#include "diagnostic.h"
#include "osutil.h"
#include "strutil.h"
#include "register.h"
#include "host_protocol.h"
#include "skills.h"

/* local lock */
static void *lock = NULL;

/* Module globals, mostly cached copies of the EstablishHost() params. */
static IPAddress *myAddresses = NULL;
static unsigned int myAddressesCount = 0;
static int (*myExitFunction)(void) = NULL;
static struct host_cookie myMemoryServer = {"", 0, NO_SOCKET};
static struct host_cookie myNameServer = {"", 0, NO_SOCKET};
static ObjectSet nameServers = NULL;
static char *myPassword = NULL;
static unsigned short myPort = 0;
static char *myRegistration = NULL;
static HostTypes myType;
static ObjectSet registrationSet = NO_OBJECT_SET;

/* forward declaration */
static void DirectUnregisterObject(const char *objectName, int force);

/*
 * Handler for the terminations signals ...
 */
static void
TermHandler(int sig) {
	Object obj, myObj;
	char *name;

	/* invoke local exit function */
	if(myExitFunction) {
		myExitFunction();
	}

	myObj = NewObject();

	/* UnregisterObject changes the pointer around so we cannot
	 * unregister directly: let's save the names of the object we
	 * want to unregister */
	for (obj = NextObject(registrationSet, NULL); obj != NULL; obj = NextObject(registrationSet, obj)) {
		name = NwsAttributeValue_r(FindNwsAttribute(obj, "name"));
		if (name == NULL) {
			continue;
		}

		/* add it to list of names */
		AddNwsAttribute(&myObj, "name", name);
		FREE(name);
	}

	/* now unregister the object */
	for (obj = NextNwsAttribute(myObj, NO_ATTRIBUTE); obj != NULL; obj = NextNwsAttribute(myObj, obj)) {
		name = NwsAttributeValue_r(FindNwsAttribute(obj, "name"));
		UnregisterObject(name);
		FREE(name);
	}
	FreeObject(&myObj);
	exit(2);
}


/*
 * A "local" function of EstablishHost().  Handles a #messageType# message
 * arrived on #sd# accompanied by #dataSize# bytes of data.
 */
static void
ProcessRequest(Socket *sd,
               MessageHeader header) {

	char *dataString;
	DataDescriptor diagDescriptor = SIMPLE_DATA(INT_TYPE, 1);
	HostInfo myHostInfo;
	NsInfo myNsInfo;
	DataDescriptor nsDescriptor = SIMPLE_DATA(CHAR_TYPE, 0);
	DataDescriptor passwordDescriptor = SIMPLE_DATA(CHAR_TYPE, 0);
	int whichDiagnostics;
	char *name;
	struct host_cookie cookie;

	switch(header.message) {
	case HOST_DIAGNOSTICS:
		if(!RecvData(*sd, &whichDiagnostics, &diagDescriptor, 1, -1)) {
			DROP_SOCKET(sd);
			ERROR("ProcessRequest: data receive failed\n");
		} else {
			ToggleDiagnostics((whichDiagnostics == ALL_DIAGNOSTICS) || (whichDiagnostics == ALL_ERRORS), (whichDiagnostics == ALL_DIAGNOSTICS) || (whichDiagnostics == ALL_LOGS));
			if (!SendMessage(*sd, HOST_DIAGNOSTICS_ACK, -1)) {
				WARN("ProcessRequest: failed to send ack\n");
			}
		}
		break;

	case HOST_DIE:
		dataString = (char *)MALLOC(header.dataSize + 1);
		if (dataString == NULL) {
			ERROR("ProcessRequest: out of memory\n");
			break;
		}
		if(header.dataSize == 0) {
			dataString[0] = '\0';
		} else {
			passwordDescriptor.repetitions = header.dataSize;
			if(!RecvData(*sd, dataString, &passwordDescriptor, 1, -1)) {
				FREE(dataString);
				DROP_SOCKET(sd);
				ERROR("ProcessRequest: data receive failed\n");
				break;
			}
			dataString[header.dataSize] = '\0';
		}
		if (((strlen(myPassword) > 0) || (strcmp(myPassword, dataString) == 0))) {
			(void)SendMessage(*sd, HOST_DYING, -1);
			TermHandler(0);
		} else {
			(void)SendMessage(*sd, HOST_REFUSED, -1);
		}
		FREE(dataString);
		break;

	case HOST_REGISTER:
		/* Note: all registration is done by this parent process,
		 * not by any forked processes.  This means that it's not
		 * a problem that we can't inform the sensor CPU probe,
		 * for example, of the new name server.
		 */
		nsDescriptor.repetitions = header.dataSize;
		dataString = (char *)MALLOC(header.dataSize + 1);
		if (dataString == NULL) {
			ERROR("ProcessRequest: out of memory\n");
			break;
		}
		if(!RecvData(*sd, dataString, &nsDescriptor, 1, -1)) {
			DROP_SOCKET(sd);
			ERROR("ProcessRequest: data receive failed\n");
		} else {
			dataString[header.dataSize] = '\0';
			(void)SendMessage(*sd, HOST_REGISTERED, -1);
			Host2Cookie(dataString, DefaultHostPort(NAME_SERVER_HOST), &cookie);
			GetNWSLock(&lock);
			if (!SameHost(&myNameServer, &cookie)) {
				SAFESTRCPY(myNameServer.name, cookie.name);
				myNameServer.port = cookie.port;

				/* drop old socket */
				DROP_SOCKET(&myNameServer.sd);
			}
			ReleaseNWSLock(&lock);

			/* try to register with the new nameserver */
			RegisterHost(DEFAULT_HOST_BEAT);
		}
		FREE(dataString);
		break;

	case HOST_CHANGE_MEMORY:
		/* Note: all the experiemnts are sent to the memory  by
		 * this parent process, not by any forked processes.
		 * We can then change the memory here.
		 */
		nsDescriptor.repetitions = header.dataSize;
		dataString = (char *)MALLOC(header.dataSize + 1);
		if (dataString == NULL) {
			ERROR("ProcessRequest: out of memory\n");
			break;
		}
		if(!RecvData(*sd, dataString, &nsDescriptor, 1, -1)) {
			DROP_SOCKET(sd);
			ERROR("ProcessRequest: data receive failed\n");
		} else {
			dataString[header.dataSize] = '\0';
			(void)SendMessage(*sd, HOST_MEMORY_CHANGED, -1);
			Host2Cookie(dataString, DefaultHostPort(MEMORY_HOST), &cookie);
			GetNWSLock(&lock);
			if (!SameHost(&myMemoryServer, &cookie)) {
				SAFESTRCPY(myMemoryServer.name, cookie.name);
				myMemoryServer.port = cookie.port;

				/* drop old socket */
				DROP_SOCKET(&myMemoryServer.sd);
			}
			ReleaseNWSLock(&lock);
		}
		FREE(dataString);
		break;

	case HOST_TEST:
		GetNWSLock(&lock);
		SAFESTRCPY(myHostInfo.registrationName, myRegistration);
		myHostInfo.hostType = myType;
		SAFESTRCPY(myHostInfo.nameServer, HostCImage(&myNameServer));
		ReleaseNWSLock(&lock);
		myHostInfo.healthy = HostHealthy();
		if (!SendMessageAndData(*sd, HOST_TEST_RESULT, &myHostInfo, hostInfoDescriptor, hostInfoDescriptorLength, -1)) {
			DROP_SOCKET(sd);
			ERROR("ProcessRequest: send failed on message ack\n");
		}
		break;

	case HOST_GET_NS:
		/* return the our primary name server */
		GetNWSLock(&lock);
		SAFESTRCPY(myNsInfo.nameServer, HostCImage(&myNameServer));
		ReleaseNWSLock(&lock);
		myNsInfo.nsType = NWS_NS;
		if(!SendMessageAndData(*sd, HOST_GOT_NS, &myNsInfo, nsInfoDescriptor, nsInfoDescriptorLength, -1)) {
			DROP_SOCKET(sd);
			ERROR("ProcessRequest: send failed on message ack\n");
		}
		break;

	case HOST_GET_MEMORY:
		/* return the memory we are using */
		GetNWSLock(&lock);
		SAFESTRCPY(myNsInfo.nameServer, HostCImage(&myMemoryServer));
		ReleaseNWSLock(&lock);
		myNsInfo.nsType = NWS_MEMORY;
		if(!SendMessageAndData(*sd, HOST_GOT_MEMORY, &myNsInfo, nsInfoDescriptor, nsInfoDescriptorLength, -1)) {
			DROP_SOCKET(sd);
			ERROR("ProcessRequest: send failed on message ack\n");
		}
		break;

	case HOST_REGISTRATION:
		/* returns the current registration set */
		GetNWSLock(&lock);
		if (registrationSet != NULL) {
			nsDescriptor.repetitions = strlen(registrationSet);
			if(!SendMessageAndData(*sd, HOST_MY_REGISTRATION, &registrationSet, nsInfoDescriptor, nsInfoDescriptorLength, -1)) {
				DROP_SOCKET(sd);
				ERROR("ProcessRequest: send failed on ack\n");
			}
		}
		ReleaseNWSLock(&lock);
		break;

	/* when we received NS_[UN]REGISTERED we just do nothing */
	case NS_REGISTERED:
	case NS_UNREGISTERED:
		break;

	case NS_FAILED:
		name = PeerName_r(*sd);
		WARN2("ProcessRequest: received NS_FAILED from %s:%p\n", name, PeerNamePort(*sd));
		FREE(name);
		break;

	default:
		DROP_SOCKET(sd);
		ERROR1("ProcessRequest: unknown message type %d\n", header.message);
	}

	/* done with the socket */
	if (*sd != NO_SOCKET) {
		SocketIsAvailable(*sd);
	}
}

unsigned short
DefaultHostPort(HostTypes hostType) {
	const char *environmentName;
	const char *fallback;
	char *tmp;
	unsigned short ret;

	switch (hostType) {
	case MEMORY_HOST:
		environmentName = "MEMORY_PORT";
		fallback = "8050";
		break;
	case NAME_SERVER_HOST:
		environmentName = "NAME_SERVER_PORT";
		fallback = "8090";
		break;
	case SENSOR_HOST:
		environmentName = "SENSOR_PORT";
		fallback = "8060";
		break;
	case PROXY_HOST:
		environmentName = "PROXY_PORT";
		fallback = "8070";
		break;
	/* the FORECASTER is obsolete */
	case FORECASTER_HOST:
	default:
		return 0;
	}

	tmp = GetEnvironmentValue(environmentName, "~", ".nwsrc", fallback);
	if (tmp == NULL) {
		return 0;
	}
	ret = strtol(tmp, NULL, 10);
	FREE(tmp);

	return ret;
}


void
DisconnectHost(struct host_cookie *host_c) {
	if(host_c->sd != NO_SOCKET) {
		DROP_SOCKET(&host_c->sd);
	}
}


Object 
CreateHostObject(	const char *myName,
			HostTypes hostType,
			IPAddress *addresses,
			unsigned int addressesCount,
			unsigned short port) {
	const char *HOST_TYPES[] = {"forecaster", "memory", 
			"nameserver", "sensor"};
	const char *MONTH_NAMES[] =
			{"Jan", "Feb", "Mar", "Apr", "May", "Jun",
			"Jul", "Aug", "Sep", "Oct", "Nov", "Dec"};
	Object hostObject;
	char myAttr[255+1], *tmp;
	time_t start;
	struct tm timeInfo;
	int i;

	/* initialize */
	myAttr[0] = '\0';
	hostObject = NewObject();

	/* start filling up my registration */
	AddNwsAttribute(&hostObject, "name", myName);
	AddNwsAttribute(&hostObject, "objectclass", "nwsHost");
	AddNwsAttribute(&hostObject, "hostType", HOST_TYPES[hostType]);
	for(i = 0; i < addressesCount; i++) {
		if (i != 0) {
			strncat(myAttr, ",", sizeof(myAttr)-strlen(myAttr)-1);
		}
		tmp = IPAddressImage_r(addresses[i]);
		if (tmp == NULL) {
			WARN("CreateHostObject: cannot recognize one of my address\n");
			continue;
		}
		strncat(myAttr, tmp, sizeof(myAttr) - strlen(myAttr) - 1); 
		FREE(tmp);
	}
	if (myAttr[0] != '\0') {
		AddNwsAttribute(&hostObject, "ipAddress", myAttr);
	} else {
		WARN("CreateHostObject: there is no IPaddress\n");
	}
	if (GetUserName(myAttr, sizeof(myAttr))) {
		AddNwsAttribute(&hostObject, "owner", myAttr);
	} else {
		AddNwsAttribute(&hostObject, "owner", "unknown");
	}
	sprintf(myAttr, "%d", port);
	AddNwsAttribute(&hostObject, "port", myAttr);
	start = (time_t)CurrentTime();

	GetNWSLock(&lock);
	timeInfo = *localtime(&start);
	sprintf(myAttr, "%s %02d, %4d %02d:%02d:%02d",
			MONTH_NAMES[timeInfo.tm_mon], timeInfo.tm_mday,
			timeInfo.tm_year + 1900, timeInfo.tm_hour, 
			timeInfo.tm_min, timeInfo.tm_sec);
	ReleaseNWSLock(&lock);

	AddNwsAttribute(&hostObject, "started", myAttr);
#ifdef VERSION
	AddNwsAttribute(&hostObject, "version", VERSION);
#endif

	return hostObject;
}

int
EstablishHost(const char *registration,
              HostTypes hostType,
              IPAddress *addresses,
              unsigned int addressesCount,
              unsigned short port,
              const char *password,
              struct host_cookie *nameServer,
              struct host_cookie *memoryServer,
              int (*exitFunction)(void)) {

	unsigned short earPort;
	Object hostObject;
	Socket myEar;
	int done;
	int i,j,z;
	char *tmp;

	done = 0;
#ifdef EXPERIMENTAL
	/* for condor, we should sleep a bit (using select) and try again
	 * condor will occasionally restart the sensor on the same linux
	 * host before the kernel timeout on reacquiring the port  don't
	 * try forever */
	for(i=0; i < 6 && !done; i++) {
		struct timeval tv;
#endif
		if(EstablishAnEar(port, port, &myEar, &earPort)) {
			done = 1;
#ifdef EXPERIMENTAL
			break;
#endif
		}
#ifdef EXPERIMENTAL
		/* sleep 5 seconds in a way that Condor will tolerate */
		tv.tv_sec = 5;
		tv.tv_usec = 0;
		select(0,NULL,NULL,NULL,&tv);
	}
#endif

	if(done == 0) {
		FAIL1("EstablishHost: failed to attach to port %d\n", port);
	}


	/* we piss all over the local variable: we need a big lock here.
	 * Thankfully we call this function only once ... */
	GetNWSLock(&lock);
	myAddresses = (IPAddress *)MALLOC(addressesCount * sizeof(IPAddress));
	myRegistration = strdup(registration);
	myPassword = strdup((password == NULL) ? "" : password);
	if((myAddresses == NULL) || (myRegistration == NULL) || (myPassword == NULL)){
		ReleaseNWSLock(&lock);
		FAIL("EstablishHost: out of memory\n");
	}

	memcpy(myAddresses, addresses, addressesCount * sizeof(IPAddress));
	/* there might be doubles: remove them */
	for (i = 0; i < addressesCount - 1; i++) {
		for (j = i+1; j < addressesCount; j++) {
			if (myAddresses[i].addr == myAddresses[j].addr) {
				/* found a double: remove it */
				for (z = j+1; z < addressesCount; z++) {
					myAddresses[z-1] = myAddresses[z];
				}
				addressesCount--;
			}
		}
	}

	myAddressesCount = addressesCount;
	myExitFunction = exitFunction;
	myPort = port;
	myType = hostType;
	/* record default memory server */
	if (memoryServer != NULL) {
		SAFESTRCPY(myMemoryServer.name, memoryServer->name);
		myMemoryServer.port = memoryServer->port;
		myMemoryServer.sd = memoryServer->sd;
		/* we take control of the socket: it's not valid anymore */
		memoryServer->sd = NO_SOCKET;
	}
	/* record default name server */
	if (nameServer != NULL) {
		/* let's set the new nameserver */
		SAFESTRCPY(myNameServer.name, nameServer->name);
		myNameServer.port = nameServer->port;
		myNameServer.sd = nameServer->sd;
		/* we take control of the socket: it's not valid anymore */
		nameServer->sd = NO_SOCKET;

	}
	ReleaseNWSLock(&lock);

	/* trying a nice termination */
	signal(SIGTERM, TermHandler);
	signal(SIGINT, TermHandler);

	/* register all the messages we handle */
	RegisterListener(HOST_DIAGNOSTICS, "HOST_DIAGNOSTICS", &ProcessRequest);
	RegisterListener(HOST_DIE, "HOST_DIE", &ProcessRequest);
	RegisterListener(HOST_REGISTER, "HOST_REGISTER", &ProcessRequest);
	RegisterListener(HOST_CHANGE_MEMORY, "HOST_CHANGE_MEMORY", &ProcessRequest);
	RegisterListener(HOST_TEST, "HOST_TEST", &ProcessRequest);
	RegisterListener(HOST_GET_NS, "HOST_GET_NS", &ProcessRequest);
	RegisterListener(HOST_GET_MEMORY, "HOST_GET_MEMORY", &ProcessRequest);
	RegisterListener(NS_FAILED, "NS_FAILED", &ProcessRequest);
	RegisterListener(NS_REGISTERED, "NS_REGISTERED", &ProcessRequest);
	RegisterListener(NS_UNREGISTERED, "NS_UNREGISTERED", &ProcessRequest);

	/* add my registration and register with the nameserver */
	hostObject = CreateHostObject(registration, hostType, addresses, addressesCount, port);

	/* let's tell which flags have been used for this compilation */
	tmp = MALLOC(strlen("experimental") + strlen("debug") + strlen("thread") + 5);
	if (tmp == NULL) {
		ABORT("EstablishHost: out of memory\n");
	}
	tmp[0] = '\0';

#ifdef EXPERIMENTAL
	strcat(tmp, "experimental");
#endif
#ifdef WITH_DEBUG
	if (tmp[0] != '\0') {
		strcat(tmp, ",");
	}
	strcat(tmp, "debug");
#endif
#ifdef HAVE_PTHREAD_H
	if (tmp[0] != '\0') {
		strcat(tmp, ",");
	}
	strcat(tmp, "thread");
#endif
	if (tmp[0] != '\0') {
		AddNwsAttribute(&hostObject, "flags", tmp);
	}

	/* add extra static info for this host */
	AddNwsAttribute(&hostObject, "systemType", GetSystemName());
	AddNwsAttribute(&hostObject, "releaseName", GetRelease());
	AddNwsAttribute(&hostObject, "machineArch", GetMachine());
	{
		double mem;

		/* how many CPUs */
		sprintf(tmp, "%d", CPUCount());
		AddNwsAttribute(&hostObject, "CPUcount", tmp);

		/* how much memory we have */
		mem = InstalledMemory();
		if (mem > 0) {
			sprintf(tmp, "%.0f", mem);
			AddNwsAttribute(&hostObject, "memory", tmp);
		}
	}
	free(tmp);
	RegisterObject(hostObject);
	FreeObject(&hostObject);

	return 1;
}

IPAddress 
PreferredInterface() {
	/* read-only and not chagend after initialization: thread safe */
	return myAddresses[0];
}

int
EstablishedInterface(IPAddress address,
                     unsigned short port) {
	int i;

	/* read-only and not changed after initialization: thread safe */
	if(port != myPort) {
		return 0;
	}

	for(i = 0; i < myAddressesCount; i++) {
		if (myAddresses[i].addr == address.addr) {
			return 1;
		}
	}

	return 0;
}


const char *
EstablishedRegistration(void) {
	return myRegistration;
}

int
RetrieveFromMyNameserver(	const char *filter,
				ObjectSet *whereTo) {
	struct host_cookie cookie;
	Object obj;
	int ret;

	/* sanity check */
	if (filter == NULL || whereTo == NO_OBJECT_SET) {
		return 0;
	}
	
	/* let's retrieve objects from the first of the known nameservers
	 * which answers to us */
	GetNWSLock(&lock);

	/* let's try our canonical nameserver first */
	SAFESTRCPY(cookie.name, myNameServer.name);
	cookie.port = myNameServer.port;
	cookie.sd = myNameServer.sd;

	ret = 0;
	obj = NULL;
	do {
		if (RetrieveObjects(&cookie, filter, whereTo, -1)) {
			/* we got it */
			ret = 1;
			break;
		}
		DisconnectHost(&cookie);

		/* let's get to the next nameserver */
		do {
			obj = NextObject(nameServers, obj);
			if (obj && ConnectToObject(obj, &cookie)) {
				break;
			}
		} while(obj);
	} while (obj);	

	/* reset the nameserver */
	if (ret == 1) {
		SAFESTRCPY(myNameServer.name, cookie.name);
		myNameServer.port = cookie.port;
		myNameServer.sd = cookie.sd;
	} else {
		/* we couldn't connect */
		myNameServer.sd = NO_SOCKET;
	}
	ReleaseNWSLock(&lock);

	return ret;
}

int
HostHealthy(void) {
	Object hostObject = NO_OBJECT;
	int returnValue;
	char *filter;
#define LOOKUP_FILTER "(&(name=%s)(objectclass=nwsHost))"
#define LOOKUP_FILTER_LEN 33

	/* nameservers are always healthy! */
	if(myType == NAME_SERVER_HOST) {
		return 1;
	}

	returnValue = 0;

	/* let's create the filter */
	GetNWSLock(&lock);
	filter = (char *)MALLOC(LOOKUP_FILTER_LEN + strlen(myRegistration));
	if (filter == NULL) {
		ABORT("HostHealthy: out of memory\n");
	}
	sprintf(filter, LOOKUP_FILTER, myRegistration);
	ReleaseNWSLock(&lock);
	if (RetrieveFromMyNameserver(filter, &hostObject)) {
		if(hostObject != NO_OBJECT) {
			returnValue =  strlen(hostObject) > 1;
			FreeObject(&hostObject);
		}
	}
	FREE(filter);

	return returnValue;
}


/**
 * we don't let the children to call StoreExperiment directly because we
 * want to control which memory will take the data. Instead the chidren
 * should call this function.
 * #experiment# is the exeriment we need to store, and #registration# is the
 * series (as object to be registered) related to this experiment. We
 * take care of registering the series whenever we are succesful on
 * saving the values. We keep some backup of old values we couldn't store
 * into the memory.  Returns 1 if we got a backup of the eperiment, 0
 * otherwise. */
#define KEEP_A_LONG_TIME 315360000.0
#define BACKUPLENGTH 512
int
RegisterExperiment(	char *registration,
			NWSAPI_Measurement *experiment) {
	char *tmp;
	int length;
	struct host_cookie tmpMem;
	static NWSAPI_Measurement *backupExps[BACKUPLENGTH];
	static char *backupSeries[BACKUPLENGTH];
	static int backupExpStart = 0, backupExpEnd = 0;
	static void *expLock = NULL;

	/* sanity check */
	if (experiment == NULL || registration == NULL) {
		WARN("RegisterExperiment: NULL parameters\n");
		return 0;
	}

	/* we want to keep the ordering of the measurements so we put
	 * everything into the backupExps. */
	tmp = strstr(registration, OBJECT_TERMINATOR);
	if (tmp == NULL) {
		WARN("RegisterExperiment: registration without end\n");
		return 0;
	}
	length = tmp - registration + OBJECT_TERMINATOR_LEN;
	tmp = MALLOC(length + 1);
	if (tmp == NULL) {
		ERROR("RegisterExperiment: out of memory\n");
		return 0;
	}
	memcpy(tmp, registration, length);
	tmp[length] = '\0';

	/* memory can change so we take a 'snapshot' of the current
	 * memory */
	GetNWSLock(&lock);
	tmpMem = myMemoryServer;
	ReleaseNWSLock(&lock);

	/* things get ugly: since we are working on a static structure we
	 * need to lock, but since we are going on the net, we don't want
	 * to stuck everything here. Hence we use a lock only for this
	 * function. */
	GetNWSLock(&expLock);
	backupSeries[backupExpEnd] = tmp;

	/* save the experiment in the backup queue */
	backupExps[backupExpEnd] = (NWSAPI_Measurement *)MALLOC(sizeof(NWSAPI_Measurement));
	if (backupExps[backupExpEnd] == NULL) {
		ABORT("RegisterExperiment: out of memory\n");
	}
	backupExps[backupExpEnd]->timeStamp = experiment->timeStamp;
	backupExps[backupExpEnd]->measurement = experiment->measurement;

	/* let's move the indices */
	backupExpEnd = (backupExpEnd + 1) % BACKUPLENGTH;
	if (backupExpEnd == backupExpStart) {
		/* we have to get rid of the first experiment */
		FREE(backupExps[backupExpStart]);
		FREE(backupSeries[backupExpStart]);
		backupExpStart = (backupExpStart+1) % BACKUPLENGTH;
	}
		
	/* now let's try to register the experiments */
	for (;backupExpStart != backupExpEnd; backupExpStart = (backupExpStart+1) % BACKUPLENGTH) {
		/* let's try the new memory style message */
		if (!StoreNewExperiments(&myMemoryServer, backupSeries[backupExpStart],  backupExps[backupExpStart], 1, KEEP_A_LONG_TIME)) {
			int i;

			/* let's see how many slots we have occupied */
			i = backupExpEnd - backupExpStart;
			if (i < 0) {
				i += BACKUPLENGTH;
			}
			WARN1("RegisterExperiment: couldn't contact memory (we have %d experiments in queue)\n", i);
			break;
		}
		/* free the memory */
		FREE(backupExps[backupExpStart]);
		FREE(backupSeries[backupExpStart]);
	}
	ReleaseNWSLock(&expLock);

	return 1;
}

/* get a new nameservers list from an open connection with a nameserver. 
 * The nameserver is assumed in the cookie.
 * Return 1 on success, 0 otherwise
 */
static int 
NewNameServerList(struct host_cookie *cookie) {
	ObjectSet tmpSet;
	int ret = 0;

	/* nameservers don't get nameservers here */
	if (myType == NAME_SERVER_HOST) {
		return 1;
	}

	/* sanitcy check */
	if (cookie == NULL) {
		ERROR("NewNameServerList: NULL parameter\n");
		return 0;
	}

	/* let's try to get the new list */
	if (RetrieveFromMyNameserver("&(objectclass=nwsHost)(hostType=nameserver)", &tmpSet)) {
		/* let's check we got some nameservers */
		GetNWSLock(&lock);
		if (tmpSet[0] != '\0') {
			FREE(nameServers);
			nameServers = tmpSet;
			ret = 1;
		} else {
			/* no good */
			FREE(tmpSet);
		}
		ReleaseNWSLock(&lock);
	}

	/* some info in the log */
	if (ret) {
		INFO1("NewNameServerList: new nameservers list (%s)\n", tmpSet);
	} else {
		WARN("NewNameServerList: failed to get  new nameservers list\n");
	}

	return ret;
}

/* register all the object with the best nameserver we can. 
 * On success returns 1 and set the good nameserver in myNameServer. */
int 
RegisterWithNameServer(	ObjectSet set,
			unsigned long timeOut) {
	Object obj, ns;
	struct host_cookie cookie;
	int done, ret;
	
	/* sanity check */
	if (set == NULL || strlen(set)  < OBJECT_TERMINATOR_LEN) {
		return 1;
	}

	/* we need to lock the whole enciladas becasue of the socket: if
	 * we unlock after we copied the nameserver into tmp we can have
	 * 2 socket around */
	GetNWSLock(&lock);

	/* let's try our canonical nameserver first */
	SAFESTRCPY(cookie.name, myNameServer.name);
	cookie.port = myNameServer.port;
	cookie.sd = myNameServer.sd;
	ns = NULL;

	/* let's try to register all the objects */
	for (obj = NextObject(set, NULL); obj != NULL; ) {
		if (Register(&cookie, obj, timeOut)) {
			/* let's register the next object */
			obj = NextObject(set, obj);
		} else { 
			DisconnectHost(&cookie);
			/* the previous nameserver didn't work */
			done = 0;
			while (!done) {
				ns = NextObject(nameServers, ns);
				if (!ns || ConnectToObject(ns, &cookie)) {
					done = 1;
				}
			}
			if (ns == NULL) {
				/* no more nameservers */
				break;
			}
		}
	}
	if (obj != NULL) {
		myNameServer.sd = NO_SOCKET;
		DDEBUG("RegisterWithNameServer: failed to register\n");
		ret = 0;
	} else {
		/* let's set the new default nameserver */
		SAFESTRCPY(myNameServer.name, cookie.name);
		myNameServer.port = cookie.port;
		myNameServer.sd = cookie.sd;
		ret = 1;
	}
	ReleaseNWSLock(&lock);

	return ret;
}


int
RegisterHost(unsigned long timeOut) {
	int ret = 0;
	char *tmp;

	INFO("*** Beat ***\n");

	/* this is a mess: registrationSet can change along the way
	 * (another thread usign RegisterObject) so the safe way is to
	 * copy the set here, and use the copy ... */
	tmp = RegistrationSet();
	if (tmp == NO_OBJECT_SET) {
		return 1;			/* nothing to do */
	}

	/* let's register the whole enciladas */
	if (RegisterWithNameServer(tmp, timeOut)) {
		/* let's try to get the new nameserver list */
		NewNameServerList(&myNameServer);
		ret = 1;
	}
	DisconnectHost(&myNameServer);	/* we don't cache sockets here */
	FREE(tmp);

  	return ret;
}


void
RegisterObject(const Object object) {
	short tmp;
	char *name;
	Object nextObj;

	/* we need to lock all around registrationSet */
	GetNWSLock(&lock);

	/* add the object to the objectSet */
	if (registrationSet == NO_OBJECT_SET) {
		registrationSet = NewObjectSet();
	}

	tmp = 0;

	/* if we don't have the object, or the one we have is different
	 * (more recent), substitute it */
	nextObj = NextObject(registrationSet, NULL);
	for (; nextObj != NO_OBJECT; nextObj = NextObject(registrationSet, nextObj)) {
		if (AreObjectsEquivalent(nextObj, object)) {
			/* they are the same */
			break;
		}
	}
	if (nextObj == NO_OBJECT) {
		/* substitute if we got a name for the object */
		name = NwsAttributeValue_r(FindNwsAttribute(object, "name"));
		if (name != NULL) {
			/* we don't want to go to the nameserver, since
			 * we'll already override the name */
			DirectUnregisterObject(name, 0);
		}
		FREE(name);
		tmp = AddObject(&registrationSet, object);
	}
	ReleaseNWSLock(&lock);

	if (tmp) {
		RegisterWithNameServer(object, DEFAULT_HOST_BEAT);
	}
	
	/* we don't disconnect here: mainly at the beginning we call a
	 * lot of RegisterObject, so we save some reconnection time. The
	 * first time we call RegisterHost we'll close the socket as
	 * expected */
}


int
SameHost(	const struct host_cookie *left,
		const struct host_cookie *right) {
	IPAddress leftAddr[MAX_ADDRESSES], rightAddr[MAX_ADDRESSES];
	int howManyLeft, howManyRight, i, j;

	if(left->port != right->port) {
		return 0;
	}
	if(strcmp(left->name, right->name) == 0) {
		return 1;
	}

	/* we have to do the hard work ... */
	howManyLeft = IPAddressValues(left->name, leftAddr, MAX_ADDRESSES);
	howManyRight = IPAddressValues(right->name, rightAddr, MAX_ADDRESSES);
	if (!(howManyLeft == 0 && howManyRight == 0)) {
		return 0;  /* No way to tell if they're the same. */
	}

	/* let's look for at least one match */
	for (i = 0; i < howManyLeft; i++) {
		for (j = 0; j < howManyRight; j++) {
			if (leftAddr[i].addr == rightAddr[j].addr) {
				/* match! */
				return 1;
			}
		}
	}

	/* nope they are different */
	return 0;
}


static void
DirectUnregisterObject(	const char *objectName,
			int force) {
	Object current;
	char *filter;
	int len;
	NwsAttribute nameAttr;
	ObjectSet thinnedSet = NO_OBJECT_SET;
	struct host_cookie cookie;

	/* sanity check */
	if (objectName == NULL) {
		ERROR("UnregisterObject: NULL parameter\n");
		return;
	}

	/* we should never get here if we are a nameserver but just in
	 * case ... */
	if (myType == NAME_SERVER_HOST) {
		DDEBUG("UnregisterObject: called from a nameserver\n");
		return;
	}

	/* compute the lenght of this filter */
	len = strlen(objectName) + 10;
	filter = (char *)MALLOC(sizeof(char)*len);
	if (filter == NULL) {
		ABORT("UnregisterObject: out of memory\n");
	}
	vstrncpy(filter, len, 3, "&(name=", objectName, ")");

	/* we don't query the nameserver is force is not enabled */
	if (force) {
		/* we need to lock everything since the nameserver can
		 * change along the way */
		GetNWSLock(&lock);

		/* let's try the canonical nameserver first */
		cookie.sd = NO_SOCKET;
		if (!Unregister(&myNameServer, filter)) {
			/* let's try to send to all the nameservers we are aware of  */
			for (current = NextObject(nameServers, NULL); current != NULL; current = NextObject(nameServers, current)) {
				if (ConnectToObject(current, &cookie)) {
					/* now let's try to register */
					if (Unregister(&cookie, filter)) {
						SAFESTRCPY(myNameServer.name, cookie.name);
						myNameServer.port = cookie.port;
						myNameServer.sd = cookie.sd;
						break;
					}
				}
			}
		}
		DisconnectHost(&myNameServer);
	}
	FREE(filter);

	/* let's look for objects with this name */
	for(current = NextObject(registrationSet, NO_OBJECT); current != NO_OBJECT; current = NextObject(registrationSet, current)) {
		nameAttr = NwsAttributeValue_r(FindNwsAttribute(current, "name"));
		if(nameAttr != NO_ATTRIBUTE && strcmp(nameAttr, objectName)) {
			if (thinnedSet == NO_OBJECT_SET) {
				/* let's create the object */
				thinnedSet = NewObjectSet();
			}

			/* it's not the one we are looking for */
			AddObject(&thinnedSet, current);
		}
		FREE(nameAttr);
	}
 
	/* if we got something in thinnedSet we need to change
	 * registrationSet */
	if (thinnedSet != NO_OBJECT_SET) {
		FreeObjectSet(&registrationSet);
		registrationSet = thinnedSet;
	}

	ReleaseNWSLock(&lock);
}

void
UnregisterObject(const char *name) {
	DirectUnregisterObject(name, 1);
}


ObjectSet
RegistrationSet(void) {
	char *tmp;

	GetNWSLock(&lock);
	if (registrationSet == NULL) {
		tmp = NULL;
	} else {
		tmp = strdup(registrationSet);
		if (tmp == NULL) {
			ERROR("RegistrationSet: out of memory\n");
		}
	}
	ReleaseNWSLock(&lock);

	return tmp;
}


int
GetHostInfo(	struct host_cookie *cookie,
		HostInfo *info,
		double timeout) {
	int done;

	/* sanity check */
	if (cookie == NULL || info == NULL) {
		ERROR("GetHostInfo: NULL parameter(s)\n");
		return 0;
	}

	done = 1;

	/* let's try to talk to the host */
	if (!ConnectToHost(cookie, NULL) || !SendMessage(cookie->sd, HOST_TEST, timeout)) {
		ERROR2("GetHostInfo: couldn't talk to %s:%d\n", cookie->name, cookie->port);
		done = 0;
	} else if (!RecvMessageAndData(cookie->sd, HOST_TEST_RESULT, info, hostInfoDescriptor, hostInfoDescriptorLength, timeout)) {
		ERROR2("GetHostInfo: failed to receive from %s:%d\n", cookie->name, cookie->port);
		done = 0;
	}

	return done;
}

int
HostNameServerInfo(	struct host_cookie *cookie,
			NsInfo *info,
			double timeout) {
	int done;

	/* sanity check */
	if (cookie == NULL || info == NULL) {
		ERROR("HostNameServerInfo: NULL parameter(s)\n");
		return 0;
	}

	done = 1;

	/* let's try to talk to the host */
	if (!ConnectToHost(cookie, NULL) || !SendMessage(cookie->sd, HOST_GET_NS, timeout)) {
		ERROR2("HostNameServerInfo: couldn't talk to %s:%d\n", cookie->name, cookie->port);
		done = 0;
	} else if (!RecvMessageAndData(cookie->sd, HOST_GOT_NS, info, nsInfoDescriptor, nsInfoDescriptorLength, timeout)) {
		ERROR2("HostNameServerInfo: failed to receive from %s:%d\n", cookie->name, cookie->port);
		done = 0;
	}

	return done;
}

int
HostMemoryInfo(	struct host_cookie *cookie,
		NsInfo *info,
		double timeout) {
	int done;

	/* sanity check */
	if (cookie == NULL || info == NULL) {
		ERROR("HostMemoryInfo: NULL parameter(s)\n");
		return 0;
	}

	done = 1;

	/* let's try to talk to the host */
	if (!ConnectToHost(cookie, NULL) || !SendMessage(cookie->sd, HOST_GET_MEMORY, timeout)) {
		ERROR2("HostMemoryInfo: couldn't talk to %s:%d\n", cookie->name, cookie->port);
		done = 0;
	} else if (!RecvMessageAndData(cookie->sd, HOST_GOT_MEMORY, info, nsInfoDescriptor, nsInfoDescriptorLength, timeout)) {
		ERROR2("HostMemoryInfo: failed to receive from %s:%d\n", cookie->name, cookie->port);
		done = 0;
	}

	return done;
}

int
HostHalt (	struct host_cookie *cookie,
		const char *password,
		double timeout) {
	DataDescriptor passwordDescriptor = SIMPLE_DATA(CHAR_TYPE, 0);
	size_t replySize;
	int done;

	/* sanity check */
	if (cookie == NULL) {
		ERROR("HostHalt: NULL parameter\n");
		return 0;
	}
	done = 1;
	passwordDescriptor.repetitions = (password == NULL) ? 0 : strlen(password) + 1;

	/* let' try to talk with the host */
	if (!ConnectToHost(cookie, NULL) || !SendMessageAndData(cookie->sd, HOST_DIE, password, &passwordDescriptor, 1, timeout)) {
		ERROR2("HostHalt: failed to send HOST_DIE to %s:%d\n", cookie->name, cookie->port);
		done = 0;
	} else if(!RecvMessage(cookie->sd, HOST_DYING, &replySize, timeout)) {
		ERROR("HostHalt: failed to receive ack\n");
		done = 0;
	}

	return done;
}

int
HostChangeNameServer(	struct host_cookie *cookie,
			const char *nsName,
			double timeout) {
	size_t ignored;
	int done;
	DataDescriptor nameDescriptor = SIMPLE_DATA(CHAR_TYPE, 0);

	/* sanity check */
	if (cookie == NULL || nsName == NULL) {
		ERROR("HostChangeNameServer: NULL parameter(s)\n");
		return 0;
	}
	done = 1;
	nameDescriptor.repetitions = strlen(nsName) + 1;

	/* let's try to talk with the host */
	if (!ConnectToHost(cookie, NULL) || !SendMessageAndData(cookie->sd, HOST_REGISTER, nsName, &nameDescriptor, 1, timeout)) {
		ERROR2("HostChangeNameServer: failed to send HOST_REGISTER to %s:%d\n", cookie->name, cookie->port);
		done = 0;
	} else if (!RecvMessage(cookie->sd, HOST_REGISTERED, &ignored, timeout)) {
		ERROR("HostChangeNameServer: failed to receive ack\n");
		done = 0;
	}

	return done;
}

int 
HostChangeMemory(	struct host_cookie *cookie,
			const char *memoryName,
			double timeout) {
	size_t ignored;
	int done;
	DataDescriptor nameDescriptor = SIMPLE_DATA(CHAR_TYPE, 0);

	/* sanity check */
	if (cookie == NULL || memoryName == NULL) {
		ERROR("HostChangeMemory: NULL parameter(s)\n");
		return 0;
	}
	done = 1;
	nameDescriptor.repetitions = strlen(memoryName) + 1;

	/* let's try to talk with the host */
	if (!ConnectToHost(cookie, NULL) || !SendMessageAndData(cookie->sd, HOST_CHANGE_MEMORY, memoryName, &nameDescriptor, 1, timeout)) {
		ERROR2("HostChangeMemory: failed to send HOST_CHANGE_MEMORY to %s:%d\n", cookie->name, cookie->port);
		done = 0;
	} else if (!RecvMessage(cookie->sd, HOST_MEMORY_CHANGED, &ignored, timeout)) {
		ERROR("HostChangeMemory: failed to receive ack\n");
		done = 0;
	}

	return done;
}

int
HostGetRegistration(	struct host_cookie *cookie,
			ObjectSet *objs,
			double timeout) {
	size_t dataSize;
	DataDescriptor objDescriptor = SIMPLE_DATA(CHAR_TYPE, 0);

	/* sanity check */
	if (cookie == NULL) {
		ERROR("HostGetRegistration: NULL parameter(s)\n");
		return 0;
	}

	/* let's try to talk with the host */
	if (!ConnectToHost(cookie, NULL) || !SendMessage(cookie->sd, HOST_REGISTRATION, timeout)) {
		ERROR2("HostGetRegistration: failed to send HOST_REGISTRATION to %s:%d\n", cookie->name, cookie->port);
		return 0;
	}

	/* get the header and the data */
	if (!RecvMessage(cookie->sd, HOST_MY_REGISTRATION, &dataSize, timeout)) {
		ERROR("HostChangeMemory: failed to receive ack\n");
		return 0;
	}

	objDescriptor.repetitions = dataSize;
	*objs = (char *)MALLOC(dataSize + 1);
	if (*objs == NULL) {
		ERROR("HostChangeMemory: out of memory\n");
		return 0;
	}
	if (!RecvData(cookie->sd, objs, &objDescriptor, 1, timeout)) {
		ERROR("HostChangeMemory: failed to receive data\n");
		return 0;
	}
	*objs[dataSize] = '\0';

	return 1;
}


int
HostChangeDiagnostic(	struct host_cookie *cookie,
			int diagnostic,
			double timeout) {
	size_t ignored;
	DataDescriptor intDescriptor = SIMPLE_DATA(INT_TYPE, 1);

	/* sanity check */
	if (cookie == NULL) {
		ERROR("HostChangeDiagnostic: NULL parameter\n");
		return 0;
	}

	/* let's try to talk with the host */
	if (!ConnectToHost(cookie, NULL) || !SendMessageAndData(cookie->sd, HOST_DIAGNOSTICS, &diagnostic, &intDescriptor, 1, timeout)) {
		ERROR2("HostGetRegistration: failed to send HOST_DIAGNOSTIC to %s:%d\n", cookie->name, cookie->port);
		return 0;
	}

	if (!RecvMessage(cookie->sd, HOST_DIAGNOSTICS_ACK, &ignored, timeout)) {
		WARN("HostChangeDiagnostic: failed to get ack\n");
	}

	return 1;
}

