/* NVClock 0.8 - Linux overclocker for NVIDIA cards
 *
 * site: http://nvclock.sourceforge.net
 *
 * Copyright(C) 2001-2004 Roderick Colenbrander
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <stdio.h>
#include "nvclock.h"
#include "backend.h"

float nv30_get_fanspeed()
{
    /* Bit 21-16 of register 0x10f0 are used to store the fanspeed in %. 1 bit = 100/64 ~ 1.56% */
    return ((nv_card.PMC[0x10f0/4] >> 16) & 0x3f) * 100.0/64;
}

void nv30_set_fanspeed(float speed)
{
    int value;
    
    /* Don't set the fan lower than 10% for safety reasons */
    if(speed < 10 || speed > 100)
	return;

    value = (nv_card.PMC[0x10f0/4] & 0xffc00000) + (((int)(speed * 63/100) & 0x3f) << 16) + (nv_card.PMC[0x10f0/4] & 0xffff);
    nv_card.PMC[0x10f0/4] = value;
}

float GetClock_nv30(int base_freq, unsigned int pll)
{
    /* All GeforceFX cards except the FX5200/FX5600/FX5700 use this algorithm */
    /* If bit 7 is set use the new algorithm */
    if(pll & 0x80)
    {
        int m1, m2, n1, n2, p;

	m1 = NV30_PLL_M1(pll);
	m2 = NV30_PLL_M2(pll);
    	n1 = NV30_PLL_N1(pll);
	n2 = NV30_PLL_N2(pll);
    	p = NV30_PLL_P(pll);

	/* Perhaps the 0x1f for n2 is wrong .. ? */
	    
	if(nv_card.debug)
	    printf("m1=%d m2=%d n1=%d n2=%d p=%d\n", m1, m2, n1, n2, p);
	    
	return (float)CalcSpeed_nv30(base_freq, m1, m2, n1, n2, p)/1000;
    }
    else
    {
        int m, n, p;

        m = NV30_PLL_M1(pll);
        n = NV30_PLL_N1(pll);
        p = NV30_PLL_P(pll);

        if(nv_card.debug)
	    printf("m=%d n=%d p=%d\n", m, n, p);
	    
	return (float)CalcSpeed(base_freq, m, n, p)/1000;
    }
}

void ClockSelect_nv30(int clockIn, int p_current, unsigned int *pllOut)
{
    unsigned diff, diffOld;
    unsigned VClk, Freq;
    unsigned m, m2, n, n2, p = 0;
    int base_freq = 27000;

    diffOld = 0xFFFFFFFF;

    if(clockIn < 125)
	p = 3;
    else if(clockIn < 250)
	p = 2;
    else if(clockIn < 500)
	p = 1;
    else
	p = 0;
	
    VClk = (unsigned)clockIn;

    Freq = VClk;
    if ((Freq >= 75000) && (Freq <= 1100000))
    {
        for(m = 1; m <= 4; m++)
        {
            for (m2 = 1; m2 <= 4; m2++)
	    {
	        for(n = 1; n <= 31; n++)
	        {
        	    n2 = (int)((float)((VClk << p) * m * m2) / (float)(base_freq * n)+.5);

		    if((n2 < 24) && (n >= n2) && (m >= m2))
		    {
            	        Freq = ((base_freq * n * n2) / (m * m2)) >> p;
            	        if (Freq > VClk)
                	    diff = Freq - VClk;
                	else
		    	    diff = VClk - Freq;

                	/* When the difference is 0 or less than .5% accept the speed */
                	if( (float)diff/(float)clockIn <= 0.005)
                	{
                            *pllOut = m + (m2<<4) + (n<<8) + ((n2 & 0x7) << 19) + ((n2 & 0x18)<<21) + (p<<16) + (1<<7);
                    	    return;
                	}
                	if(diff < diffOld)
			{
                            *pllOut = m + (m2<<4) + (n<<8) + ((n2 & 0x7) << 19) + ((n2 & 0x18)<<21) + (p<<16) + (1<<7);
                    	    diffOld  = diff;
                	}
            	    }
    	        }
    	    }
        }
    }
}


float nv30_get_gpu_speed()
{
    int nvpll = nv_card.PRAMDAC[0x500/4];
    if(nv_card.debug == 1)
    {
	printf("NVPLL_COEFF=%08x\n", nvpll);
    }

    return (float)GetClock_nv30(nv_card.base_freq, nvpll);
}

void nv30_set_gpu_speed(unsigned int nvclk)
{
    unsigned int PLL;
    int p;
    nvclk *= 1000;

    p = NV30_PLL_P(nv_card.PRAMDAC[0x500/4]);
    ClockSelect_nv30(nvclk, p, &PLL);

    if(nv_card.debug)
	printf("NVPLL_COEFF: %08x\n", PLL);

    /* Unlock the programmable NVPLL/MPLL */
    nv_card.PRAMDAC[0x50c/4] |= 0x500;
    nv_card.PRAMDAC[0x500/4] = PLL;
}

float nv30_get_memory_speed()
{
    int mpll = nv_card.PRAMDAC[0x504/4];
    if(nv_card.debug == 1)
    {
	printf("MPLL_COEFF=%08x\n", mpll);
    }
    return (float)GetClock_nv30(nv_card.base_freq, mpll);
}

void nv30_set_memory_speed(unsigned int memclk)
{
    unsigned int PLL;
    int p;
    memclk *= 1000;

    p = NV30_PLL_P(nv_card.PRAMDAC[0x504/4]);
    ClockSelect_nv30(memclk, p, &PLL);

    if(nv_card.debug)
	printf("MPLL_COEFF: %08x\n", PLL);

    /* Unlock the programmable NVPLL/MPLL */
    nv_card.PRAMDAC[0x50c/4] |= 0x500;
    nv_card.PRAMDAC[0x504/4] = PLL;
}

void nv30_reset_gpu_speed()
{
    /* FIXME: we need to use our bios info */
    /* Unlock the programmable NVPLL/MPLL */
    nv_card.PRAMDAC[0x50c/4] |= 0x500;

    /* Set the gpu speed */    
    nv_card.PRAMDAC[0x500/4] = nv_card.nvpll;
}

void nv30_reset_memory_speed()
{
    /* FIXME: we need to use our bios info */
    /* Unlock the programmable NVPLL/MPLL */
    nv_card.PRAMDAC[0x50c/4] |= 0x500;

    /* Set the memory speed */    
    nv_card.PRAMDAC[0x504/4] = nv_card.mpll;
}
