/* access.c - functions for access control

   Copyright (C) 2003  Russell Kroll <rkroll@exploits.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include "common.h"
#include "access.h"
#include "levels.h"

	struct 	acl_t	*acl_head = NULL;
	struct	access_t	*access_head = NULL;

/* see if <addr> matches the acl <aclname> */
int acl_check(const char *aclname, const struct sockaddr_in *addr)
{
	struct	acl_t	*tmp;
	int	aclchk, addrchk;

	tmp = acl_head;
	while (tmp != NULL) {
		if (!strcmp(tmp->name, aclname)) {
			aclchk = tmp->addr & tmp->mask;
			addrchk = ntohl(addr->sin_addr.s_addr) & tmp->mask;

			if (aclchk == addrchk) 
				return 1;	/* match */
		}

		tmp = tmp->next;
	}

	return 0;	/* not found */
}

/* this is only used for simple stuff: MONITOR and BASE */
int access_check(const struct sockaddr_in *addr, int level)
{
	struct	access_t	*tmp;
	int	ret;

	/* check for insanity */
	if (level > LEVEL_MONITOR) {
		upslogx(LOG_ERR, "checkaccess called with level=%d", level);
		return 0;	/* failed */
	}

	tmp = access_head;

	while (tmp != NULL) {
		ret = acl_check(tmp->aclname, addr);

		upsdebugx(3, "acl_check: %s: match %d, level %d (%d)", 
			tmp->aclname, ret, tmp->level, level);

		/* if ip matches and access line provides the right level.. */

		if ((ret == 1) && (tmp->level >= level)) {

			if (tmp->action == ACTION_GRANT)
				return 1;	/* allowed */

			upsdebugx(1, "access denied");
			return 0;
		} 	/* if (ip matches) && (level is provided) */

		/* otherwise ip didn't match or the level was inadequate */

		tmp = tmp->next;
	}

	/* default = deny */
	return 0;
}

/* add to the master list of ACL names */
void acl_add(const char *aclname, char *ipblock)
{
	struct	acl_t	*tmp, *last;
	char	*addr, *mask;

	/* are we sane? */
	if ((!aclname) || (!ipblock))
		return;

	/* ipblock must be in the form <addr>/<mask>	*/

	mask = strchr(ipblock, '/');

	/* 192.168.1.1/32: valid */
	/* 192.168.1.1/255.255.255.255: valid */
	/* 192.168.1.1: invalid */

	if (!mask) 	/* no slash = broken acl declaration */
		return;

	*mask++ = '\0';
	addr = ipblock;

	tmp = last = acl_head;

	while (tmp != NULL) {	/* find end */
		last = tmp;
		tmp = tmp->next;
	}

	tmp = xmalloc(sizeof (struct acl_t));
	tmp->name = xstrdup(aclname);
	tmp->addr = ntohl(inet_addr(addr));
	tmp->next = NULL;

	if (strstr(mask, ".") == NULL) { /* must be a /nn CIDR type block */
		if (atoi(mask) != 32)
			tmp->mask = ((unsigned int) ((1 << atoi(mask)) - 1) << 
			            (32 - atoi(mask)));
		else
			tmp->mask = 0xffffffff;	/* avoid overflow from 2^32 */
	}
	else
		tmp->mask = ntohl(inet_addr (mask));

	if (last == NULL)	/* first */
		acl_head = tmp;
	else
		last->next = tmp;
}

/* add to the access linked list */
void access_add(const char *action, const char *level, const char *aclname)
{
	struct	access_t	*tmp, *last;

	/* more sanity checking (null password is OK) */
	if ((!action) || (!level) || (!aclname))
		return;

	tmp = last = access_head;

	while (tmp != NULL) {	/* find end */
		last = tmp;
		tmp = tmp->next;
	}

	tmp = xmalloc(sizeof(struct access_t));
	tmp->action = 0;
	tmp->level = 0;

	if (!strcasecmp(action, "grant"))
		tmp->action = ACTION_GRANT;
	if (!strcasecmp(action, "deny"))
		tmp->action = ACTION_DENY;

	/* we don't do "drop" any more - convert to deny */
	if (!strcasecmp(action, "drop")) {
		upslogx(LOG_WARNING, "ACCESS action of 'drop' deprecated; change to 'deny'");
		tmp->action = ACTION_DENY;
	}

	if (!strcasecmp(level, "base"))
		tmp->level = LEVEL_BASE;
	if (!strcasecmp(level, "monitor"))
		tmp->level = LEVEL_MONITOR;

	/* deprecated in 1.2, removed in 1.3 */
	if ((!strcasecmp(level, "login")) ||
	   (!strcasecmp(level, "master"))) {

		upslogx(LOG_WARNING, "LOGIN and MASTER no longer supported "
			" in upsd.conf - switch to upsd.users");

		/* give them something somewhat useful */
		tmp->level = LEVEL_MONITOR;
	}

	/* deprecated in 1.0, removed in 1.1 */
	if (!strcasecmp(level, "manager")) {
		upslogx(LOG_WARNING, "ACCESS type manager no longer supported -"
			" switch to upsd.users");

		/* but don't leave them totally out in the cold */
		tmp->level = LEVEL_MONITOR;
	}

	if (!strcasecmp(level, "all"))
		tmp->level = LEVEL_ALL;

	tmp->aclname = xstrdup(aclname);
	tmp->next = NULL;

	if (last == NULL)	/* first */
		access_head = tmp;
	else
		last->next = tmp;	
}

void acl_free(void)
{
	struct	acl_t	*ptr, *next;

	ptr = acl_head;

	while (ptr) {
		next = ptr->next;
		
		if (ptr->name)
			free(ptr->name);
		free(ptr);

		ptr = next;
	}

	acl_head = NULL;
}

void access_free(void)
{
	struct	access_t	*ptr, *next;

	ptr = access_head;

	while (ptr) {
		next = ptr->next;

		if (ptr->aclname)
			free(ptr->aclname);
		free(ptr);

		ptr = next;
	}

	access_head = NULL;
}	
